﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    internal class IfOptimizePrimitiveAlgorithmForsyth : IfOptimizePrimitiveAlgorithm
    {
        //---------------------------------------------------------------------
        // 処理頂点数
        public int ProcessingVertexCount { get; private set; }

        // 結果のインデックス列
        public int[] Result { get; private set; }

        // エミュレーションキャッシュサイズ
        public int LruCacheSize { get; set; }

        // 入力インデックスリスト
        private readonly ReadOnlyCollection<int> IndexList;

        //---------------------------------------------------------------------
        // コンストラクタ
        public IfOptimizePrimitiveAlgorithmForsyth(ReadOnlyCollection<int> source)
        {
            this.IndexList = source;
            s_vertexScoresComputed = ComputeVertexScores();
            LruCacheSize = 32;
        }

        //---------------------------------------------------------------------
        // 最適化
        public void Optimize(int minProcessingVertexCount)
        {
            int indexCount = this.IndexList.Count;
            int vertexCount = this.IndexList.Max() + 1;
            Result = new int[indexCount];

            OptimizeVertexData[] vertexDataList = new OptimizeVertexData[vertexCount];

            for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
            {
                vertexDataList[vertexIndex] = new OptimizeVertexData();
            }

            // compute face count per vertex
            for (int indexListIndex = 0; indexListIndex < indexCount; ++indexListIndex)
            {
                int vertexIndex = IndexList[indexListIndex];
                Nintendo.Foundation.Contracts.Assertion.Operation.True(vertexIndex < vertexCount);
                OptimizeVertexData vertexData = vertexDataList[vertexIndex];
                vertexData.ActiveFaceListSize++;
            }

            const ushort EvictedCacheIndex = ushort.MaxValue;

            int curActiveFaceListPos = 0;
            {
                // allocate face list per vertex
                for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
                {
                    OptimizeVertexData vertexData = vertexDataList[vertexIndex];
                    vertexData.CachePos0 = EvictedCacheIndex;
                    vertexData.CachePos1 = EvictedCacheIndex;
                    vertexData.ActiveFaceListStart = curActiveFaceListPos;
                    curActiveFaceListPos += vertexData.ActiveFaceListSize;
                    vertexData.Score = FindVertexScore(vertexData.ActiveFaceListSize, vertexData.CachePos0, LruCacheSize);
                    vertexData.ActiveFaceListSize = 0;
                }
            }
            int[] activeFaceList = new int[curActiveFaceListPos];

            // fill out face list per vertex
            for (int indexListIndex = 0; indexListIndex < indexCount; indexListIndex += 3)
            {
                for (int subIndex = 0; subIndex < 3; ++subIndex)
                {
                    int vertexIndex = IndexList[indexListIndex + subIndex];
                    OptimizeVertexData vertexData = vertexDataList[vertexIndex];
                    activeFaceList[vertexData.ActiveFaceListStart + vertexData.ActiveFaceListSize] = indexListIndex;
                    vertexData.ActiveFaceListSize++;
                }
            }

            byte[] processedFaceList = new byte[indexCount];

            int[] vertexCacheBuffer = new int[(MaxVertexCacheSize + 3) * 2];
            int cache0 = 0;
            int cache1 = MaxVertexCacheSize + 3;
            int entriesInCache0 = 0;

            int bestFace = 0;
            float bestScore = -1.0f;

            float maxValenceScore = FindVertexScore(1, EvictedCacheIndex, LruCacheSize) * 3.0f;

            for (int indexListIndex = 0; indexListIndex < indexCount; indexListIndex += 3)
            {
                if (bestScore < 0.0f)
                {
                    // no verts in the cache are used by any unprocessed faces so
                    // search all unprocessed faces for a new starting point
                    for (int index = 0; index < indexCount; index += 3)
                    {
                        if (processedFaceList[index] == 0)
                        {
                            int face = index;
                            float faceScore = 0.0f;
                            for (int subIndex = 0; subIndex < 3; ++subIndex)
                            {
                                int vertexIndex = IndexList[face + subIndex];
                                OptimizeVertexData vertexData = vertexDataList[vertexIndex];
                                Nintendo.Foundation.Contracts.Assertion.Operation.True(vertexData.ActiveFaceListSize > 0);
                                Nintendo.Foundation.Contracts.Assertion.Operation.True(vertexData.CachePos0 >= LruCacheSize);
                                faceScore += vertexData.Score;
                            }

                            if (faceScore > bestScore)
                            {
                                bestScore = faceScore;
                                bestFace = face;

                                Nintendo.Foundation.Contracts.Assertion.Operation.True(bestScore <= maxValenceScore);
                                if (bestScore >= maxValenceScore)
                                {
                                    break;
                                }
                            }
                        }
                    }
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(bestScore >= 0.0f);
                }

                processedFaceList[bestFace] = 1;
                int entriesInCache1 = 0;

                // add bestFace to LRU cache and to newIndexList
                for (int subIndex = 0; subIndex < 3; ++subIndex)
                {
                    int vertexIndex = IndexList[bestFace + subIndex];
                    Result[indexListIndex + subIndex] = vertexIndex;

                    OptimizeVertexData vertexData = vertexDataList[vertexIndex];

                    if (vertexData.CachePos1 >= entriesInCache1)
                    {
                        vertexData.CachePos1 = entriesInCache1;
                        vertexCacheBuffer[cache1 + entriesInCache1] = vertexIndex;
                        entriesInCache1++;

                        if (vertexData.ActiveFaceListSize == 1)
                        {
                            --vertexData.ActiveFaceListSize;
                            continue;
                        }
                    }

                    Nintendo.Foundation.Contracts.Assertion.Operation.True(vertexData.ActiveFaceListSize > 0);
                    int begin = vertexData.ActiveFaceListStart;
                    int end = vertexData.ActiveFaceListStart + vertexData.ActiveFaceListSize;
                    int it = Array.FindIndex(activeFaceList, begin, vertexData.ActiveFaceListSize, x => x.Equals(bestFace));
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(it != end);
                    int tmp = activeFaceList[it];
                    activeFaceList[it] = activeFaceList[end - 1];
                    activeFaceList[end - 1] = tmp;
                    --vertexData.ActiveFaceListSize;
                    vertexData.Score = FindVertexScore(vertexData.ActiveFaceListSize, vertexData.CachePos1, LruCacheSize);
                }

                // move the rest of the old verts in the cache down and compute their new scores
                for (uint c0 = 0; c0 < entriesInCache0; ++c0)
                {
                    int index = vertexCacheBuffer[cache0 + c0];
                    OptimizeVertexData vertexData = vertexDataList[index];

                    if (vertexData.CachePos1 >= entriesInCache1)
                    {
                        vertexData.CachePos1 = entriesInCache1;
                        vertexCacheBuffer[cache1 + entriesInCache1] = index;
                        entriesInCache1++;
                        vertexData.Score = FindVertexScore(vertexData.ActiveFaceListSize, vertexData.CachePos1, LruCacheSize);
                    }
                }

                // find the best scoring triangle in the current cache (including up to 3 that were just evicted)
                bestScore = -1.0f;
                for (int c1 = 0; c1 < entriesInCache1; ++c1)
                {
                    int index = vertexCacheBuffer[cache1 + c1];
                    OptimizeVertexData vertexData = vertexDataList[index];
                    vertexData.CachePos0 = vertexData.CachePos1;
                    vertexData.CachePos1 = EvictedCacheIndex;
                    for (int activeFaceListIndex = 0; activeFaceListIndex < vertexData.ActiveFaceListSize; ++activeFaceListIndex)
                    {
                        int face = activeFaceList[vertexData.ActiveFaceListStart + activeFaceListIndex];
                        float faceScore = 0.0f;
                        for (int subIndex = 0; subIndex < 3; subIndex++)
                        {
                            int faceIndex = IndexList[face + subIndex];
                            OptimizeVertexData faceVertexData = vertexDataList[faceIndex];
                            faceScore += faceVertexData.Score;
                        }
                        if (faceScore > bestScore)
                        {
                            bestScore = faceScore;
                            bestFace = face;
                        }
                    }
                }

                int cacheTmp = cache0;
                cache0 = cache1;
                cache1 = cacheTmp;
                entriesInCache0 = System.Math.Min(entriesInCache1, LruCacheSize);
            }

            // 処理頂点数をカウント
            this.ProcessingVertexCount =
                G3dProcessingVertexCounter.CountTrianglesNX(this.Result);

            // 最適化が完了していれば、メモリを解放できるようにしておく
            if (this.ProcessingVertexCount > minProcessingVertexCount)
            {
                // 処理頂点数が最小値以下でなければ、結果を解放する
                this.Result = null;
            }
        }

        private float ComputeVertexCacheScore(int cachePosition, int vertexCacheSize)
        {
            const float FindVertexScore_CacheDecayPower = 1.5f;
            const float FindVertexScore_LastTriScore = 0.75f;

            float score = 0.0f;
            if (cachePosition < 0)
            {
                // Vertex is not in FIFO cache - no score.
            }
            else
            {
                if (cachePosition < 3)
                {
                    // This vertex was used in the last triangle,
                    // so it has a fixed score, whichever of the three
                    // it's in. Otherwise, you can get very different
                    // answers depending on whether you add
                    // the triangle 1,2,3 or 3,1,2 - which is silly.
                    score = FindVertexScore_LastTriScore;
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(cachePosition < vertexCacheSize);
                    // Points for being high in the cache.
                    float scaler = 1.0f / (vertexCacheSize - 3);
                    score = 1.0f - (cachePosition - 3) * scaler;
                    score = (float)Math.Pow(score, FindVertexScore_CacheDecayPower);
                }
            }

            return score;
        }

        private float ComputeVertexValenceScore(int numActiveFaces)
        {
            const float FindVertexScore_ValenceBoostScale = 2.0f;
            const float FindVertexScore_ValenceBoostPower = 0.5f;

            float score = 0.0f;

            // Bonus points for having a low number of tris still to
            // use the vert, so we get rid of lone verts quickly.
            float valenceBoost = (float)Math.Pow(numActiveFaces,
                -FindVertexScore_ValenceBoostPower);
            score += FindVertexScore_ValenceBoostScale * valenceBoost;

            return score;
        }

        private const int MaxVertexCacheSize = 64;
        private const uint MaxPrecomputedVertexValenceScores = 64;
        private float[,] s_vertexCacheScores = new float[MaxVertexCacheSize + 1, MaxVertexCacheSize];
        private float[] s_vertexValenceScores = new float[MaxPrecomputedVertexValenceScores];

        private bool ComputeVertexScores()
        {
            for (int cacheSize = 0; cacheSize <= MaxVertexCacheSize; ++cacheSize)
            {
                for (int cachePos = 0; cachePos < cacheSize; ++cachePos)
                {
                    s_vertexCacheScores[cacheSize, cachePos] = ComputeVertexCacheScore(cachePos, cacheSize);
                }
            }

            for (int valence = 0; valence < MaxPrecomputedVertexValenceScores; ++valence)
            {
                s_vertexValenceScores[valence] = ComputeVertexValenceScore(valence);
            }

            return true;
        }

        private bool s_vertexScoresComputed = false;

        private float FindVertexScore(int numActiveFaces, int cachePosition, int vertexCacheSize)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(s_vertexScoresComputed);

            if (numActiveFaces == 0)
            {
                // No tri needs this vertex!
                return -1.0f;
            }

            float score = 0.0f;
            if (cachePosition < vertexCacheSize)
            {
                score += s_vertexCacheScores[vertexCacheSize, cachePosition];
            }

            if (numActiveFaces < MaxPrecomputedVertexValenceScores)
            {
                score += s_vertexValenceScores[numActiveFaces];
            }
            else
            {
                score += ComputeVertexValenceScore(numActiveFaces);
            }

            return score;
        }

        private class OptimizeVertexData
        {
            public float Score { get; set; }
            public int ActiveFaceListStart { get; set; }
            public int ActiveFaceListSize { get; set; }
            public int CachePos0 { get; set; }
            public int CachePos1 { get; set; }
        }
    }
}
