﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // 組み合わせによるプリミティブ最適化
    internal class IfOptimizePrimitiveAlgorithmCombo : IfOptimizePrimitiveAlgorithm
    {
        #region 最適化パターンの列挙
        // デフォルト最適化の列挙（2 パターン）
        internal static IEnumerable<IfOptimizePrimitiveAlgorithm> EnumerateDefault(
            ReadOnlyCollection<int> source)
        {
            IfOptimizePrimitiveAlgorithmCombo combo0 =
                new IfOptimizePrimitiveAlgorithmCombo(
                    "St:Btm Ch:Old  Cc:Old  Rg:Max  Al:Btm", source);
            combo0.StartVertexIndexSelector = combo0.SelectIndexBottom;
            combo0.VertexIndexSelectors.Add(combo0.SelectIndexVertexCacheHitOld);
            combo0.VertexIndexSelectors.Add(combo0.SelectIndexVertexCacheOld);
            combo0.VertexIndexSelectors.Add(combo0.SelectIndexRegisteredVertexMax);
            combo0.VertexIndexSelectors.Add(combo0.SelectIndexBottom);
            yield return combo0;

            IfOptimizePrimitiveAlgorithmCombo combo1 =
                new IfOptimizePrimitiveAlgorithmCombo(
                    "St:Max Ch:Old Cc:Old Rg:Max Al:Btm", source);
            combo1.StartVertexIndexSelector = combo1.SelectIndexMax;
            combo1.VertexIndexSelectors.Add(combo1.SelectIndexVertexCacheHitOld);
            combo1.VertexIndexSelectors.Add(combo1.SelectIndexVertexCacheOld);
            combo1.VertexIndexSelectors.Add(combo1.SelectIndexRegisteredVertexMax);
            combo1.VertexIndexSelectors.Add(combo1.SelectIndexBottom);
            yield return combo1;
        }

        // 強い最適化の列挙（144 パターン）
        internal static IEnumerable<IfOptimizePrimitiveAlgorithm> EnumerateForce(
            ReadOnlyCollection<int> source)
        {
            string[] startName = new string[] { "Btm", "Max" };
            string[] cacheHitName = new string[] { "Old ", "None" };
            string[] cacheName = new string[] { "New ", "Old ", "None" };
            string[] registeredName = new string[]
            {
                "Top ", "Max ", "Min ", "None"
            };
            string[] allName = new string[] { "Btm", "Min", "Max" };

            // 開始頂点
            for (int start = 0; start < 2; start++)
            {
                // キャッシュヒット頂点
                for (int cacheHit = 0; cacheHit < 2; cacheHit++)
                {
                    // キャッシュ頂点
                    for (int cache = 0; cache < 3; cache++)
                    {
                        // 登録済み頂点
                        for (int registered = 0; registered < 4; registered++)
                        {
                            // 全頂点
                            for (int all = 0; all < 3; all++)
                            {
                                string name = string.Format(
                                    "St:{0} Ch:{1} Cc:{2} Rg:{3} Al:{4}",
                                    startName[start],
                                    cacheHitName[cacheHit],
                                    cacheName[cache],
                                    registeredName[registered],
                                    allName[all]);

                                IfOptimizePrimitiveAlgorithmCombo combo =
                                    new IfOptimizePrimitiveAlgorithmCombo(name, source);

                                // 開始頂点
                                Func<int>[] startSelectors = new Func<int>[]
                                {
                                    combo.SelectIndexBottom,
                                    combo.SelectIndexMax,
                                };
                                combo.StartVertexIndexSelector = startSelectors[start];

                                // キャッシュヒット頂点
                                Func<int>[] cacheHitSelectors = new Func<int>[]
                                {
                                    combo.SelectIndexVertexCacheHitOld
                                };
                                if (cacheHit < cacheHitSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        cacheHitSelectors[cacheHit]);
                                }

                                // キャッシュ頂点
                                Func<int>[] cacheSelectors = new Func<int>[]
                                {
                                    combo.SelectIndexVertexCacheNew,
                                    combo.SelectIndexVertexCacheOld,
                                };
                                if (cache < cacheSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        cacheSelectors[cache]);
                                }

                                // 登録済み頂点
                                Func<int>[] registeredSelectors = new Func<int>[]
                                {
                                    combo.SelectIndexRegisteredVertexTop,
                                    combo.SelectIndexRegisteredVertexMax,
                                    combo.SelectIndexRegisteredVertexMin
                                };
                                if (registered < registeredSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        registeredSelectors[registered]);
                                }

                                // 全頂点
                                Func<int>[] allSelectors = new Func<int>[]
                                {
                                    combo.SelectIndexBottom,
                                    combo.SelectIndexMin,
                                    combo.SelectIndexMax
                                };
                                combo.VertexIndexSelectors.Add(allSelectors[all]);

                                yield return combo;
                            }
                        }
                    }
                }
            }
        }

        // 全ての最適化の列挙（1200 パターン）
        internal static IEnumerable<IfOptimizePrimitiveAlgorithm> EnumerateBruteForce(
            ReadOnlyCollection<int> source)
        {
            string[] startName = new string[] { "Top", "Btm", "Max", "Min" };
            string[] cacheHitName = new string[] { "Max ", "Min ", "New ", "Old ", "None" };
            string[] cacheName = new string[] { "New ", "Old ", "None" };
            string[] registeredName = new string[]
            {
                "Top ", "Btm ", "Max ", "Min ", "None"
            };
            string[] allName = new string[] { "Top", "Btm", "Max", "Min" };

            // 開始頂点
            for (int start = 0; start < 4; start++)
            {
                // キャッシュヒット頂点
                for (int cacheHit = 0; cacheHit < 5; cacheHit++)
                {
                    // キャッシュ頂点
                    for (int cache = 0; cache < 3; cache++)
                    {
                        // 登録済み頂点
                        for (int registered = 0; registered < 5; registered++)
                        {
                            // 全頂点
                            for (int all = 0; all < 4; all++)
                            {
                                string name = string.Format(
                                    "St:{0} Ch:{1} Cc:{2} Rg:{3} Al:{4}",
                                    startName[start],
                                    cacheHitName[cacheHit],
                                    cacheName[cache],
                                    registeredName[registered],
                                    allName[all]);

                                IfOptimizePrimitiveAlgorithmCombo combo =
                                    new IfOptimizePrimitiveAlgorithmCombo(name, source);

                                // 開始頂点
                                Func<int>[] startSelectors = new Func<int>[]
                                {
                                combo.SelectIndexTop,
                                combo.SelectIndexBottom,
                                combo.SelectIndexMax,
                                combo.SelectIndexMin
                                };
                                combo.StartVertexIndexSelector = startSelectors[start];

                                // キャッシュヒット頂点
                                Func<int>[] cacheHitSelectors = new Func<int>[]
                                {
                                combo.SelectIndexVertexCacheHitMax,
                                combo.SelectIndexVertexCacheHitMin,
                                combo.SelectIndexVertexCacheHitNew,
                                combo.SelectIndexVertexCacheHitOld
                                };
                                if (cacheHit < cacheHitSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        cacheHitSelectors[cacheHit]);
                                }

                                // キャッシュ頂点
                                Func<int>[] cacheSelectors = new Func<int>[]
                                {
                                combo.SelectIndexVertexCacheNew,
                                combo.SelectIndexVertexCacheOld,
                                };
                                if (cache < cacheSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        cacheSelectors[cache]);
                                }

                                // 登録済み頂点
                                Func<int>[] registeredSelectors = new Func<int>[]
                                {
                                combo.SelectIndexRegisteredVertexTop,
                                combo.SelectIndexRegisteredVertexBottom,
                                combo.SelectIndexRegisteredVertexMax,
                                combo.SelectIndexRegisteredVertexMin
                                };
                                if (registered < registeredSelectors.Length)
                                {
                                    combo.VertexIndexSelectors.Add(
                                        registeredSelectors[registered]);
                                }

                                // 全頂点
                                Func<int>[] allSelectors = new Func<int>[]
                                {
                                combo.SelectIndexTop,
                                combo.SelectIndexBottom,
                                combo.SelectIndexMax,
                                combo.SelectIndexMin
                                };
                                combo.VertexIndexSelectors.Add(allSelectors[all]);

                                yield return combo;
                            }
                        }
                    }
                }
            }
        }
        #endregion

        //---------------------------------------------------------------------
        // コンストラクタ
        internal IfOptimizePrimitiveAlgorithmCombo(
            string name, ReadOnlyCollection<int> source)
        {
            this.Name = name;
            this.Source = source;
        }

        //---------------------------------------------------------------------
        // 最適化
        public void Optimize(int minProcessingVertexCount)
        {
            this.Result = new int[this.Source.Count];

            // 頂点数の計測
            int maxVertexIndex = -1;
            foreach (int index in this.Source)
            {
                if (index > maxVertexIndex) { maxVertexIndex = index; }
            }
            int vertexCount = maxVertexIndex + 1;

            // 頂点の構築
            for (int i = 0; i < vertexCount; i++)
            {
                Vertex vertex = new Vertex(i);
                this.Vertices.Add(vertex);
                vertex.RegistableVerticesNode = this.RegistableVertices.AddLast(vertex);
            }

            // ポリゴンの構築
            for (int i = 0; i < this.TriangleCount; i++)
            {
                this.Triangles.Add(new Triangle(this.Source, i, this.Vertices));
            }

            // 各 sub_mesh の index が 0 から順番に振られている前提のコードへの対処
            // base_vertex 削除により無駄な処理がある状態
            for (int i = 0; i < vertexCount; i++)
            {
                Vertex vertex = this.Vertices[i];
                if (vertex.UnregisteredTriangleCount == 0)
                {
                    this.RegistableVertices.Remove(vertex.RegistableVerticesNode);
                }
            }

            // 最適化処理
            int targetVertexIndex = this.StartVertexIndexSelector();
            while (targetVertexIndex != -1)
            {
                RegisterTriangles(targetVertexIndex);
                targetVertexIndex = -1;
                if (this.Result.Length == this.ResultIndex) { break; }

                foreach (Func<int> selector in this.VertexIndexSelectors)
                {
                    targetVertexIndex = selector();
                    if (targetVertexIndex != -1) { break; }
                }
            }

            // 最適化完了
            Nintendo.Foundation.Contracts.Assertion.Operation.True(this.Result.Length == this.ResultIndex);
            this.ProcessingVertexCount =
                G3dProcessingVertexCounter.CountTrianglesNX(this.Result);

            // 最適化が完了していれば、メモリを解放できるようにしておく
            if (this.ProcessingVertexCount > minProcessingVertexCount)
            {
                // 処理頂点数が最小値以下でなければ、結果を解放する
                this.Result = null;
            }
            this.Vertices.Clear();
            this.Triangles.Clear();
            this.RegisteredVertexIndices.Clear();
        }

        // 対象の頂点を使用するトライアングルの登録
        private void RegisterTriangles(int targetIndex)
        {
            Vertex target = this.Vertices[targetIndex];
            if (target.UnregisteredTriangleCount <= 0)
            {
                Debug.WriteLine(target.UnregisteredTriangleCount);
            }
            Nintendo.Foundation.Contracts.Assertion.Operation.True(target.UnregisteredTriangleCount > 0);
            int[,] registerTable = { { 0, 1, 2 }, { 1, 2, 0 }, { 2, 0, 1 } };

            foreach (Triangle triangle in target.Triangles)
            {
                // トライアングルが使用済みなら何もしない
                if (triangle.IsRegistered) { continue; }
                triangle.IsRegistered = true;

                // 最初に targetIndex を登録して、順番を変更せずに残りの 2 頂点を登録する
                int tableIndex = Array.IndexOf<int>(triangle.VertexIndices, targetIndex);
                for (int i = 0; i < 3; i++)
                {
                    RegisterVertex(triangle.VertexIndices[registerTable[tableIndex, i]]);
                }

                // target にトライアングルが残っていなければ抜ける
                if (target.UnregisteredTriangleCount == 0) { break; }
            }
        }

        // 頂点の登録
        private void RegisterVertex(int index)
        {
            this.Result[this.ResultIndex] = index;
            this.ResultIndex++;

            // 頂点情報の更新
            Vertex vertex = this.Vertices[index];
            vertex.UnregisteredTriangleCount--;
            if (vertex.UnregisteredTriangleCount == 0)
            {
                this.RegistableVertices.Remove(vertex.RegistableVerticesNode);
            }
            else
            {
                this.RegisteredVertexIndices.Add(index);
            }

            // 頂点キャッシュの更新
            if (!this.VertexCache.Contains(index))
            {
                this.VertexCache.Enqueue(index);
                if (this.VertexCache.Count > G3dConstant.VertexCacheSize)
                {
                    this.VertexCache.Dequeue();
                }
            }
        }

        //---------------------------------------------------------------------
        // トライアングルを全て登録でき、キャッシュアウトが遠い頂点キャッシュの頂点を選択
        internal int SelectIndexVertexCacheHitMax()
        {
            int cacheIndex = 0;
            int maxCacheOut = 0;
            int maxIndex = -1;
            foreach (int vertexIndex in this.VertexCache)
            {
                Vertex vertex = this.Vertices[vertexIndex];
                int triangleCount = vertex.UnregisteredTriangleCount;
                if (triangleCount > 0)
                {
                    int cacheOut = cacheIndex - ((triangleCount - 1) * 2);
                    if (cacheOut >= 0)
                    {
                        if (cacheOut > maxCacheOut)
                        {
                            maxCacheOut = cacheOut;
                            maxIndex = vertexIndex;
                        }
                    }
                }

                cacheIndex++;
            }
            return maxIndex;
        }

        // トライアングルを全て登録でき、キャッシュアウトが近い頂点キャッシュの頂点を選択
        internal int SelectIndexVertexCacheHitMin()
        {
            int cacheIndex = 0;
            int minCacheOut = int.MaxValue;
            int minIndex = -1;
            foreach (int vertexIndex in this.VertexCache)
            {
                Vertex vertex = this.Vertices[vertexIndex];
                int triangleCount = vertex.UnregisteredTriangleCount;
                if (triangleCount > 0)
                {
                    int cacheOut = cacheIndex - ((triangleCount - 1) * 2);
                    if (cacheOut >= 0)
                    {
                        if (cacheOut < minCacheOut)
                        {
                            minCacheOut = cacheOut;
                            minIndex = vertexIndex;
                        }
                    }
                }

                cacheIndex++;
            }
            return minIndex;
        }

        // トライアングルを全て登録でき、新しい頂点キャッシュの頂点を選択
        internal int SelectIndexVertexCacheHitNew()
        {
            int result = -1;
            int cacheIndex = 0;
            foreach (int vertexIndex in this.VertexCache)
            {
                Vertex vertex = this.Vertices[vertexIndex];
                int triangleCount = vertex.UnregisteredTriangleCount;
                if (triangleCount > 0)
                {
                    if (cacheIndex >= ((triangleCount - 1) * 2)) { result = vertexIndex; }
                }
                cacheIndex++;
            }
            return result;
        }

        // トライアングルを全て登録でき、古い頂点キャッシュの頂点を選択
        internal int SelectIndexVertexCacheHitOld()
        {
            int cacheIndex = 0;
            foreach (int vertexIndex in this.VertexCache)
            {
                Vertex vertex = this.Vertices[vertexIndex];
                int triangleCount = vertex.UnregisteredTriangleCount;
                if (triangleCount > 0)
                {
                    if (cacheIndex >= ((triangleCount - 1) * 2)) { return vertexIndex; }
                }
                cacheIndex++;
            }
            return -1;
        }

        //---------------------------------------------------------------------
        // 頂点キャッシュの新しい頂点から選択
        internal int SelectIndexVertexCacheNew()
        {
            int result = -1;
            foreach (int index in this.VertexCache)
            {
                if (this.Vertices[index].UnregisteredTriangleCount > 0) { result = index; }
            }
            return result;
        }

        // 頂点キャッシュの古い頂点から選択
        internal int SelectIndexVertexCacheOld()
        {
            foreach (int index in this.VertexCache)
            {
                if (this.Vertices[index].UnregisteredTriangleCount > 0) { return index; }
            }
            return -1;
        }

        //---------------------------------------------------------------------
        // 登録済み頂点の中で、最初に見つかった頂点
        internal int SelectIndexRegisteredVertexTop()
        {
            CreanupRegisteredVertexIndices();
            if (this.RegisteredVertexIndices.Count == 0) { return -1; }
            return this.RegisteredVertexIndices.First();
        }

        // 登録済み頂点の中で、最後に見つかった頂点
        internal int SelectIndexRegisteredVertexBottom()
        {
            CreanupRegisteredVertexIndices();
            if (this.RegisteredVertexIndices.Count == 0) { return -1; }
            return this.RegisteredVertexIndices.Last();
        }

        // 登録済み頂点の中で、最も大きな登録可能トライアングル数を持つ頂点
        internal int SelectIndexRegisteredVertexMax()
        {
            CreanupRegisteredVertexIndices();
            if (this.RegisteredVertexIndices.Count == 0) { return -1; }

            int maxIndex = -1;
            int maxCount = 0;
            foreach (int vertexIndex in this.RegisteredVertexIndices)
            {
                int count = this.Vertices[vertexIndex].UnregisteredTriangleCount;
                if (count > maxCount)
                {
                    maxCount = count;
                    maxIndex = vertexIndex;
                }
            }
            return maxIndex;
        }

        // 登録済み頂点の中で、最も小さな登録可能トライアングル数を持つ頂点
        internal int SelectIndexRegisteredVertexMin()
        {
            CreanupRegisteredVertexIndices();
            if (this.RegisteredVertexIndices.Count == 0) { return -1; }

            int minIndex = -1;
            int minCount = int.MaxValue;
            foreach (int vertexIndex in this.RegisteredVertexIndices)
            {
                int count = this.Vertices[vertexIndex].UnregisteredTriangleCount;
                if (count < minCount)
                {
                    minCount = count;
                    minIndex = vertexIndex;
                }
            }
            return minIndex;
        }

        // 登録済み頂点のクリーンアップ
        private void CreanupRegisteredVertexIndices()
        {
            // 登録できるトライアングルの無い頂点インデックスの削除
            this.RegisteredVertexIndices.RemoveAll(delegate (int vertexIndex)
            {
                return (this.Vertices[vertexIndex].UnregisteredTriangleCount == 0);
            });
        }

        //---------------------------------------------------------------------
        // 最初に見つかったトライアングルを登録可能な頂点
        internal int SelectIndexTop()
        {
            return this.RegistableVertices.First.Value.Index;
        }

        // 最後に見つかったトライアングルを登録可能な頂点
        internal int SelectIndexBottom()
        {
            return this.RegistableVertices.Last.Value.Index;
        }

        // 最もトライアングル登録数の多い頂点を選択
        internal int SelectIndexMax()
        {
            // パフォーマンス低下の回避
            if ((this.RegistableVertices.Count != this.Vertices.Count) &&
                (this.RegistableVertices.Count > 65535))
            {
                return SelectIndexTop();
            }
            int maxIndex = -1;
            int maxCount = 0;
            foreach (Vertex vertex in this.RegistableVertices)
            {
                int count = vertex.UnregisteredTriangleCount;
                if (count > maxCount)
                {
                    maxCount = count;
                    maxIndex = vertex.Index;
                }
            }
            return maxIndex;
        }

        // 最もトライアングル登録数の少ない頂点を選択
        internal int SelectIndexMin()
        {
            // パフォーマンス低下の回避
            if ((this.RegistableVertices.Count != this.Vertices.Count) &&
                (this.RegistableVertices.Count > 65535))
            {
                return SelectIndexBottom();
            }
            int minIndex = -1;
            int minCount = int.MaxValue;
            foreach (Vertex vertex in this.RegistableVertices)
            {
                int count = vertex.UnregisteredTriangleCount;
                if (count < minCount)
                {
                    minCount = count;
                    minIndex = vertex.Index;
                }
            }
            return minIndex;
        }

        //---------------------------------------------------------------------
        // 文字列変換
        public override string ToString()
        {
            return string.Format("{0} {1,6} {2}",
                this.Name, this.ProcessingVertexCount,
                (float)this.ProcessingVertexCount / this.TriangleCount);
        }

        //---------------------------------------------------------------------
        // 開始頂点の選択
        internal Func<int> StartVertexIndexSelector { get; set; }

        // インデックスの選択
        internal List<Func<int>> VertexIndexSelectors { get; set; } = new List<Func<int>>();

        //---------------------------------------------------------------------
        // 処理頂点数
        public int ProcessingVertexCount { get; private set; }

        // 結果のインデックス列
        public int[] Result { get; private set; }

        // トライアングル数
        private int TriangleCount { get { return this.Source.Count / 3; } }

        private readonly string Name;
        private readonly ReadOnlyCollection<int> Source;
        private readonly List<Vertex> Vertices = new List<Vertex>();
        private readonly LinkedList<Vertex> RegistableVertices = new LinkedList<Vertex>();
        private readonly List<Triangle> Triangles = new List<Triangle>();
        private readonly Queue<int> VertexCache = new Queue<int>();
        private readonly List<int> RegisteredVertexIndices = new List<int>();
        private int ResultIndex;

        //---------------------------------------------------------------------
        // トライアングル
        private class Triangle
        {
            // コンストラクタ
            internal Triangle(ReadOnlyCollection<int> source,
                int index, List<Vertex> vertices)
            {
                this.Index = index;
                int baseIndex = this.Index * 3;
                for (int i = 0; i < 3; i++)
                {
                    this.VertexIndices[i] = source[baseIndex + i];
                    this.Vertices[i] = vertices[this.VertexIndices[i]];
                    this.Vertices[i].LinkTriangle(this);
                }
            }

            internal readonly int[] VertexIndices = new int[3];
            internal readonly Vertex[] Vertices = new Vertex[3];
            internal readonly int Index;
            internal bool IsRegistered { get; set; }
        }

        //---------------------------------------------------------------------
        // 頂点
        private class Vertex
        {
            // コンストラクタ
            internal Vertex(int index)
            {
                this.Index = index;
            }

            // トライアングルにリンクする
            internal void LinkTriangle(Triangle triangle)
            {
                this.Triangles.Add(triangle);
                UnregisteredTriangleCount++;
            }

            internal readonly int Index;
            internal readonly List<Triangle> Triangles = new List<Triangle>();
            internal int UnregisteredTriangleCount { get; set; }
            internal LinkedListNode<Vertex> RegistableVerticesNode { get; set; }
        }

        public int minProcessingVertexCount { get; set; }
    }
}
