﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルのプリミティブ最適化
    public class IfModelPrimitiveOptimizer : IfModelOptimizer
    {
        // コンストラクタ
        public IfModelPrimitiveOptimizer(string argument, bool useSortSubmeshIndex)
            : base("IfModelPrimitiveOptimizer_Log", argument)
        {
            this.useSortSubmeshIndex = useSortSubmeshIndex;
        }

        public override string ToString()
        {
            int triangle = this.GetTriangleCount();
            int source = GetSourceProcessingVertexCount();
            int result = GetProcessingVertexCount();
            return string.Format(
                "{0} processing_vertex({1}<{2}) per_triangle({3:f3}<{4:f3})", base.ToString(),
                result, source,
                (triangle != 0) ? ((float)result / triangle) : 0.0f,
                (triangle != 0) ? ((float)source / triangle) : 0.0f);
        }

        // プロセス
        public override string Process
        {
            get { return "optimize_primitive"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            int processingVertexCount = GetProcessingVertexCount();
            int sourceProcessingVertexCount = GetSourceProcessingVertexCount();
            int triangleCount = GetTriangleCount();
            int totalVertexCount = triangleCount * 3;
            StringBuilder builder = new StringBuilder();
            builder.AppendFormat(
                "processing_vertex[{0}({1:f0}%)/{2}({3:f0}%)/{4}]",
                processingVertexCount, 100f * processingVertexCount / totalVertexCount,
                sourceProcessingVertexCount,
                100f * sourceProcessingVertexCount / totalVertexCount, totalVertexCount);
            builder.AppendFormat(
                " processing_vertex_per_triangle[{0:f3}/{1:f3}]",
                (float)processingVertexCount / triangleCount,
                (float)sourceProcessingVertexCount / triangleCount);
            return builder.ToString();
        }

        //---------------------------------------------------------------------
        // 最適化
        protected override void Optimize()
        {
            if (this.Target.shape_array == null ||
                this.Target.shape_array.shape == null) { return; }

            /// TODO: Argument を中間ファイルの定義にする
            foreach (shapeType shape in this.Target.shape_array.Enumerate())
            {
                this.Optimizers.Add(new IfShapePrimitiveOptimizer(
                    this.Target, this.Streams, shape, this.Argument, this.useSortSubmeshIndex));
            }

            G3dParallel.ForEach(this.Optimizers, delegate(IfShapePrimitiveOptimizer optimizer)
            {
                optimizer.Optimize();
            });

            if (GetSourceProcessingVertexCount() != GetProcessingVertexCount())
            {
                this.EnableProcessLog = true;
            }
        }

        //---------------------------------------------------------------------
        // トライアングル数の取得
        internal int GetTriangleCount()
        {
            return this.Optimizers.Sum(
                delegate(IfShapePrimitiveOptimizer optimizer)
                { return optimizer.GetTriangleCount(); });
        }

        // 元の処理頂点数
        internal int GetSourceProcessingVertexCount()
        {
            return this.Optimizers.Sum(
                delegate(IfShapePrimitiveOptimizer optimizer)
                { return optimizer.GetSourceProcessingVertexCount(); });
        }

        // 処理頂点数
        internal int GetProcessingVertexCount()
        {
            return this.Optimizers.Sum(
                delegate(IfShapePrimitiveOptimizer optimizer)
                { return optimizer.GetProcessingVertexCount(); });
        }

        //---------------------------------------------------------------------
        private readonly List<IfShapePrimitiveOptimizer> Optimizers =
            new List<IfShapePrimitiveOptimizer>();

        private bool useSortSubmeshIndex = false;
    }
}
