﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Collections.ObjectModel;

namespace nw.g3d.iflib
{
    internal class IfMeshPrimitiveOptimizer
    {
        internal IfMeshPrimitiveOptimizer(
            modelType model, List<G3dStream> streams, shapeType shape,
            meshType mesh, string mode, bool useSortSubmeshIndex)
        {
            this.model = model;
            this.Streams = streams;
            this.shape = shape;
            this.Mode = mode;
            this.mesh = mesh;

            // トライアングルズのみ対応
            if (this.mesh.mode != mesh_modeType.triangles)
            {
                IfStrings.Throw("IfShapePrimitiveOptimizer_Error_NotTriangles", this.shape.name);
            }

            G3dStream stream = this.Streams[this.mesh.stream_index];
            // 整数ストリームでない
            if (stream.type != stream_typeType.@int)
            {
                IfStrings.Throw("IfShapePrimitiveOptimizer_Error_InvalidStreamType", this.shape.name);
            }
            this.IndexStream = stream.IntData;

            if (useSortSubmeshIndex)
            {
                // サブメッシュの範囲は重複しない必要がある
                foreach (submeshType submesh in this.mesh.submesh_array.Enumerate())
                {
                    Optimizers.Add(new IfSubmeshPrimitiveIndexSortOptimizer(
                        this.model, this.Streams, this.shape,
                        this.IndexStream, submesh, this.Mode));
                }
            }
            else
            {
                // サブメッシュの範囲は重複しない必要がある
                foreach (submeshType submesh in this.mesh.submesh_array.Enumerate())
                {
                    Optimizers.Add(new IfSubmeshPrimitiveOptimizer(
                        this.model, this.Streams, this.shape,
                        this.IndexStream, submesh, this.Mode));
                }
            }
        }

        //---------------------------------------------------------------------
        // 初期化
        public void Initialize()
        {
            foreach (var optimizer in this.Optimizers)
            {
                optimizer.Initialize();
            }
        }

        //---------------------------------------------------------------------
        // 最適化
        internal void Optimize()
        {
            // サブメッシュ単位では並列処理をしない
            foreach (var optimizer in this.Optimizers)
            {
                optimizer.Optimize();
            }
        }

        //---------------------------------------------------------------------
        // メッシュの整列
        internal void AlignMesh(vertexType vertex, int vertexCount)
        {
            this.AlignedVertexIndices = Enumerable.Repeat<int>(-1, vertexCount).ToArray();
            this.AlignedIndexIndices = Enumerable.Repeat<int>(-1, this.mesh.count).ToArray();

            AlignIndices(this.Streams[mesh.stream_index].IntData.ToArray());
        }

        //---------------------------------------------------------------------
        // 頂点を共有したメッシュの整列
        internal void AlignSharedMesh(vertexType vertex, Dictionary<int, int> sharedVertexIndices)
        {
            this.AlignedIndexIndices = Enumerable.Repeat<int>(-1, this.mesh.count).ToArray();
            var sourceIndices = this.Streams[mesh.stream_index].IntData;

            foreach (submeshType submesh in this.mesh.submesh_array.submesh)
            {
                for (int idxIdx = 0; idxIdx < submesh.count; ++idxIdx)
                {
                    int index = idxIdx + submesh.offset;
                    this.AlignedIndexIndices[index] = sharedVertexIndices[sourceIndices[index]];
                }
            }
#if DEBUG
            foreach (int alignedIndex in AlignedIndexIndices)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(alignedIndex != -1);
            }
#endif
        }

        // インデックスの整列
        private void AlignIndices(int[] sourceIndices)
        {
            int alignedVertexCount = 0;
            int alignedIndexCount = 0;
            this.UsedVertices = new Dictionary<int, int>(AlignedVertexIndices.Length);
            foreach (var submesh in this.mesh.submesh_array.submesh)
            {
                for (int idxIdx = 0; idxIdx < submesh.count; ++idxIdx)
                {
                    int index = sourceIndices[idxIdx + submesh.offset];

                    if (this.UsedVertices.ContainsKey(index))
                    {
                        // 使用済み頂点
                        this.AlignedIndexIndices[alignedIndexCount] = this.UsedVertices[index];
                        ++alignedIndexCount;
                    }
                    else
                    {
                        // 未使用頂点
                        this.AlignedVertexIndices[alignedVertexCount] = index;
                        this.AlignedIndexIndices[alignedIndexCount] = alignedVertexCount;
                        this.UsedVertices[index] = alignedVertexCount;
                        ++alignedVertexCount;
                        ++alignedIndexCount;
                    }
                }
            }

#if DEBUG
            // 全ての頂点とインデックスが利用されているか
            foreach (int alignedVertex in this.AlignedVertexIndices)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(alignedVertex != -1);
            }
            foreach (int alignedIndex in this.AlignedIndexIndices)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(alignedIndex != -1);
            }
#endif
        }

        //---------------------------------------------------------------------
        // トライアングル数の取得
        internal int GetTriangleCount()
        {
            return Optimizers.Sum(
                delegate(ISubmeshPrimitiveOptimizer submeshOptimizer)
                { return submeshOptimizer.TriangleCount; });
        }

        // 元の処理頂点数の取得
        internal int GetSourceProcessingVertexCount()
        {
            return Optimizers.Sum(
                delegate(ISubmeshPrimitiveOptimizer submeshOptimizer)
                { return submeshOptimizer.SourceProcessingVertexCount; });
        }

        // 処理頂点数の取得
        internal int GetProcessingVertexCount()
        {
            return Optimizers.Sum(
                delegate(ISubmeshPrimitiveOptimizer submeshOptimizer)
                { return submeshOptimizer.ProcessingVertexCount; });
        }

        // 成功したか
        internal bool IsSuccess()
        {
            foreach (ISubmeshPrimitiveOptimizer submeshOptimizer in this.Optimizers)
            {
                if (submeshOptimizer.IsSuccess()) { return true; }
            }
            return false;
        }

        //---------------------------------------------------------------------
        internal int[] AlignedVertexIndices { get; set; }
        internal int[] AlignedIndexIndices { get; set; }
        internal Dictionary<int, int> UsedVertices { get; set; }

        private readonly modelType model;
        private readonly List<G3dStream> Streams;
        private readonly shapeType shape;
        private readonly meshType mesh;
        private readonly List<int> IndexStream;
        private readonly string Mode;

        private readonly List<ISubmeshPrimitiveOptimizer> Optimizers =
            new List<ISubmeshPrimitiveOptimizer>();
    }
}
