﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using nw.g3d.nw4f_3dif;
using System.Linq;

namespace nw.g3d.iflib
{
    // モデルの量子化分析
    public class IfModelQuantizationAnalysisOptimizer : IfModelOptimizer
    {
        private const float PosDiffMax = 1.0f / 4096.0f;
        private const float UvDiffMax = 1.0f / 256.0f;

        private const float ColorDiffMax = 1.0f / 255.0f;
        private const string ShapeMatDelim = "__";

        // コンストラクタ
        public IfModelQuantizationAnalysisOptimizer() :
            base("IfModelQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        public struct StreamSet
        {
            // LOD 含む
            public List<float> floatList { get; set; }
            public List<int> intList { get; set; }

            // ベースモデルのみ
            public List<float> floatBaseList { get; set; }
            // intBaseList に相当するものは使用されていないので追加しない。
        }

        // 最適化
        protected override void Optimize()
        {
            // Unite された LOD モデルにも対応済み。

            if (this.Target.shape_array != null &&
                this.Target.shape_array.shape != null)
            {
                G3dParallel.ForEach(this.Target.shape_array.shape, delegate(shapeType shape)
                {
                    // quantize_type がすべて none でなく、同じであれば何もしない。
                    bool toOptimize = false;
                    foreach (var mesh in shape.mesh_array.mesh)
                    {
                        if (mesh.quantize_type == mesh_quantize_typeType.none ||
                            shape.mesh_array.mesh[0].quantize_type != mesh.quantize_type)
                        {
                            toOptimize = true;
                        }
                    }
                    if (!toOptimize)
                    {
                        return;
                    }

                    this.EnableProcessLog = true;

                    uint data_max = 0;

                    foreach (var mesh in shape.mesh_array.mesh)
                    {
                        int stream_index = mesh.stream_index;
                        G3dStream stream = this.Streams[stream_index];

                        foreach (var data in stream.IntData)
                        {
                            data_max = (data_max < data) ? (uint)data : data_max;
                        }
                    }

                    if (data_max > ushort.MaxValue)
                    {
                        foreach (var mesh in shape.mesh_array.mesh)
                        {
                            mesh.quantize_type = mesh_quantize_typeType.uint_32;
                        }
                    }
                    else
                    {
                        foreach (var mesh in shape.mesh_array.mesh)
                        {
                            mesh.quantize_type = mesh_quantize_typeType.uint_16;
                        }
                    }
                });
            }

            if (this.Target.vertex_array != null &&
                this.Target.vertex_array.vertex != null)
            {
                G3dParallel.ForEach(this.Target.vertex_array.vertex, delegate(vertexType vertex)
                {
                    vtx_attribType[] attribs = vertex.vtx_attrib_array.vtx_attrib;
                    vtx_attrib_quantize_typeType weightType = vtx_attrib_quantize_typeType.none;
                    vtx_attrib_quantize_typeType indexType = vtx_attrib_quantize_typeType.none;

                    if (vertex.vtx_buffer_array != null &&
                        vertex.vtx_buffer_array.vtx_buffer != null)
                    {
                        int baseVertexCount = -1;
                        if (!object.ReferenceEquals(vertex.lod_offset, null))
                        {
                            var parse = G3dDataParser.ParseIntArray(vertex.lod_offset.Value).Where(p => p > 0);
                            if (parse.Count() > 0)
                            {
                                baseVertexCount = parse.Min();
                            }
                        }

                        foreach (var buffer in vertex.vtx_buffer_array.vtx_buffer)
                        {
                            foreach (var input in buffer.input_array.input)
                            {
                                vtx_attribType currentAttrib = attribs[input.attrib_index];
                                G3dStream currentStream = Streams[currentAttrib.stream_index];

                                StreamSet streamSet;
                                int baseStreamCount = baseVertexCount * currentStream.column;
                                MakeVectorList(currentStream, out streamSet, baseStreamCount);
                                if (streamSet.floatList.Count != 0)
                                {
                                    int count = streamSet.floatList.Count;
                                    vtx_attrib_quantize_typeType quantizeType = vtx_attrib_quantize_typeType.none;

                                    if (currentAttrib.hint.StartsWith("position"))
                                    {
                                        quantizeType = QuantizePos(streamSet);
                                    }
                                    else if (currentAttrib.hint.StartsWith("normal"))
                                    {
                                        Vector3 errorVec;
                                        if (!IsNormalized(currentStream, out errorVec))
                                        {
                                            IfStrings.Throw("IfModelQuantizationAnalysisOptimizer_Error_NotNormalized",
                                                currentAttrib.name, currentAttrib.stream_index, errorVec);
                                        }
                                        quantizeType = QuantizeNorm(streamSet);
                                    }
                                    else if (currentAttrib.hint.StartsWith("tangent") ||
                                        currentAttrib.hint.StartsWith("binormal"))
                                    {
                                        Vector3 errorVec;
                                        if (!IsNormalized(currentStream, out errorVec))
                                        {
                                            IfStrings.Throw("IfModelQuantizationAnalysisOptimizer_Error_NotNormalized",
                                                currentAttrib.name, currentAttrib.stream_index, errorVec);
                                        }
                                        quantizeType = QuantizeTangBinorm(streamSet);
                                    }
                                    else if (currentAttrib.hint.StartsWith("color"))
                                    {
                                        quantizeType = QuantizeColor(streamSet, currentAttrib.type);
                                    }
                                    else if (currentAttrib.hint.CompareTo("uv") >= 0)
                                    {
                                        quantizeType = QuantizeUv(streamSet);
                                    }
                                    else if (currentAttrib.hint.StartsWith("blendweight"))
                                    {
                                        if (weightType == vtx_attrib_quantize_typeType.none)
                                        {
                                            Regex regex = new Regex("^blendweight[0-9]+$");

                                            // blendweight は常に 0 から始まる連続したストリームとして扱い量子化型を一致させます。
                                            // current 以外のストリームを追加します。
                                            foreach (var attrib in attribs)
                                            {
                                                if (regex.IsMatch(attrib.hint) && (attrib != currentAttrib))
                                                {
                                                    G3dStream stream = Streams[attrib.stream_index];
                                                    AddVectorList(stream, streamSet, baseVertexCount);
                                                }
                                            }

                                            quantizeType = QuantizeWeight(streamSet, currentAttrib.type, vtx_attrib_quantize_typeType.none);
                                            weightType = quantizeType;
                                        }
                                        else
                                        {
                                            quantizeType = QuantizeWeight(streamSet, currentAttrib.type, weightType);
                                        }
                                    }

                                    currentAttrib.quantize_type = quantizeType;
                                }
                                else if (streamSet.intList.Count != 0)
                                {
                                    int count = streamSet.intList.Count;
                                    vtx_attrib_quantize_typeType quantizeType = vtx_attrib_quantize_typeType.none;

                                    if (currentAttrib.hint.StartsWith("blendindex"))
                                    {
                                        if (indexType == vtx_attrib_quantize_typeType.none)
                                        {
                                            Regex regex = new Regex("^blendindex[0-9]+$");

                                            // blendindex は常に 0 から始まる連続したストリームとして扱います。
                                            // current 以外のストリームを追加します。
                                            foreach (var attrib in attribs)
                                            {
                                                if (regex.IsMatch(attrib.hint) && (attrib != currentAttrib))
                                                {
                                                    G3dStream stream = Streams[attrib.stream_index];
                                                    AddVectorList(stream, streamSet, baseVertexCount);
                                                }
                                            }

                                            quantizeType = QuantizeIndex(streamSet, currentAttrib.type, vtx_attrib_quantize_typeType.none);
                                            indexType = quantizeType;
                                        }
                                        else
                                        {
                                            quantizeType = QuantizeIndex(streamSet, currentAttrib.type, indexType);
                                        }
                                    }
                                    currentAttrib.quantize_type = quantizeType;
                                }
                            }
                        }
                    }
                });
            }

            AlignPositionQuantizationWithDccToolShapeUnit(this.Target);
        }

        /// <summary>
        /// DCC ツール上でのシェイプ単位で量子化設定を制度の高いものに揃えます。
        /// </summary>
        /// <param name="model">対象モデルです。</param>
        internal void AlignPositionQuantizationWithDccToolShapeUnit(modelType model)
        {
            if (model.shape_array == null || model.shape_array.shape == null)
            {
                return;
            }

            if (model.vertex_array == null || model.vertex_array.vertex == null)
            {
                return;
            }

            List<DccToolShapeUnit> dccToolShapes = CreateDccToolShapes(model);
            foreach (DccToolShapeUnit dccToolShape in dccToolShapes)
            {
                foreach (shapeType shape in dccToolShape.ChildShapes)
                {
                    vertexType vertexData = model.vertex_array.vertex[shape.shape_info.vertex_index];
                    var posAttr = vertexData.vtx_attrib_array.vtx_attrib.FirstOrDefault(x => IsPositionAttr(x));
                    if (posAttr == null)
                    {
                        continue;
                    }

                    // 複数の DccToolShapeUnit に登録されている場合があるので、その中で最大の量子化型を採用する
                    if (IsGraterThan(dccToolShape.PositionQuantizeType, posAttr.quantize_type))
                    {
                        posAttr.quantize_type = dccToolShape.PositionQuantizeType;
                    }
                }
            }
        }

        private bool IsGraterThan(vtx_attrib_quantize_typeType rhs, vtx_attrib_quantize_typeType lhs)
        {
            if (rhs == lhs)
            {
                return false;
            }
            else if (lhs == vtx_attrib_quantize_typeType.none)
            {
                return false;
            }
            else if (rhs == vtx_attrib_quantize_typeType.none)
            {
                return true;
            }
            else
            {
                return rhs > lhs;
            }
        }

        private class DccToolShapeUnit
        {
            public string Name { get; set; } = string.Empty;
            public List<shapeType> ChildShapes { get; } = new List<shapeType>();
            public vtx_attrib_quantize_typeType PositionQuantizeType { get; private set; } = vtx_attrib_quantize_typeType.float_16_16_16_16;

            public void UpdatePositionQuantizeType(vtx_attrib_quantize_typeType quantizeType)
            {
                if (!this.ExaminesPostionQuantizable())
                {
                    this.PositionQuantizeType = vtx_attrib_quantize_typeType.float_32_32_32;
                    return;
                }

                switch (quantizeType)
                {
                    case vtx_attrib_quantize_typeType.float_16_16_16_16:
                        break;
                    case vtx_attrib_quantize_typeType.float_32_32_32:
                        {
                            if (this.PositionQuantizeType == vtx_attrib_quantize_typeType.float_16_16_16_16)
                            {
                                this.PositionQuantizeType = vtx_attrib_quantize_typeType.float_32_32_32;
                            }
                        }
                        break;
                    case vtx_attrib_quantize_typeType.none:
                        this.PositionQuantizeType = vtx_attrib_quantize_typeType.none;
                        break;
                    default:
                        throw new Exception($"Unexpected position quantization type {quantizeType}");
                }
            }

            /// <summary>
            /// _p0 頂点属性が量子化可能であるかどうかを返します。
            /// </summary>
            /// <returns>量子化可能であれば true、そうでなければ false を返します。</returns>
            private bool ExaminesPostionQuantizable()
            {
                // 座標の原点が異なるシェイプが混在している場合は穴が空く可能性があるので量子化不可
                int skinningCount = -1;
                foreach (shapeType shape in this.ChildShapes)
                {
                    switch (skinningCount)
                    {
                    case -1:
                        skinningCount = shape.shape_info.vertex_skinning_count;
                        break;
                    case 0:
                        if (shape.shape_info.vertex_skinning_count != 0)
                        {
                            // リジッドボディとスキニングが混在
                            return false;
                        }
                        break;
                    case 1:
                        if (shape.shape_info.vertex_skinning_count != 1)
                        {
                            // リジッドスキニングとリジッドボディ、もしくはスムーススキニングが混在
                            return false;
                        }
                        break;
                    default:
                        if (shape.shape_info.vertex_skinning_count < 2)
                        {
                            // スムーススキニングとリジッドスキニング、もしくはリジッドボディが混在
                            return false;
                        }
                        break;
                    }
                }

                return true;
            }
        }

        private string EstimateOriginalBoneName(shapeType targetShape, shapeType[] allShapes)
        {
            // 中間ファイルの original_bone_name は途中から入ったので、既存の中間ファイルでは空になっているものが
            // 存在するのでシェイプ名からボーン名を抽出する
            int matNameIndex = targetShape.name.IndexOf(targetShape.shape_info.mat_name);
            if (matNameIndex > 0)
            {
                // マテリアル名がシェイプ名中に存在する
                if (!targetShape.name.Substring(targetShape.name.Length - targetShape.shape_info.mat_name.Length, targetShape.shape_info.mat_name.Length).Equals(targetShape.shape_info.mat_name))
                {
                    // シェイプ名が末尾でない
                    return null;
                }

                // シェイプ名の末尾がマテリアル名の場合
                string matNameRemoved = targetShape.name.Remove(matNameIndex, targetShape.shape_info.mat_name.Length);
                if (matNameRemoved.Length <= ShapeMatDelim.Length
                 || !matNameRemoved.Substring(matNameRemoved.Length - ShapeMatDelim.Length, ShapeMatDelim.Length).Equals(ShapeMatDelim))
                {
                    // DCC ツールから出力されたデータではない
                    return null;
                }

                return matNameRemoved.Substring(0, matNameRemoved.Length - ShapeMatDelim.Length);
            }
            else
            {
                // マテリアル名がシェイプ名中に存在しない
                // マテリアル圧縮されている可能性があるので他のシェイプとマテリアルが共有されているか調べる
                bool isMaterialCompressed = allShapes.Any(x => x != targetShape && x.shape_info.mat_name == targetShape.shape_info.mat_name);
                if (!isMaterialCompressed)
                {
                    return null;
                }

                // マテリアル圧縮されている場合は元のマテリアル名は不明なので、"__" 区切りで元のシェイプ名を推測する
                string[] splitedStrings = targetShape.name.Split(new string[] { ShapeMatDelim }, StringSplitOptions.RemoveEmptyEntries);
                if (splitedStrings.Length != 2)
                {
                    // ボーン名かマテリアル名に "__" が使われているケースはあきらめる
                    return null;
                }

                return splitedStrings[0];
            }
         }

        private List<DccToolShapeUnit> CreateDccToolShapes(modelType model)
        {
            List<DccToolShapeUnit> dccToolShapes = new List<DccToolShapeUnit>();

            foreach (shapeType shape in model.shape_array.shape)
            {
                vertexType vertexData = model.vertex_array.vertex[shape.shape_info.vertex_index];
                var posAttr = vertexData.vtx_attrib_array.vtx_attrib.FirstOrDefault(x => IsPositionAttr(x));
                if (posAttr == null)
                {
                    continue;
                }

                string dccToolShapeNameText = shape.shape_info.original_bone_name;
                if (string.IsNullOrEmpty(dccToolShapeNameText))
                {
                    dccToolShapeNameText = EstimateOriginalBoneName(shape, model.shape_array.shape);
                }

                if (string.IsNullOrEmpty(dccToolShapeNameText))
                {
                    continue;
                }

                string[] dccToolShapeNames = dccToolShapeNameText.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                foreach (string dccToolShapeName in dccToolShapeNames)
                {

                    DccToolShapeUnit dccToolShape = dccToolShapes.FirstOrDefault(x => x.Name == dccToolShapeName);
                    if (dccToolShape == null)
                    {
                        dccToolShape = new DccToolShapeUnit() { Name = dccToolShapeName };
                        dccToolShapes.Add(dccToolShape);
                    }

                    dccToolShape.ChildShapes.Add(shape);

                    dccToolShape.UpdatePositionQuantizeType(posAttr.quantize_type);
                }
            }

            return dccToolShapes;
        }

        private bool IsPositionAttr(vtx_attribType vertexAttr)
        {
            return vertexAttr.name.StartsWith("_p") && vertexAttr.hint.StartsWith("position");
        }

        private void MakeVectorList(G3dStream stream, out StreamSet streamSet, int baseStreamCount)
        {
            streamSet = new StreamSet();
            streamSet.floatList = new List<float>(stream.FloatData);
            streamSet.intList = new List<int>(stream.IntData);
            if (baseStreamCount > 0 && stream.FloatData.Count > 0)
            {
                streamSet.floatBaseList = new List<float>(stream.FloatData.GetRange(0, baseStreamCount));
            }
            else
            {
                streamSet.floatBaseList = new List<float>(stream.FloatData);
            }

            if (streamSet.floatList.Count != 0)
            {
                streamSet.floatList.Sort();
            }

            if (streamSet.intList.Count != 0)
            {
                streamSet.intList.Sort();
            }

            if (streamSet.floatBaseList.Count != 0)
            {
                streamSet.floatBaseList.Sort();
            }
        }

        private void AddVectorList(G3dStream stream, StreamSet streamSet, int baseVertexCount)
        {
            streamSet.floatList.AddRange(stream.FloatData);
            streamSet.intList.AddRange(stream.IntData);
            if (baseVertexCount > 0 && stream.FloatData.Count > 0)
            {
                streamSet.floatBaseList.AddRange(stream.FloatData.GetRange(0, baseVertexCount));
            }
            else
            {
                streamSet.floatBaseList.AddRange(stream.FloatData);
            }

            if (streamSet.floatList.Count != 0)
            {
                streamSet.floatList.Sort();
            }

            if (streamSet.intList.Count != 0)
            {
                streamSet.intList.Sort();
            }

            if (streamSet.floatBaseList.Count != 0)
            {
                streamSet.floatBaseList.Sort();
            }
        }

        /// <summary>
        /// 正規化が行われているデータかどうかを確認します。
        /// データ列は必ず Vector3 のデータである必要があります。
        /// </summary>
        private bool IsNormalized(G3dStream stream, out Vector3 errorVec)
        {
            float[] data = stream.FloatData.ToArray();
            int column = stream.column;
            const float E = 0.01f * 0.01f;
            for (int i = 0; i < data.Length / column; ++i)
            {
                Vector3 vec = new Vector3(data[i * column], data[i * column + 1], data[i * column + 2]);
                float len = vec.Length;
                if (1.0f + E < len || len < 1.0f - E)
                {
                    errorVec = vec;
                    return false;
                }
            }
            errorVec = new Vector3();
            return true;
        }

        /// <summary>
        /// 最少誤差を返します。
        /// </summary>
        private float GetMinDiff(List<float> floatList)
        {
            const float MAX_DIFF = 100000.0f;
            float diff = MAX_DIFF;

            if (floatList.Count > 1)
            {
                for (int i = 1; i < floatList.Count; ++i)
                {
                    float distance = Math.Abs(floatList[i] - floatList[i - 1]);

                    // 1/1280以下の違いはスケールなしでnearプレーンにべったりついた場合でも
                    // 出ないから。マージンを見て1/4096以下の違いは違いとみなさないことにする。
                    if (distance < diff)
                    {
                        if (distance >= (1.0 / 4096.0))
                        {
                            diff = distance;
                        }
                    }
                }
            }

            return diff;
        }

        /// <summary>
        /// 整数部分の桁数を返します。
        /// 最低でも halffloat で 1.0 の精度を保証します。
        /// </summary>
        private bool GetMinIntWidth(List<float> floatList, out int minInt)
        {
            float min = floatList[0];
            float max = floatList[floatList.Count - 1];

            max = Math.Abs(min) > Math.Abs(max) ? Math.Abs(min) : Math.Abs(max);
            uint ival = (uint)(Math.Floor(max + 0.5f));

            int i;
            // 半精度の int が表現できる最大値である 2^16
            for (i = 0; i <= 16; ++i)
            {
                if (ival < (1 << i))
                {
                    minInt = i;
                    return true;
                }
            }

            // ival >= 0x10000
            // 17ビット以上のものは量子化できない。
            minInt = 0;
            return false;
        }

        /// <summary>
        /// 量子化した場合の最大絶対誤差を求めます。
        /// </summary>
        private float GetMaxAbsError(List<float> floatList, int frac)
        {
            double rval = 0.0;
            double fracValue = Math.Pow(2.0, frac);

            for (int i = 0; i < floatList.Count; ++i)
            {
                float value = floatList[i];
                double tmp = Math.Abs(value * fracValue);
                double e = Math.Abs(tmp - Math.Floor(tmp + 0.5));

                if (e > rval)
                {
                    rval = e;
                }
            }
            return (float)(rval * Math.Pow(0.5, frac));
        }

        /// <summary>
        /// 半精度浮動小数に量子化した場合の最大絶対誤差を求めます。
        /// </summary>
        private float GetMaxAbsErrorHalf(List<float> floatList)
        {
            float ret = 0.0f;
            foreach (var value in floatList)
            {
                int binary = BitConverter.ToInt32(BitConverter.GetBytes(value), 0);
                float half = BitConverter.ToSingle(BitConverter.GetBytes(binary & 0xFFFFE000), 0);
                ret = Math.Max(ret, Math.Abs(value - half));
            }
            return ret;
        }

        /// <summary>
        /// 量子化した場合の最大相対誤差を求めます。
        /// </summary>
        private float GetMaxRelError(StreamSet streamSet, int frac)
        {
            double rval = 0.0;
            double fracValue = Math.Pow(2.0, frac);

            for (int i = 0; i < streamSet.floatList.Count; ++i)
            {
                float value = streamSet.floatList[i];
                double tmp = Math.Abs(value * fracValue);
                double e = Math.Abs(tmp - Math.Floor(tmp + 0.5)) / tmp;

                if (e > rval)
                {
                    rval = e;
                }
            }

            return (float)rval;
        }

        private vtx_attrib_quantize_typeType QuantizePos(StreamSet streamSet)
        {
            int minInt = 0;
            bool canQuantize = GetMinIntWidth(streamSet.floatList, out minInt); // この minInt は使用しない

            // 誤差はベースモデルの頂点のみを見て判別する
            GetMinIntWidth(streamSet.floatBaseList, out minInt);

            if (canQuantize)
            {
                // 誤差はベースモデルの頂点のみを見て判別する
                float mindiff = GetMinDiff(streamSet.floatBaseList);

#if false
                if (streamSet.floatList[0] >= 0)
                {
                    // 負の数がデータの中に１つもなかった場合
                    int frac_u10 = 10 - minInt;
                    if (frac_u10 >= 0)
                    {
                        // float10_11_11 でOKかどうかテスト
                        if (mindiff > Math.Pow(0.5, frac_u10))
                        {
                            if (GetMaxAbsError(streamSet, frac_u10) < POS_DIFF_MAX)
                            {
                                return vtx_attrib_quantize_typeType.float_10_11_11;
                            }
                        }
                    }
                }
#endif

                int frac_s16 = 15 - minInt;
                if (frac_s16 >= 0)
                {
                    if ((mindiff > Math.Pow(0.5, frac_s16)))
                    {
                        if (GetMaxAbsErrorHalf(streamSet.floatBaseList) < PosDiffMax)
                        {
                            return vtx_attrib_quantize_typeType.float_16_16_16_16;
                        }
                    }
                }
            }

            return vtx_attrib_quantize_typeType.float_32_32_32;
        }

        private vtx_attrib_quantize_typeType QuantizeNorm(StreamSet streamSet)
        {
            float min = streamSet.floatList[0];
            float max = streamSet.floatList[streamSet.floatList.Count - 1];
            max = Math.Abs(min) > Math.Abs(max) ? Math.Abs(min) : Math.Abs(max);

            // 絶対値が 1.0 より小さいなら強制的に snorm_10_10_10_2 にする。
            if (max <= 1.0f)
            {
                return vtx_attrib_quantize_typeType.snorm_10_10_10_2;
            }

            return vtx_attrib_quantize_typeType.float_32_32_32;
        }

        private vtx_attrib_quantize_typeType QuantizeTangBinorm(StreamSet streamSet)
        {
            float min = streamSet.floatList[0];
            float max = streamSet.floatList[streamSet.floatList.Count - 1];
            max = Math.Abs(min) > Math.Abs(max) ? Math.Abs(min) : Math.Abs(max);

            // 絶対値が 1.0 より小さいなら強制的に snorm_8_8_8_8 にする。
            if (max <= 1.0f)
            {
                return vtx_attrib_quantize_typeType.snorm_8_8_8_8;
            }

            return vtx_attrib_quantize_typeType.float_32_32_32_32;
        }

        private vtx_attrib_quantize_typeType QuantizeColor(StreamSet streamSet, vtx_attrib_typeType type)
        {
            var quantize_table_u8 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.unorm_8,
                vtx_attrib_quantize_typeType.unorm_8_8,
                vtx_attrib_quantize_typeType.unorm_8_8_8_8,
                vtx_attrib_quantize_typeType.unorm_8_8_8_8
            };

            var quantize_table_s8 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.snorm_8,
                vtx_attrib_quantize_typeType.snorm_8_8,
                vtx_attrib_quantize_typeType.snorm_8_8_8_8,
                vtx_attrib_quantize_typeType.snorm_8_8_8_8
            };

            var quantize_table_f16 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.float_16,
                vtx_attrib_quantize_typeType.float_16_16,
                vtx_attrib_quantize_typeType.float_16_16_16_16,
                vtx_attrib_quantize_typeType.float_16_16_16_16
            };

            var quantize_table_f32 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.float_32,
                vtx_attrib_quantize_typeType.float_32_32,
                vtx_attrib_quantize_typeType.float_32_32_32,
                vtx_attrib_quantize_typeType.float_32_32_32_32
            };

            int typeIndex = (int)type % 4;
            float min = streamSet.floatList[0];
            float max = streamSet.floatList[streamSet.floatList.Count - 1];
            max = Math.Abs(min) > Math.Abs(max) ? Math.Abs(min) : Math.Abs(max);

            // 負の数がデータの中に１つもなかった場合
            if (max <= 1.0f)
            {
                if (streamSet.floatList[0] >= 0)
                {
                    return quantize_table_u8[typeIndex];
                }
                else
                {
                    return quantize_table_s8[typeIndex];
                }
            }
            else
            {
                int minInt = 0;
                bool canQuantize = GetMinIntWidth(streamSet.floatList, out minInt);

                if (canQuantize)
                {
                    return quantize_table_f16[typeIndex];
                }
            }

            return quantize_table_f32[typeIndex];
        }

        private vtx_attrib_quantize_typeType QuantizeUv(StreamSet streamSet)
        {
            float min = streamSet.floatList[0];
            float max = streamSet.floatList[streamSet.floatList.Count - 1];
            max = Math.Abs(min) > Math.Abs(max) ? Math.Abs(min) : Math.Abs(max);

            // 負の数がデータの中に１つもなかった場合
            if (max <= 1.0f)
            {
                // 誤差はベースモデルの頂点のみを見て判別する
                float mindiff = GetMinDiff(streamSet.floatBaseList);

                if (streamSet.floatList[0] >= 0)
                {
                    if (mindiff >= Math.Pow(0.5, 8))
                    {
                        return vtx_attrib_quantize_typeType.unorm_8_8;
                    }

                    if (mindiff >= Math.Pow(0.5, 16))
                    {
                        return vtx_attrib_quantize_typeType.unorm_16_16;
                    }
                }
                else
                {
                    if (mindiff >= Math.Pow(0.5, 7))
                    {
                        return vtx_attrib_quantize_typeType.snorm_8_8;
                    }

                    if (mindiff >= Math.Pow(0.5, 15))
                    {
                        return vtx_attrib_quantize_typeType.snorm_16_16;
                    }
                }
            }

            int minInt = 0;
            bool canQuantize = GetMinIntWidth(streamSet.floatList, out minInt);

            if (canQuantize)
            {
                // 誤差はベースモデルの頂点のみを見て判別する
                float diff = GetMaxAbsErrorHalf(streamSet.floatBaseList);

                if (diff < UvDiffMax)
                {
                    return vtx_attrib_quantize_typeType.float_16_16;
                }
            }

            return vtx_attrib_quantize_typeType.float_32_32;
        }

        private vtx_attrib_quantize_typeType QuantizeWeight(StreamSet streamSet, vtx_attrib_typeType type, vtx_attrib_quantize_typeType refType)
        {
            var quantize_table_u8 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.unorm_8,
                vtx_attrib_quantize_typeType.unorm_8_8,
                vtx_attrib_quantize_typeType.unorm_8_8_8_8,
                vtx_attrib_quantize_typeType.unorm_8_8_8_8
            };

            var quantize_table_u16 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.unorm_16,
                vtx_attrib_quantize_typeType.unorm_16_16,
                vtx_attrib_quantize_typeType.unorm_16_16_16_16,
                vtx_attrib_quantize_typeType.unorm_16_16_16_16
            };

            var quantize_table_f32 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.float_32,
                vtx_attrib_quantize_typeType.float_32_32,
                vtx_attrib_quantize_typeType.float_32_32_32,
                vtx_attrib_quantize_typeType.float_32_32_32_32
            };

            int typeIndex = (int)type % 4;

            if (refType == vtx_attrib_quantize_typeType.none)
            {
                // 誤差はベースモデルの頂点のみを見て判別する
                float mindiff = GetMinDiff(streamSet.floatBaseList);

                if (mindiff >= Math.Pow(0.5, 8))
                {
                    return quantize_table_u8[typeIndex];
                }

                if (mindiff >= Math.Pow(0.5, 16))
                {
                    return quantize_table_u16[typeIndex];
                }
            }
            else if (refType >= vtx_attrib_quantize_typeType.unorm_8 &&
                    refType <= vtx_attrib_quantize_typeType.unorm_8_8_8_8)
            {
                return quantize_table_u8[typeIndex];
            }
            else if (refType >= vtx_attrib_quantize_typeType.unorm_16 &&
                    refType <= vtx_attrib_quantize_typeType.unorm_16_16_16_16)
            {
                return quantize_table_u16[typeIndex];
            }

            return quantize_table_f32[typeIndex];
        }

        private vtx_attrib_quantize_typeType QuantizeIndex(StreamSet streamSet, vtx_attrib_typeType type, vtx_attrib_quantize_typeType refType)
        {
            var quantize_table_u8 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.uint_8,
                vtx_attrib_quantize_typeType.uint_8_8,
                vtx_attrib_quantize_typeType.uint_8_8_8_8,
                vtx_attrib_quantize_typeType.uint_8_8_8_8
            };

            var quantize_table_u16 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.uint_16,
                vtx_attrib_quantize_typeType.uint_16_16,
                vtx_attrib_quantize_typeType.uint_16_16_16_16,
                vtx_attrib_quantize_typeType.uint_16_16_16_16
            };

            var quantize_table_u32 = new vtx_attrib_quantize_typeType[4]
            {
                vtx_attrib_quantize_typeType.uint_32,
                vtx_attrib_quantize_typeType.uint_32_32,
                vtx_attrib_quantize_typeType.uint_32_32_32_32,
                vtx_attrib_quantize_typeType.uint_32_32_32_32
            };

            int typeIndex = (int)type % 4;
            if (refType == vtx_attrib_quantize_typeType.none)
            {
                uint data_max = (uint)streamSet.intList[streamSet.intList.Count - 1];

                const uint UINT8_MAX = 255;
                const uint UINT16_MAX = ushort.MaxValue;

                if (data_max < UINT8_MAX)
                {
                    return quantize_table_u8[typeIndex];
                }
                else if (data_max < UINT16_MAX)
                {
                    return quantize_table_u16[typeIndex];
                }
            }
            else if (refType >= vtx_attrib_quantize_typeType.uint_8 &&
                    refType <= vtx_attrib_quantize_typeType.uint_8_8_8_8)
            {
                return quantize_table_u8[typeIndex];
            }
            else if (refType >= vtx_attrib_quantize_typeType.uint_16 &&
                    refType <= vtx_attrib_quantize_typeType.uint_32_32_32_32)
            {
                return quantize_table_u16[typeIndex];
            }

            return quantize_table_u32[typeIndex];
        }
    }
}
