﻿using System;
using System.Collections.Generic;
using System.Linq;

using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シェーダーパラメーターアニメーションの量子化分析
    public class IfMaterialAnimQuantizationAnalysisOptimizer
        : IfMaterialAnimOptimizer
    {
        // コンストラクタ
        public IfMaterialAnimQuantizationAnalysisOptimizer() :
            base("IfMaterialAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        private int FrameSize;
        private int QuantizedFrameSize;

        private class ParamTarget
        {
            public shader_param_typeType type { get; set; }
            public param_anim_targetType target { get; set; } = null;
        }

        // 最適化
        protected override void Optimize()
        {
            var materialAnim = this.Target;
            if (materialAnim.per_material_anim_array == null
                || materialAnim.per_material_anim_array.per_material_anim == null
                || materialAnim.per_material_anim_array.per_material_anim.Length == 0)
            {
                return;
            }

            var materialArray = materialAnim.per_material_anim_array.Items;
            var materialAnimInfo = materialAnim.material_anim_info;

            // シェーダーパラメーターアニメーションの存在を確認する。
            var hasShaderParameterAnim =
                materialArray.Any(x => x.shader_param_anim_array != null);

            // テクスチャパターンアニメーションの存在を確認する。
            var hasTexPatternAnim = (materialAnim.tex_pattern_array != null
                                     && materialAnim.tex_pattern_array.tex_pattern != null
                                     && materialAnim.tex_pattern_array.tex_pattern.Length > 0
                                     && materialAnim.per_material_anim_array.per_material_anim.Any(
                                         x => x.tex_pattern_anim_array != null && x.tex_pattern_anim_array.length > 0));

            // マテリアルビジビリティアニメーションの存在を確認する。
            var hasVisibilityAnim =
                materialArray.Any(x => x.material_visibility_anim != null);
            if (!hasShaderParameterAnim && !hasTexPatternAnim && !hasVisibilityAnim)
            {
                return;
            }

            // シェーダーパラメーターアニメーション最適化
            if (hasShaderParameterAnim)
            {
                var curveDic = new Dictionary<ParamTarget, IfAnimCurve>();
                var shaderAnims =
                    materialAnim.per_material_anim_array.per_material_anim.Where(
                        x =>
                        x.shader_param_anim_array != null && x.shader_param_anim_array.param_anim != null
                        && x.shader_param_anim_array.param_anim.Length > 0)
                        .SelectMany(x => x.shader_param_anim_array.param_anim)
                        .Where(x => x.param_anim_target != null && x.param_anim_target.Length > 0);

                // カーブデータを構築する

                foreach (var param in shaderAnims)
                {
                    foreach (var target in param.param_anim_target)
                    {
                        if (target.Item == null)
                        {
                            continue;
                        }
                        var animCurve = new IfAnimCurve();
                        switch (param.type)
                        {
                            case shader_param_typeType.@bool:
                            case shader_param_typeType.bool2:
                            case shader_param_typeType.bool3:
                            case shader_param_typeType.bool4:
                                animCurve.curveType = IfAnimCurve.CurveType.Bool;
                                break;
                            case shader_param_typeType.@int:
                            case shader_param_typeType.int2:
                            case shader_param_typeType.int4:
                            case shader_param_typeType.int3:
                            case shader_param_typeType.@uint:
                            case shader_param_typeType.uint2:
                            case shader_param_typeType.uint3:
                            case shader_param_typeType.uint4:
                                animCurve.curveType = IfAnimCurve.CurveType.Int;
                                break;
                            default:
                                animCurve.curveType = IfAnimCurve.CurveType.Float;
                                break;
                        }

                        // シェーダーパラメータータイプが srt2d と srt3d と texsrt の場合はランタイムの動作に
                        // 合わせて Radian に変換してから量子化解析を行う。
                        var isDegree = false;
                        var compIdx = target.component_index;
                        switch (param.type)
                        {
                            case shader_param_typeType.srt2d:
                                if (compIdx == (int)IfShaderParamAnimQuantizationAnalysis.Srt2dIndex.Rotate)
                                {
                                    isDegree = true;
                                }
                                break;
                            case shader_param_typeType.srt3d:
                                if (compIdx == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateX
                                    || compIdx == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateY
                                    || compIdx == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateZ)
                                {
                                    isDegree = true;
                                }
                                break;
                            case shader_param_typeType.texsrt:
                                if (compIdx == (int)IfShaderParamAnimQuantizationAnalysis.TexsrtIndex.Rotate)
                                {
                                    isDegree = true;
                                }
                                break;
                        }

                        animCurve.BuildIfAnimCurve(target.Item, isDegree, this.Streams);

                        var paramTarget = new ParamTarget { type = param.type, target = target };
                        curveDic.Add(paramTarget, animCurve);
                    }
                }

                foreach (var curve in curveDic)
                {
                    // カーブの量子化を評価して中間ファイルに適用する
                    var result = IfMaterialAnimQuantizationAnalysis.AnalyseShaderParameterCurve(
                        materialAnimInfo,
                        curve.Key.type,
                        curve.Key.target.component_index,
                        curve.Value,
                        false);

                    this.EnableProcessLog |= !result.skipped;
                    ApplyQuantize(curve.Key, result);
                }
            }

            // テクスチャパターンアニメーション最適化
            if (hasTexPatternAnim)
            {
                var curveDic = new Dictionary<pattern_anim_targetType, IfAnimCurve>();
                var texPatternAnims =
                    materialAnim.per_material_anim_array.per_material_anim.Where(
                        x =>
                        x.tex_pattern_anim_array != null && x.tex_pattern_anim_array.pattern_anim != null
                        && x.tex_pattern_anim_array.pattern_anim.Length > 0)
                        .SelectMany(x => x.tex_pattern_anim_array.pattern_anim)
                        .Where(x => x.step_curve != null);

                // カーブデータを構築する
                foreach (var target in texPatternAnims)
                {
                    var animCurve = new IfAnimCurve { curveType = IfAnimCurve.CurveType.Int };
                    animCurve.BuildIfAnimCurve(target.step_curve, false, this.Streams);
                    curveDic.Add(target, animCurve);
                }

                // frame の量子化を評価して type を選択する。
                foreach (var curve in curveDic)
                {
                    var result = IfMaterialAnimQuantizationAnalysis.AnalyseTexPatternCurve(
                        materialAnimInfo,
                        curve.Value,
                        false);

                    this.EnableProcessLog |= !result.skipped;

                    ApplyQuantize(curve.Key, result);
                }
            }

            // マテリアルビジビリティアニメーション最適化
            if (hasVisibilityAnim)
            {
                var curveDic =
                    new Dictionary<material_visibility_animType, IfAnimCurve>();
                var visibilityAnims =
                    materialAnim.per_material_anim_array.per_material_anim
                        .Where(
                            x => x.material_visibility_anim != null && x.material_visibility_anim.step_curve != null)
                        .Select(x => x.material_visibility_anim);

                // カーブデータを構築する
                foreach (var visAnim in visibilityAnims)
                {
                    var animCurve = new IfAnimCurve();
                    animCurve.curveType = IfAnimCurve.CurveType.Bool;
                    animCurve.BuildIfAnimCurve(visAnim.step_curve, false, this.Streams);
                    curveDic.Add(visAnim, animCurve);
                }

                // frame の量子化を評価して type を選択する。
                foreach (var curve in curveDic)
                {
                    IfQuantizationAnalysisResult result =
                        IfMaterialAnimQuantizationAnalysis.AnalyseMaterialVisibilityCurve(
                            materialAnimInfo,
                            curve.Value,
                            false);

                    this.EnableProcessLog |= !result.skipped;

                    ApplyQuantize(curve.Key, result);
                }
            }
        }

        /// <summary>
        /// シェーダーパラメーターアニメーションのIfAnimCurve を生成します。
        /// </summary>
        public static IfAnimCurve BuildIfAnimCurve(IG3dQuantizedCurve curve, shader_param_typeType paramType, int componentIndex, G3dStream stream)
        {
            IfAnimCurve animCurve = new IfAnimCurve();
            switch (paramType)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.bool2:
                case shader_param_typeType.bool3:
                case shader_param_typeType.bool4:
                    animCurve.curveType = IfAnimCurve.CurveType.Bool;
                    break;
                case shader_param_typeType.@int:
                case shader_param_typeType.int2:
                case shader_param_typeType.int4:
                case shader_param_typeType.int3:
                case shader_param_typeType.@uint:
                case shader_param_typeType.uint2:
                case shader_param_typeType.uint3:
                case shader_param_typeType.uint4:
                    animCurve.curveType = IfAnimCurve.CurveType.Int;
                    break;
                default:
                    animCurve.curveType = IfAnimCurve.CurveType.Float;
                    break;
            }

            // シェーダーパラメータータイプが srt2d と srt3d の場合はランタイムの動作に
            // 合わせて Radian に変換してから量子化解析を行う。
            bool isDegree = false;
            if (paramType == shader_param_typeType.srt2d)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt2dIndex.Rotate)
                {
                    isDegree = true;
                }
            }
            else if (paramType == shader_param_typeType.srt3d)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateX ||
                    componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateY ||
                    componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.Srt3dIndex.RotateZ)
                {
                    isDegree = true;
                }
            }
            else if (paramType == shader_param_typeType.texsrt)
            {
                if (componentIndex == (int)IfShaderParamAnimQuantizationAnalysis.TexsrtIndex.Rotate)
                {
                    isDegree = true;
                }
            }

            animCurve.BuildIfAnimCurve(curve, isDegree, stream);

            return animCurve;
        }

        /// <summary>
        /// シェーダーパラメーターアニメーションに量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(ParamTarget paramTarget, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            if (paramTarget.target.Item is hermite_curveType)
            {
                hermite_curveType hermiteCurve = paramTarget.target.Item as hermite_curveType;
                hermiteCurve.frame_type = analysisResult.frameType;
                hermiteCurve.key_type = analysisResult.keyType;
                hermiteCurve.scale = analysisResult.scale;
                hermiteCurve.offset = analysisResult.offset;

                frameType = hermiteCurve.frame_type;
                keyType = hermiteCurve.key_type;
                frameCount = hermiteCurve.count;
            }
            else if (paramTarget.target.Item is linear_curveType)
            {
                linear_curveType linearCurve = paramTarget.target.Item as linear_curveType;
                linearCurve.frame_type = analysisResult.frameType;
                linearCurve.key_type = analysisResult.keyType;
                linearCurve.scale = analysisResult.scale;
                linearCurve.offset = analysisResult.offset;

                frameType = linearCurve.frame_type;
                keyType = linearCurve.key_type;
                frameCount = linearCurve.count;
            }
            else if (paramTarget.target.Item is step_curveType)
            {
                step_curveType stepCurve = paramTarget.target.Item as step_curveType;
                stepCurve.frame_type = analysisResult.frameType;
                stepCurve.key_type = analysisResult.keyType;
                stepCurve.scale = analysisResult.scale;
                stepCurve.offset = analysisResult.offset;

                frameType = stepCurve.frame_type;
                keyType = stepCurve.key_type;
                frameCount = stepCurve.count;
            }
            else
            {
                throw new NotImplementedException();
            }

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }

        /// <summary>
        /// テクスチャパターンアニメーションに量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(pattern_anim_targetType target, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            step_curveType stepCurve = target.step_curve;
            stepCurve.frame_type = analysisResult.frameType;
            stepCurve.key_type = analysisResult.keyType;
            stepCurve.scale = analysisResult.scale;
            stepCurve.offset = analysisResult.offset;

            frameType = stepCurve.frame_type;
            keyType = stepCurve.key_type;
            frameCount = stepCurve.count;

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }

        /// <summary>
        /// マテリアルビジビリティアニメーションに量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(material_visibility_animType target, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            step_curveType stepCurve = target.step_curve;
            stepCurve.frame_type = analysisResult.frameType;
            stepCurve.key_type = analysisResult.keyType;
            stepCurve.scale = analysisResult.scale;
            stepCurve.offset = analysisResult.offset;

            frameType = stepCurve.frame_type;
            keyType = stepCurve.key_type;
            frameCount = stepCurve.count;

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
