﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    [Serializable]
    public class IfQuantizationAnalysisResult
    {
        public IfQuantizationAnalysisResult()
        {
            frameType = curve_frame_typeType.none;
            keyType = curve_key_typeType.none;
            scale = 1.0f;
            offset = 0.0f;
            skipped = false;
        }

        public curve_frame_typeType frameType{ get; set; }
        public curve_key_typeType keyType{ get; set; }
        public float scale{ get; set; }
        public float offset{ get; set; }
        public bool skipped { get; set; }
    }

    public class IfAnimCurveUtility
    {
        // float から 16bit 固定小数に変換して返します。
        public static short QuantizeFrameTo16(float frame)
        {
            if (frame >= 1024.0f ||
                frame <= -1024.0f)
            {
                return 0x0;
            }
            else
            {
                return Convert.ToInt16(frame * (float)Math.Pow(2.0, 5));
            }
        }

        public static float DequantizeFrameFrom16(short frame)
        {
            return Convert.ToSingle(frame) * (1.0f / (float)Math.Pow(2.0, 5));
        }

        // float から 8bit 整数に変換して返します。
        public static byte QuantizeFrameTo8(float frame)
        {
            if (frame < 0.0f ||
                frame > 255.0f)
            {
                return 0x0;
            }
            else
            {
                return Convert.ToByte(frame);
            }
        }

        public static float DequantizeFrameFrom8(byte frame)
        {
            return Convert.ToSingle(frame);
        }
    }

    public class IfAnimCurve
    {
        private void FindFrame(float frame, out int keyIndex, out float param)
        {
            int endFrameIndex = this.numFrame - 1;
            float start = 0.0f;
            float rcpTotal = 0.0f;

            // frame が所属するキーを二分探索で見つける。
            // 長さ0のキーは終端を除いて存在しない前提。
            if (this.frameType == curve_frame_typeType.frame32 ||
                this.frameType == curve_frame_typeType.none)
            {
                // 量子化しないで比較する。
                if (frame >= this.frameValues[endFrameIndex].frame32)
                {
                    // 終端フレームにちょうど一致した場合は以下のアルゴリズムでは
                    // 範囲外アクセスしてしまうので、特別処理で判定する。
                    keyIndex = endFrameIndex;
                    param = 0.0f;
                    return;
                }

                // 探索コストが frame の値に依存しない二分探索を行う。
                // 32ビットに限定した実装となっている
                int lower = 0;
                int upper = endFrameIndex;
                do
                {
                    int center = (lower + upper) >> 1;
                    int mask = 0x0;
                    if (frame < this.frameValues[center].frame32)
                    {
                        mask = ~0x0;
                    }
                    // A == B ^ A ^ B
                    upper = ((upper ^ (center - 1)) & mask) ^ upper;            // upper or (center - 1)
                    lower = ((++center ^ lower) & mask) ^ center;               // (center + 1) or lower
                    // center が目標値だった場合は通り過ぎるが境界がそのまま目標値に隣接し続ける。
                } while (lower <= upper);
                if (frame < this.frameValues[lower].frame32) // 範囲を検索するので1つは行き過ぎる可能性がある。
                {
                    --lower;
                }
                keyIndex = lower;

                start = this.frameValues[keyIndex].frame32;
                rcpTotal = 1.0f / (this.frameValues[keyIndex + 1].frame32 - start);
            }
            else if (this.frameType == curve_frame_typeType.frame16)
            {
                short quantized = IfAnimCurveUtility.QuantizeFrameTo16(frame);
                if (quantized >= this.frameValues[endFrameIndex].frame16)
                {
                    // 終端フレームにちょうど一致した場合は以下のアルゴリズムでは
                    // 範囲外アクセスしてしまうので、特別処理で判定する。
                    keyIndex = endFrameIndex;
                    param = 0.0f;
                    return;
                }

                // 探索コストが frame の値に依存しない二分探索を行う。
                // 32ビットに限定した実装となっている
                int lower = 0;
                int upper = endFrameIndex;
                do
                {
                    int center = (lower + upper) >> 1;
                    int mask = 0x0;
                    if (quantized < this.frameValues[center].frame16)
                    {
                        mask = ~0x0;
                    }
                    // A == B ^ A ^ B
                    upper = ((upper ^ (center - 1)) & mask) ^ upper;            // upper or (center - 1)
                    lower = ((++center ^ lower) & mask) ^ center;               // (center + 1) or lower
                    // center が目標値だった場合は通り過ぎるが境界がそのまま目標値に隣接し続ける。
                } while (lower <= upper);
                if (quantized < this.frameValues[lower].frame16) // 範囲を検索するので1つは行き過ぎる可能性がある。
                {
                    --lower;
                }
                keyIndex = lower;

                start = IfAnimCurveUtility.DequantizeFrameFrom16(this.frameValues[keyIndex].frame16);
                rcpTotal = 1.0f / (IfAnimCurveUtility.DequantizeFrameFrom16(this.frameValues[keyIndex + 1].frame16) - start);
            }
            else
            {
                byte quantized = IfAnimCurveUtility.QuantizeFrameTo8(frame);
                if (quantized >= this.frameValues[endFrameIndex].frame8)
                {
                    // 終端フレームにちょうど一致した場合は以下のアルゴリズムでは
                    // 範囲外アクセスしてしまうので、特別処理で判定する。
                    keyIndex = endFrameIndex;
                    param = 0.0f;
                    return;
                }

                // 探索コストが frame の値に依存しない二分探索を行う。
                // 32ビットに限定した実装となっている
                int lower = 0;
                int upper = endFrameIndex;
                do
                {
                    int center = (lower + upper) >> 1;
                    int mask = 0x0;
                    if (quantized < this.frameValues[center].frame8)
                    {
                        mask = ~0x0;
                    }
                    // A == B ^ A ^ B
                    upper = ((upper ^ (center - 1)) & mask) ^ upper;            // upper or (center - 1)
                    lower = ((++center ^ lower) & mask) ^ center;               // (center + 1) or lower
                    // center が目標値だった場合は通り過ぎるが境界がそのまま目標値に隣接し続ける。
                } while (lower <= upper);
                if (quantized < this.frameValues[lower].frame8) // 範囲を検索するので1つは行き過ぎる可能性がある。
                {    // キーが２つのみで、その２つが同じ時間を持つ場合に lower が 0 になる。
                    // その時に lower をデクリメントしないように値を確認する。
                    if (lower > 0)
                    {
                        --lower;
                    }
                }
                keyIndex = lower;

                start = IfAnimCurveUtility.DequantizeFrameFrom8(this.frameValues[keyIndex].frame8);
                rcpTotal = 1.0f / (IfAnimCurveUtility.DequantizeFrameFrom8(this.frameValues[keyIndex + 1].frame8) - start);
            }

            // セグメント内での比率を求める。
            param = (frame - start) * rcpTotal;
        }

        private float EvalCubic(float frame)
        {
            int keyIndex = 0;
            float param = 0.0f;

            FindFrame(frame, out keyIndex, out param);

            return this.hermiteValues[keyIndex].Get(param, this.keyType);
        }

        private float EvalLinear(float frame)
        {
            int keyIndex = 0;
            float param = 0.0f;

            FindFrame(frame, out keyIndex, out param);

            return this.linearValues[keyIndex].Get(param, this.keyType);
        }

        private float EvalStep(float frame)
        {
            // 直前のKeyの値を返すようにしておきます。
            int keyIndex = 0;
            float param = 0.0f;
            FindFrame(frame, out keyIndex, out param);
            Nintendo.Foundation.Contracts.Assertion.Operation.True(keyIndex < this.stepValues.Count);
            return this.stepValues[keyIndex].Get(this.keyType);
        }

        private float WrapFrame(float frame, float startFrame, float endFrame, curve_wrapType preWrap, curve_wrapType postWrap)
        {
            // カーブ単位でのフレームの折り返しを処理する。
            float upper = frame - endFrame;
            float lower = startFrame - frame;
            float frameDiff;
            bool outWrap;
            curve_wrapType wrapType;
            if (upper * lower >= 0)
            {
                return frame; // 既に範囲内
            }
            else if (lower > 0)
            {
                wrapType = preWrap;
                if (curve_wrapType.clamp == preWrap)
                {
                    return startFrame;
                }

                frameDiff = lower;
                outWrap = false;
            }
            else
            {
                wrapType = postWrap;
                if (curve_wrapType.clamp == postWrap)
                {
                    return endFrame;
                }

                frameDiff = upper;
                outWrap = true;
            }

            // クランプ以外の場合の処理。
            float duration = endFrame - startFrame;
            int count = (int)(frameDiff / duration);
            float frameMod = frameDiff - duration * count;
            bool repeat = (curve_wrapType.repeat == wrapType) ||
                (curve_wrapType.mirror == wrapType && ((count & 0x1) == 1));
            if (repeat != outWrap) // outWrap の場合は inWarp と逆になるので裏返す。
            {
                frame = endFrame - frameMod;
            }
            else
            {
                frame = startFrame + frameMod;
            }

            return frame;
        }

        private float EvalFloat(float frame, float scale, curve_key_typeType keyType)
        {
            curve_key_typeType tempKeyType = this.keyType;
            float tempScale = this.scale;
            float tempOffset = this.offset;

            this.keyType = keyType;
            this.scale = scale;

            if (keyType == curve_key_typeType.key16 ||
                keyType == curve_key_typeType.key8)
            {
                this.offset = this.quantizedOffset;
            }
            else
            {
                this.offset = 0.0f;
            }

            float anmValue = this.EvalFloat(frame);

            this.keyType = tempKeyType;
            this.scale = tempScale;
            this.offset = tempOffset;

            return anmValue;
        }

        public float EvalFloat(float frame)
        {
            float wrappedFrame = WrapFrame(frame, this.startFrame, this.endFrame, this.preWrap, this.postWrap);
            float result = 0.0f;
            if (this.hermiteValues.Count != 0)
            {
                result = EvalCubic(wrappedFrame);
            }
            else if (this.linearValues.Count != 0)
            {
                result = EvalLinear(wrappedFrame);
            }
            else if (this.stepValues.Count != 0)
            {
                result = EvalStep(wrappedFrame);
            }
            else
            {
                throw new NotImplementedException();
            }

            return result * this.scale + this.offset;
        }

        /// <summary>
        /// hermite の係数を計算します。
        /// </summary>
        private void BuildHermite(float[] values, bool isDegreeValue, out HermiteValue hermite)
        {
            hermite = new HermiteValue();

            float f = values[(int)IfAnimCurve.HermiteDef.Frame1] - values[(int)IfAnimCurve.HermiteDef.Frame0];
            float v0 = values[(int)IfAnimCurve.HermiteDef.Value0];
            float v1 = values[(int)IfAnimCurve.HermiteDef.Value1];
            float so = values[(int)IfAnimCurve.HermiteDef.OutSlope0];
            float si = values[(int)IfAnimCurve.HermiteDef.InSlope1];

            if (isDegreeValue)
            {
                v0 = FloatUtility.DegreeToRadian(v0);
                v1 = FloatUtility.DegreeToRadian(v1);
                so = FloatUtility.DegreeToRadian(so);
                si = FloatUtility.DegreeToRadian(si);
            }

            // 32ビットフォーマット
            {
                float v = v1 - v0;
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef3] = -2 * v + (so + si) * f;
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2] = 3 * v - (2 * so + si) * f;
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef1] = so * f;
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] = v0;
            }
        }

        /// <summary>
        /// linear の係数を計算します。
        /// </summary>
        private void BuildLinear(float[] values, bool isDegreeValue, out LinearValue linear)
        {
            linear = new LinearValue();
            float v0 = values[(int)IfAnimCurve.LinearDef.Value0];
            float v1 = values[(int)IfAnimCurve.LinearDef.Value1];

            if (isDegreeValue)
            {
                v0 = FloatUtility.DegreeToRadian(v0);
                v1 = FloatUtility.DegreeToRadian(v1);
            }

            // 32ビットフォーマット
            {
                float v = v1 - v0;
                linear.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef1] = v;
                linear.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] = v0;
            }
        }

        /// <summary>
        /// step の係数を計算します。
        /// </summary>
        private void BuildStep(float[] values, bool isDegreeValue, out StepValue step)
        {
            step = new StepValue();
            float v0 = values[(int)IfAnimCurve.StepDef.Value0];

            if (isDegreeValue)
            {
                v0 = FloatUtility.DegreeToRadian(v0);
            }

            // 32ビットフォーマット
            {
                step.coefs32[(int)IfAnimCurve.StepCoefDef.Coef1] = 0.0f;
                step.coefs32[(int)IfAnimCurve.StepCoefDef.Coef0] = v0;
            }
        }

        /// <summary>
        /// step の係数を計算します。
        /// </summary>
        private void BuildStepInt(float[] values, bool isDegreeValue, out IntValue step)
        {
            step = new IntValue();
            float v0 = values[(int)IfAnimCurve.StepDef.Value0];

            if (isDegreeValue)
            {
                v0 = FloatUtility.DegreeToRadian(v0);
            }

            // 32ビットフォーマット
            {
                step.int32 = Convert.ToInt32(v0);
            }
        }

        /// <summary>
        /// step の係数を計算します。
        /// </summary>
        private void BuildStepBool(float[] values, out BoolValue step)
        {
            step = new BoolValue();
            float v0 = values[(int)IfAnimCurve.StepDef.Value0];

            // 32ビットフォーマット
            {
                step.value = Convert.ToBoolean(v0);
            }
        }

        /// <summary>
        /// scale と offset に応じて 量子化係数を計算します。
        /// </summary>
        public void QuantizeCurve()
        {
            if (this.hermiteValues.Count != 0)
            {
                QuantizeHermite();
            }
            else if (this.linearValues.Count != 0)
            {
                QuantizeLinear();
            }
            else if (this.stepValues.Count != 0)
            {
                QuantizeStep();
            }
            else if (this.intValues.Count != 0)
            {
                QuantizeStepInt();
            }
        }

        private void QuantizeHermite()
        {
            float maxCoef = float.MinValue;
            float minCoef = float.MaxValue;
            foreach (HermiteValue hermite in hermiteValues)
            {
                maxCoef = Math.Max(maxCoef, hermite.coefs32[0]);
                minCoef = Math.Min(minCoef, hermite.coefs32[0]);
            }

            // coef[0] 用の offset を計算する
            // offset は coef[0] の max と min の中間値
            quantizedOffset = (maxCoef + minCoef) / 2.0f;

            maxCoef = float.MinValue;
            minCoef = float.MaxValue;
            foreach (HermiteValue hermite in hermiteValues)
            {
                maxCoef = Math.Max(maxCoef, hermite.coefs32[0] - quantizedOffset);
                minCoef = Math.Min(minCoef, hermite.coefs32[0] - quantizedOffset);
                maxCoef = Math.Max(maxCoef, hermite.coefs32[1]);
                minCoef = Math.Min(minCoef, hermite.coefs32[1]);
                maxCoef = Math.Max(maxCoef, hermite.coefs32[2]);
                minCoef = Math.Min(minCoef, hermite.coefs32[2]);
                maxCoef = Math.Max(maxCoef, hermite.coefs32[3]);
                minCoef = Math.Min(minCoef, hermite.coefs32[3]);
            }

            // scale を計算
            scale16 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / short.MaxValue;
            scale8 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / sbyte.MaxValue;

            // slope が 0.0 になっている場合
            if (scale16 == 0.0f)
            {
                scale16 = 1.0f;
            }

            // slope が 0.0 になっている場合
            if (scale8 == 0.0f)
            {
                scale8 = 1.0f;
            }

            foreach (HermiteValue hermite in hermiteValues)
            {
                // エンコーディングは R エンコーディングを用いる。
                // ToInt16, ToSByte は数値を丸め（四捨五入）する。

                // 16ビットフォーマット
                {
                    hermite.coefs16[(int)IfAnimCurve.HermiteCoefDef.Coef3] =
                        Convert.ToInt16(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef3] / scale16);
                    hermite.coefs16[(int)IfAnimCurve.HermiteCoefDef.Coef2] =
                        Convert.ToInt16(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2] / scale16);
                    hermite.coefs16[(int)IfAnimCurve.HermiteCoefDef.Coef1] =
                        Convert.ToInt16(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef1] / scale16);
                    hermite.coefs16[(int)IfAnimCurve.HermiteCoefDef.Coef0] =
                        Convert.ToInt16((hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] - quantizedOffset) / scale16);
                }

                // 8ビットフォーマット
                {
                    hermite.coefs8[(int)IfAnimCurve.HermiteCoefDef.Coef3] =
                        Convert.ToSByte(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef3] / scale8);
                    hermite.coefs8[(int)IfAnimCurve.HermiteCoefDef.Coef2] =
                        Convert.ToSByte(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2] / scale8);
                    hermite.coefs8[(int)IfAnimCurve.HermiteCoefDef.Coef1] =
                        Convert.ToSByte(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef1] / scale8);
                    hermite.coefs8[(int)IfAnimCurve.HermiteCoefDef.Coef0] =
                        Convert.ToSByte((hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] - quantizedOffset) / scale8);
                }
            }
        }

        private void QuantizeLinear()
        {
            float maxCoef = float.MinValue;
            float minCoef = float.MaxValue;
            foreach (LinearValue linear in linearValues)
            {
                maxCoef = Math.Max(maxCoef, linear.coefs32[0]);
                minCoef = Math.Min(minCoef, linear.coefs32[0]);
            }

            // coef[0] 用の offset を計算する
            // offset は coef[0] の max と min の中間値
            quantizedOffset = (maxCoef + minCoef) / 2.0f;

            maxCoef = float.MinValue;
            minCoef = float.MaxValue;
            foreach (LinearValue linear in linearValues)
            {
                maxCoef = Math.Max(maxCoef, linear.coefs32[0] - quantizedOffset);
                minCoef = Math.Min(minCoef, linear.coefs32[0] - quantizedOffset);
                maxCoef = Math.Max(maxCoef, linear.coefs32[1]);
                minCoef = Math.Min(minCoef, linear.coefs32[1]);
            }

            // scale を計算
            scale16 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / short.MaxValue;
            if (scale16 == 0.0f)
            {
                scale16 = 1.0f;
            }

            scale8 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / sbyte.MaxValue;
            if (scale8 == 0.0f)
            {
                scale8 = 1.0f;
            }

            foreach (LinearValue linear in linearValues)
            {
                // 16ビットフォーマット
                {
                    linear.coefs16[(int)IfAnimCurve.LinearCoefDef.Coef1] =
                        Convert.ToInt16(linear.coefs32[(int)IfAnimCurve.LinearCoefDef.Coef1] / scale16);
                    linear.coefs16[(int)IfAnimCurve.LinearCoefDef.Coef0] =
                        Convert.ToInt16((linear.coefs32[(int)IfAnimCurve.LinearCoefDef.Coef0] - quantizedOffset) / scale16);
                }

                // 8ビットフォーマット
                {
                    linear.coefs8[(int)IfAnimCurve.LinearCoefDef.Coef1] =
                        Convert.ToSByte(linear.coefs32[(int)IfAnimCurve.LinearCoefDef.Coef1] / scale8);
                    linear.coefs8[(int)IfAnimCurve.LinearCoefDef.Coef0] =
                        Convert.ToSByte((linear.coefs32[(int)IfAnimCurve.LinearCoefDef.Coef0] - quantizedOffset) / scale8);
                }
            }
        }

        private void QuantizeStep()
        {
            float maxCoef = float.MinValue;
            float minCoef = float.MaxValue;
            foreach (StepValue step in stepValues)
            {
                maxCoef = Math.Max(maxCoef, step.coefs32[0]);
                minCoef = Math.Min(minCoef, step.coefs32[0]);
            }

            // coef[0] 用の offset を計算する
            // offset は coef[0] の max と min の中間値
            quantizedOffset = (maxCoef + minCoef) / 2.0f;

            maxCoef = float.MinValue;
            minCoef = float.MaxValue;
            foreach (StepValue step in stepValues)
            {
                maxCoef = Math.Max(maxCoef, step.coefs32[0] - quantizedOffset);
                minCoef = Math.Min(minCoef, step.coefs32[0] - quantizedOffset);
            }

            // 8,16 ビット用の scale を計算
            scale16 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / short.MaxValue;
            if (scale16 == 0.0f)
            {
                scale16 = 1.0f;
            }

            scale8 = Math.Max(Math.Abs(maxCoef), Math.Abs(minCoef)) / sbyte.MaxValue;
            if (scale8 == 0.0f)
            {
                scale8 = 1.0f;
            }

            foreach (StepValue step in stepValues)
            {
                // 16ビットフォーマット
                {
                    step.coefs16[(int)IfAnimCurve.StepCoefDef.Coef0] =
                        Convert.ToInt16((step.coefs32[(int)IfAnimCurve.StepCoefDef.Coef0] - quantizedOffset) / scale16);
                }

                // 8ビットフォーマット
                {
                    step.coefs8[(int)IfAnimCurve.StepCoefDef.Coef0] =
                        Convert.ToSByte((step.coefs32[(int)IfAnimCurve.StepCoefDef.Coef0] - quantizedOffset) / scale8);
                }
            }
        }

        public void QuantizeStepInt()
        {
            float middleValue = (maxValue + minValue) / 2.0f;
            if (middleValue > 0)
            {
                quantizedOffset = Convert.ToInt32(Math.Floor(middleValue));
            }
            else
            {
                quantizedOffset = Convert.ToInt32(Math.Ceiling(middleValue));
            }
        }

        /// <summary>
        /// 量子化結果を評価します。
        /// </summary>
        public void EvalQuantizeFrame(float tolerance)
        {
            curve_frame_typeType type = curve_frame_typeType.frame8;

            foreach (var frameValue in this.frameValues)
            {
                if (type >= curve_frame_typeType.frame8)
                {
                    float error = Math.Abs(frameValue.frame32 - IfAnimCurveUtility.DequantizeFrameFrom8(frameValue.frame8));

                    // 量子化誤差は絶対誤差を用いる。
                    if (error < tolerance)
                    {
                        type = curve_frame_typeType.frame8;
                        continue;
                    }
                }

                if (type >= curve_frame_typeType.frame16)
                {
                    float error = Math.Abs(frameValue.frame32 - IfAnimCurveUtility.DequantizeFrameFrom16(frameValue.frame16));

                    // 量子化誤差は絶対誤差を用いる。
                    if (error < tolerance)
                    {
                        type = curve_frame_typeType.frame16;
                        continue;
                    }
                }

                type = curve_frame_typeType.frame32;
                break;
            }

            int frameCount = this.frameValues.Count;
            float quantizedStartFrame = 0.0f;
            float quantizedEndFrame = 0.0f;
            if (type == curve_frame_typeType.frame8)
            {
                quantizedStartFrame =
                    IfAnimCurveUtility.DequantizeFrameFrom8(this.frameValues[0].frame8);
                quantizedEndFrame =
                    IfAnimCurveUtility.DequantizeFrameFrom8(this.frameValues[frameCount - 1].frame8);
            }
            else if (type == curve_frame_typeType.frame16)
            {
                quantizedStartFrame =
                    IfAnimCurveUtility.DequantizeFrameFrom16(this.frameValues[0].frame16);
                quantizedEndFrame =
                    IfAnimCurveUtility.DequantizeFrameFrom16(this.frameValues[frameCount - 1].frame16);
            }
            else
            {
                quantizedStartFrame = this.frameValues[0].frame32;
                quantizedEndFrame = this.frameValues[frameCount - 1].frame32;
            }

            if (quantizedEndFrame - quantizedStartFrame <= 0.0f)
            {
                IfStrings.Throw("IfAnimQuantizationAnalysis_Error_QuantizedFrameEqualZero");
            }

            this.frameType = type;
        }

        private float MaxAbsOfHermite(HermiteValue hermite)
        {
            // 端
            float ret = Math.Max(Math.Abs(hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0]),
                Math.Abs(hermite.Get(1.0f, curve_key_typeType.key32)));
            float defCoef3x3 = hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef3] * 3.0f;
            float d = -defCoef3x3 *
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef1] +
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2] *
                hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2];
            if (d >= 0)
            {
                // 極大極小
                float left = -hermite.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef2] / defCoef3x3;
                float right = (float)Math.Sqrt(d) / defCoef3x3;
                float t0 = left + right;
                float t1 = left - right;
                if (t0 > 0.0f && t0 < 1.0f)
                {
                    ret = Math.Max(ret, Math.Abs(hermite.Get(left + right, curve_key_typeType.key32)));
                }
                if (t1 > 0.0f && t1 < 1.0f)
                {
                    ret = Math.Max(ret, Math.Abs(hermite.Get(left - right, curve_key_typeType.key32)));
                }
            }
            return ret;
        }

        /// <summary>
        /// 量子化結果を評価します。
        /// </summary>
        public void EvalQuantizeFloat(float tolerance, bool useAbsEval, int framePrecision)
        {
            float invErrorBase = useAbsEval ? 1.0f : 1.0f / (maxValue - minValue);

            // それぞれの関数の単位で比較を行います。
            // フレーム量子後の関数を真の値として扱い、評価しています。

            // key16 の可能性チェック
            {
                bool ok = true;
                HermiteValue diff = new HermiteValue();
                for (int i = 0; ok && i < this.hermiteValues.Count; ++i)
                {
                    for (int coef = 0; coef < (int)IfAnimCurve.HermiteCoefDef.NumCoef; ++coef)
                    {
                        diff.coefs32[coef] = this.hermiteValues[i].coefs32[coef] -
                            this.hermiteValues[i].coefs16[coef] * this.scale16;
                    }
                    diff.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] -= this.quantizedOffset;
                    float error = MaxAbsOfHermite(diff);
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                for (int i = 0; ok && i < this.linearValues.Count; ++i)
                {
                    float diffCoef1 = this.linearValues[i].coefs32[(int)IfAnimCurve.LinearCoefDef.Coef1]
                        - this.linearValues[i].coefs16[(int)IfAnimCurve.LinearCoefDef.Coef1] * this.scale16;
                    float diffCoef0 = this.linearValues[i].coefs32[(int)IfAnimCurve.LinearCoefDef.Coef0]
                        - this.linearValues[i].coefs16[(int)IfAnimCurve.LinearCoefDef.Coef0] * this.scale16
                        - this.quantizedOffset;
                    float error = Math.Max(Math.Abs(diffCoef0), Math.Abs(diffCoef0 + diffCoef1));
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                for (int i = 0; ok && i < this.stepValues.Count; ++i)
                {
                    float error = Math.Abs(this.stepValues[i].coefs32[(int)IfAnimCurve.StepCoefDef.Coef0]
                        - this.stepValues[i].coefs16[(int)IfAnimCurve.StepCoefDef.Coef0] * this.scale16
                        - this.quantizedOffset);
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                if (!ok)
                {
                    this.keyType = curve_key_typeType.key32;
                    this.scale = 1.0f;
                    this.offset = 0.0f;
                    return;
                }
            }

            // key8 の可能性チェック
            {
                bool ok = true;
                HermiteValue diff = new HermiteValue();
                for (int i = 0; ok && i < this.hermiteValues.Count; ++i)
                {
                    for (int coef = 0; coef < (int)IfAnimCurve.HermiteCoefDef.NumCoef; ++coef)
                    {
                        diff.coefs32[coef] = this.hermiteValues[i].coefs32[coef] -
                            this.hermiteValues[i].coefs8[coef] * this.scale8;
                    }
                    diff.coefs32[(int)IfAnimCurve.HermiteCoefDef.Coef0] -= this.quantizedOffset;
                    float error = MaxAbsOfHermite(diff);
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                for (int i = 0; ok && i < this.linearValues.Count; ++i)
                {
                    float diffCoef1 = this.linearValues[i].coefs32[(int)IfAnimCurve.LinearCoefDef.Coef1]
                        - this.linearValues[i].coefs8[(int)IfAnimCurve.LinearCoefDef.Coef1] * this.scale8;
                    float diffCoef0 = this.linearValues[i].coefs32[(int)IfAnimCurve.LinearCoefDef.Coef0]
                        - this.linearValues[i].coefs8[(int)IfAnimCurve.LinearCoefDef.Coef0] * this.scale8
                        - this.quantizedOffset;
                    float error = Math.Max(Math.Abs(diffCoef0), Math.Abs(diffCoef0 + diffCoef1));
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                for (int i = 0; ok && i < this.stepValues.Count; ++i)
                {
                    float error = Math.Abs(this.stepValues[i].coefs32[(int)IfAnimCurve.StepCoefDef.Coef0]
                        - this.stepValues[i].coefs8[(int)IfAnimCurve.StepCoefDef.Coef0] * this.scale8
                        - this.quantizedOffset);
                    if (tolerance < error * invErrorBase)
                    {
                        ok = false;
                    }
                }

                if (!ok)
                {
                    this.keyType = curve_key_typeType.key16;
                    this.scale = this.scale16;
                    this.offset = this.quantizedOffset;
                    return;
                }
            }

            this.keyType = curve_key_typeType.key8;
            this.scale = this.scale8;
            this.offset = this.quantizedOffset;
        }

        public void EvalQuantizeInt()
        {
            if ((maxValue - quantizedOffset) <= sbyte.MaxValue &&
                (minValue - quantizedOffset) >= sbyte.MinValue)
            {
                this.keyType = curve_key_typeType.key8;
                this.offset = quantizedOffset;
            }
            else if ((maxValue - quantizedOffset) <= short.MaxValue &&
                     (minValue - quantizedOffset) >= short.MinValue)
            {
                this.keyType = curve_key_typeType.key16;
                this.offset = quantizedOffset;
            }
            else
            {
                this.keyType = curve_key_typeType.key32;
            }
        }

        public void BuildIfAnimCurve(object curve, bool isDegreeValue, List<G3dStream> streams)
        {
            if (curve is IG3dStreamReference)
            {
                var stream = streams[((IG3dStreamReference)curve).stream_index];
                BuildIfAnimCurve(curve, isDegreeValue, stream);
            }
            else
            {
                throw new NotImplementedException();
            }
        }

        public void BuildIfAnimCurve(object curve, bool isDegreeValue, G3dStream stream)
        {
            if (curve is hermite_curveType)
            {
                if (curveType != CurveType.Float)
                {
                    throw new InvalidOperationException("Invalid curve type.");
                }

                hermite_curveType hermiteCurve = curve as hermite_curveType;
                this.preWrap = hermiteCurve.pre_wrap;
                this.postWrap = hermiteCurve.post_wrap;
                this.frameType = hermiteCurve.frame_type;
                this.keyType = hermiteCurve.key_type;
                this.scale = hermiteCurve.scale;
                this.offset = hermiteCurve.offset;

                // 開始フレームを設定する。
                if (stream.FloatData.Count != 0)
                {
                    this.startFrame = stream.FloatData[0];
                }

                int keyIndex = 0;
                for (int index = 0; index < hermiteCurve.count; ++index)
                {
                    float[] values = new float[(int)IfAnimCurve.HermiteDef.NumValue];

                    int valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Frame0;
                    values[(int)IfAnimCurve.HermiteDef.Frame0] = stream.FloatData[valueIndex];
                    valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Value0;
                    values[(int)IfAnimCurve.HermiteDef.Value0] = stream.FloatData[valueIndex];
                    valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.InSlope0;
                    values[(int)IfAnimCurve.HermiteDef.InSlope0] = stream.FloatData[valueIndex];
                    valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.OutSlope0;
                    values[(int)IfAnimCurve.HermiteDef.OutSlope0] = stream.FloatData[valueIndex];

                    // 最終フレームはコピーで埋めておく
                    if (index == hermiteCurve.count - 1)
                    {
                        // 最終フレームを設定する。
                        this.endFrame = stream.FloatData[keyIndex + (int)IfAnimCurve.HermiteDef.Frame0];

                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Frame0;
                        values[(int)IfAnimCurve.HermiteDef.Frame1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Value0;
                        values[(int)IfAnimCurve.HermiteDef.Value1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.InSlope0;
                        values[(int)IfAnimCurve.HermiteDef.InSlope1] = stream.FloatData[valueIndex];
                    }
                    else
                    {
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Frame1;
                        values[(int)IfAnimCurve.HermiteDef.Frame1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Value1;
                        values[(int)IfAnimCurve.HermiteDef.Value1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.InSlope1;
                        values[(int)IfAnimCurve.HermiteDef.InSlope1] = stream.FloatData[valueIndex];
                    }

                    IfAnimCurve.HermiteValue hermite;
                    BuildHermite(values, isDegreeValue, out hermite);

                    this.hermiteValues.Add(hermite);

                    IfAnimCurve.FrameValue frameValue = new IfAnimCurve.FrameValue();
                    frameValue.frame32 = values[(int)IfAnimCurve.HermiteDef.Frame0];
                    frameValue.frame16 = IfAnimCurveUtility.QuantizeFrameTo16(frameValue.frame32);
                    frameValue.frame8 = IfAnimCurveUtility.QuantizeFrameTo8(frameValue.frame32);

                    this.frameValues.Add(frameValue);

                    float value = stream.FloatData[keyIndex + (int)IfAnimCurve.HermiteDef.Value0];
                    maxValue = Math.Max(maxValue, value);
                    minValue = Math.Min(minValue, value);

                    keyIndex += (int)CurveDef.HermiteElement;
                }
            }
            else if (curve is linear_curveType)
            {
                if (curveType != CurveType.Float)
                {
                    throw new InvalidOperationException("Invalid curve type.");
                }

                linear_curveType linearCurve = curve as linear_curveType;
                this.preWrap = linearCurve.pre_wrap;
                this.postWrap = linearCurve.post_wrap;
                this.frameType = linearCurve.frame_type;
                this.keyType = linearCurve.key_type;
                this.scale = linearCurve.scale;
                this.offset = linearCurve.offset;

                // 開始フレームを設定する。
                if (stream.FloatData.Count != 0)
                {
                    this.startFrame = stream.FloatData[0];
                }

                int keyIndex = 0;
                for (int index = 0; index < linearCurve.count; ++index)
                {
                    float[] values = new float[(int)IfAnimCurve.HermiteDef.NumValue];

                    int valueIndex = keyIndex + (int)IfAnimCurve.LinearDef.Frame0;
                    values[(int)IfAnimCurve.LinearDef.Frame0] = stream.FloatData[valueIndex];
                    valueIndex = keyIndex + (int)IfAnimCurve.LinearDef.Value0;
                    values[(int)IfAnimCurve.LinearDef.Value0] = stream.FloatData[valueIndex];

                    // 最終フレームはコピーで埋めておく
                    if (index == linearCurve.count - 1)
                    {
                        // 最終フレームを設定する。
                        this.endFrame = stream.FloatData[keyIndex + (int)IfAnimCurve.HermiteDef.Frame0];

                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Frame0;
                        values[(int)IfAnimCurve.LinearDef.Frame1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.HermiteDef.Value0;
                        values[(int)IfAnimCurve.LinearDef.Value1] = stream.FloatData[valueIndex];
                    }
                    else
                    {
                        valueIndex = keyIndex + (int)IfAnimCurve.LinearDef.Frame1;
                        values[(int)IfAnimCurve.LinearDef.Frame1] = stream.FloatData[valueIndex];
                        valueIndex = keyIndex + (int)IfAnimCurve.LinearDef.Value1;
                        values[(int)IfAnimCurve.LinearDef.Value1] = stream.FloatData[valueIndex];
                    }

                    IfAnimCurve.LinearValue linear;
                    BuildLinear(values, isDegreeValue, out linear);

                    this.linearValues.Add(linear);

                    IfAnimCurve.FrameValue frameValue = new IfAnimCurve.FrameValue();
                    frameValue.frame32 = values[(int)IfAnimCurve.LinearDef.Frame0];
                    frameValue.frame16 = IfAnimCurveUtility.QuantizeFrameTo16(frameValue.frame32);
                    frameValue.frame8 = IfAnimCurveUtility.QuantizeFrameTo8(frameValue.frame32);

                    this.frameValues.Add(frameValue);

                    float value = stream.FloatData[keyIndex + (int)IfAnimCurve.LinearDef.Value0];
                    maxValue = Math.Max(maxValue, value);
                    minValue = Math.Min(minValue, value);

                    keyIndex += (int)CurveDef.LinearElement;
                }
            }
            else if (curve is step_curveType)
            {
                step_curveType stepCurve = curve as step_curveType;
                this.preWrap = stepCurve.pre_wrap;
                this.postWrap = stepCurve.post_wrap;
                this.frameType = stepCurve.frame_type;
                this.keyType = stepCurve.key_type;
                this.scale = stepCurve.scale;
                this.offset = stepCurve.offset;

                // 開始フレームを設定する。
                if (stream.FloatData.Count != 0)
                {
                    this.startFrame = stream.FloatData[0];
                }
                else
                {
                    throw new NotImplementedException();
                }

                int keyIndex = 0;
                for (int index = 0; index < stepCurve.count; ++index)
                {
                    float[] values = new float[(int)IfAnimCurve.StepDef.NumValue];

                    int valueIndex = keyIndex + (int)IfAnimCurve.StepDef.Frame0;
                    values[(int)IfAnimCurve.StepDef.Frame0] = stream.FloatData[valueIndex];
                    valueIndex = keyIndex + (int)IfAnimCurve.StepDef.Value0;
                    values[(int)IfAnimCurve.StepDef.Value0] = stream.FloatData[valueIndex];

                    if (index == stepCurve.count - 1)
                    {
                        // 最終フレームを設定する。
                        this.endFrame = stream.FloatData[keyIndex + (int)IfAnimCurve.HermiteDef.Frame0];
                    }

                    if (curveType == CurveType.Float)
                    {
                        IfAnimCurve.StepValue step;
                        BuildStep(values, isDegreeValue, out step);

                        this.stepValues.Add(step);
                    }
                    else if (curveType == CurveType.Int)
                    {
                        IntValue step;
                        BuildStepInt(values, isDegreeValue, out step);

                        this.intValues.Add(step);
                    }
                    else
                    {
                        BoolValue step;
                        BuildStepBool(values, out step);

                        this.boolValues.Add(step);
                    }

                    IfAnimCurve.FrameValue frameValue = new IfAnimCurve.FrameValue();
                    frameValue.frame32 = values[(int)IfAnimCurve.StepDef.Frame0];
                    frameValue.frame16 = IfAnimCurveUtility.QuantizeFrameTo16(frameValue.frame32);
                    frameValue.frame8 = IfAnimCurveUtility.QuantizeFrameTo8(frameValue.frame32);

                    this.frameValues.Add(frameValue);

                    float value = stream.FloatData[keyIndex + (int)IfAnimCurve.StepDef.Value0];
                    maxValue = Math.Max(maxValue, value);
                    minValue = Math.Min(minValue, value);

                    keyIndex += (int)CurveDef.StepElement;
                }
            }
            else
            {
                throw new NotImplementedException();
            }

            this.numFrame = this.frameValues.Count;
        }

        public bool HasCurve()
        {
            return hermiteValues.Count != 0 ||
                   linearValues.Count != 0 ||
                   stepValues.Count != 0 ||
                   intValues.Count != 0 ||
                   boolValues.Count != 0;
        }

        public enum CurveDef
        {
            HermiteElement = 4,
            LinearElement = 2,
            StepElement = 2
        }

        public enum HermiteDef
        {
            Frame0 = 0,
            Value0,
            InSlope0,
            OutSlope0,
            Frame1,
            Value1,
            InSlope1,
            NumValue
        }

        public enum HermiteCoefDef
        {
            Coef0 = 0,
            Coef1,
            Coef2,
            Coef3,
            NumCoef
        }

        public enum LinearDef
        {
            Frame0 = 0,
            Value0,
            Frame1,
            Value1,
            NumValue
        }

        public enum LinearCoefDef
        {
            Coef0 = 0,
            Coef1,
            NumCoef
        }

        public enum StepDef
        {
            Frame0 = 0,
            Value0,
            NumValue
        }

        public enum StepCoefDef
        {
            Coef0 = 0,
            Coef1,
            NumCoef
        }

        public enum CurveType
        {
            Float,
            Int,
            Bool,
            NumType
        }

        public class HermiteValue
        {
            public float[] coefs32 { get; set; } = new float[(int)HermiteCoefDef.NumCoef];
            public short[] coefs16 { get; set; } = new short[(int)HermiteCoefDef.NumCoef];
            public sbyte[] coefs8 { get; set; } = new sbyte[(int)HermiteCoefDef.NumCoef];

            public float Get(float param, curve_key_typeType keyType)
            {
                // デコードには T 再構成を用いる。
                // 各種パラメーターを単純に float に変換することでマッピング区間の左端を返す。
                float t2 = param * param;
                if (keyType == curve_key_typeType.key8)
                {
                    return ((float)coefs8[3] * param + (float)coefs8[2]) * t2 + ((float)coefs8[1] * param + (float)coefs8[0]);
                }
                else if (keyType == curve_key_typeType.key16)
                {
                    return ((float)coefs16[3] * param + (float)coefs16[2]) * t2 + ((float)coefs16[1] * param + (float)coefs16[0]);
                }
                else
                {
                    return ((float)coefs32[3] * param + (float)coefs32[2]) * t2 + ((float)coefs32[1] * param + (float)coefs32[0]);
                }
            }
        }

        public class LinearValue
        {
            public float[] coefs32 { get; set; } = new float[(int)LinearCoefDef.NumCoef];
            public short[] coefs16 { get; set; } = new short[(int)LinearCoefDef.NumCoef];
            public sbyte[] coefs8 { get; set; } = new sbyte[(int)LinearCoefDef.NumCoef];

            public float Get(float param, curve_key_typeType keyType)
            {
                if (keyType == curve_key_typeType.key8)
                {
                    return coefs8[1] * param + coefs8[0];
                }
                else if (keyType == curve_key_typeType.key16)
                {
                    return coefs16[1] * param + coefs16[0];
                }
                else
                {
                    return coefs32[1] * param + coefs32[0];
                }
            }
        }

        public class StepValue
        {
            public float[] coefs32 { get; set; } = new float[(int)StepCoefDef.NumCoef];
            public short[] coefs16 { get; set; } = new short[(int)StepCoefDef.NumCoef];
            public sbyte[] coefs8 { get; set; } = new sbyte[(int)StepCoefDef.NumCoef];

            public float Get(curve_key_typeType keyType)
            {
                if (keyType == curve_key_typeType.key8)
                {
                    return coefs8[(int)StepCoefDef.Coef0];
                }
                else if (keyType == curve_key_typeType.key16)
                {
                    return coefs16[(int)StepCoefDef.Coef0];
                }
                else
                {
                    return coefs32[(int)StepCoefDef.Coef0];
                }
            }
        }

        public class IntValue
        {
            public int int32 { get; set; } = 0;

            public int Get(curve_key_typeType keyType)
            {
                if (keyType == curve_key_typeType.key8)
                {
                    return Convert.ToSByte(int32);
                }
                else if (keyType == curve_key_typeType.key16)
                {
                    return Convert.ToInt16(int32);
                }
                else
                {
                    return Convert.ToInt32(int32);
                }
            }
        }

        public class BoolValue
        {
            public bool value { get; set; } = false;

            public bool Get(curve_key_typeType keyType)
            {
                return value;
            }
        }

        public class FrameValue
        {
            public float frame32 { get; set; } = 0.0f;
            public short frame16 { get; set; } = 0;
            public byte frame8 { get; set; } = 0;
        }

        public IList<HermiteValue> hermiteValues { get; set; } = new List<HermiteValue>();
        public IList<LinearValue> linearValues { get; set; } = new List<LinearValue>();
        public IList<StepValue> stepValues { get; set; } = new List<StepValue>();
        public IList<FrameValue> frameValues { get; set; } = new List<FrameValue>();
        public IList<IntValue> intValues { get; set; } = new List<IntValue>();
        public IList<BoolValue> boolValues { get; set; } = new List<BoolValue>();

        public CurveType curveType { get; set; } = CurveType.Float;

        public float startFrame { get; set; } = 0.0f;
        public float endFrame { get; set; } = 0.0f;
        public float maxValue { get; set; } = float.MinValue;
        public float minValue { get; set; } = float.MaxValue;
        public float scale { get; set; } = 1.0f;
        public float offset { get; set; } = 0.0f;
        public float scale16 { get; set; } = 1.0f;
        public float scale8 { get; set; } = 1.0f;
        public float quantizedOffset { get; set; } = 0.0f;
        public curve_wrapType preWrap { get; set; } = curve_wrapType.clamp;
        public curve_wrapType postWrap { get; set; } = curve_wrapType.clamp;
        public curve_frame_typeType frameType { get; set; } = curve_frame_typeType.none;
        public curve_key_typeType keyType { get; set; } = curve_key_typeType.none;
        public int numFrame { get; set; } = 0;
    }
}
