﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シェーダーパラメーターアニメーションマージャ
    public static class IfShaderParamAnimMerger
    {
        // マージ用の情報
        public class IfShaderParamAnimMergerInfo
        {
            public IEnumerable<IfMergeSrcDstPair> MaterialNamePairTable{ get; set; }
        }

        // マージ
        public static void Merge(
            shader_param_animType newShaderParamAnim,
            List<G3dStream> newStreams,
            shader_param_animType oldShaderParamAnim,
            List<G3dStream> oldStreams)
        {
            Merge(
                newShaderParamAnim,
                newStreams,
                oldShaderParamAnim,
                oldStreams,
                null);
        }

        // オプション付きマージ
        public static void Merge(
            shader_param_animType newShaderParamAnim,
            List<G3dStream> newStreams,
            shader_param_animType oldShaderParamAnim,
            List<G3dStream> oldStreams,
            IfShaderParamAnimMergerInfo info)
        {
            // <user_data_array> <tool_data> <user_tool_data> <comment>
            IfMergeUtility.MergeRootObject(
                newShaderParamAnim, oldShaderParamAnim,
                newStreams, oldStreams);

            // quantize_tolerance_～ をマージする
            newShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_scale =
                oldShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_scale;
            newShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_rotate =
                oldShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_rotate;
            newShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_translate =
                oldShaderParamAnim.shader_param_anim_info.quantize_tolerance_tex_translate;

            // オリジナルマテリアルアニメーションが存在していなければマージ処理を行わない
            if (newShaderParamAnim.original_material_anim_array == null ||
                newShaderParamAnim.original_material_anim_array.original_material_anim == null ||
                oldShaderParamAnim.original_material_anim_array == null ||
                oldShaderParamAnim.original_material_anim_array.original_material_anim == null)
            {
                return;
            }

            // <shader_param_mat_anim_array> を newShaderParamAnim にコピーする。
            newShaderParamAnim.shader_param_mat_anim_array =
                oldShaderParamAnim.shader_param_mat_anim_array;
            // <original_material_anim> をキーに、<shader_param_mat_anim> を値に持つ辞書。
            // 同じ名前のマテリアル同士でアニメーションカーブをマージするために使用する。
            var matAnimDict =
                new Dictionary<original_material_animType, shader_param_mat_animType>();

            if (newShaderParamAnim.shader_param_mat_anim_array != null)
            {
                //  shader_param_mat_anim_array が使用する stream の移植。
                foreach (shader_param_mat_animType newMatAnim in
                    newShaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim)
                {
                    foreach (param_animType paramAnim in newMatAnim.param_anim_array.param_anim)
                    {
                        foreach (param_anim_targetType animTarget in paramAnim.param_anim_target)
                        {
                            IG3dCurve curve = animTarget.Curve;
                            if (curve == null)
                            {
                                continue;
                            }
                            G3dStream copyStream = oldStreams[curve.stream_index];
                            curve.stream_index = newStreams.Count;
                            newStreams.Add(copyStream);
                        }
                    }
                }

                // shader_param_mat_anim をキーにした辞書を作成する
                var shaderParamAnim =
                    newShaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim;
                var originalMatAnim =
                    newShaderParamAnim.original_material_anim_array.original_material_anim;
                if (info == null || info.MaterialNamePairTable == null)
                {
                    foreach (shader_param_mat_animType shaderAnim in shaderParamAnim)
                    {
                        var matAnim =
                            Array.Find(originalMatAnim,
                                mat => (mat.mat_name == shaderAnim.mat_name));
                        if (matAnim == null) { continue; }
                        matAnimDict.Add(matAnim, shaderAnim);
                    }
                }
                else
                {
                    // マテリアル名のペアテーブルが指定されていたら、その指定に従って辞書を作成する

                    // 名前のペアリストから辞書に変換する。
                    var pairTable = new Dictionary<string, string>();
                    IfMergeUtility.ConvertPairToDictionary(pairTable, info.MaterialNamePairTable);

                    foreach (shader_param_mat_animType shaderAnim in shaderParamAnim)
                    {
                        if (!pairTable.ContainsKey(shaderAnim.mat_name)) { continue; }

                        string matName = pairTable[shaderAnim.mat_name];
                        var matAnim =
                            Array.Find(originalMatAnim,
                                mat => (mat.mat_name == matName));
                        if (matAnim == null) { continue; }
                        matAnimDict.Add(matAnim, shaderAnim);
                    }
                }
            }

            // アニメーションカーブをマージする。
            // <shader_param_mat_anim> の mat_name と同じ名前の <original_material_anim> が持つカーブをマージする。
            // さらに <param_anim> の original_hint と同じ hint を持つ <original_color_anim>
            // 及び <original_texsrt_anim> が存在する場合のみマージが行われる。
            if (newShaderParamAnim.original_material_anim_array != null)
            {
                foreach (var originalMatAnim in newShaderParamAnim.original_material_anim_array.original_material_anim)
                {
                    // 同じ名前のマテリアルを取得する。
                    if (!matAnimDict.ContainsKey(originalMatAnim)) { continue; }

                    shader_param_mat_animType shaderAnim = matAnimDict[originalMatAnim];

                    // <param_anim> の original_hint と同じ <original_color_anim> hint があれば
                    // カーブを再設定する。
                    if (originalMatAnim.original_color_anim_array != null)
                    {
                        foreach (original_color_animType colorAnim in
                            originalMatAnim.original_color_anim_array.original_color_anim)
                        {
                            param_animType paramAnim = GetParamAnim(shaderAnim, colorAnim.hint);
                            if (paramAnim != null)
                            {
                                // カーブをマージする
                                MergeColorAnim(
                                    paramAnim,
                                    newStreams,
                                    colorAnim,
                                    oldStreams,
                                    newShaderParamAnim.shader_param_anim_info);
                            }
                        }
                    }

                    // <param_anim> の original_hint と同じ <original_texsrt_anim> hint があれば
                    // カーブを再設定する。
                    if (originalMatAnim.original_texsrt_anim_array != null)
                    {
                        foreach (original_texsrt_animType texsrtAnim in
                            originalMatAnim.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            param_animType paramAnim = GetParamAnim(shaderAnim, texsrtAnim.hint);
                            if (paramAnim != null)
                            {
                                // カーブをマージする
                                MergeTexsrtAnim(
                                    paramAnim,
                                    newStreams,
                                    texsrtAnim,
                                    oldStreams,
                                    newShaderParamAnim.shader_param_anim_info);
                            }
                        }
                    }
                }
            }

            // 使用されていないストリームを削除し、
            // 残ったストリームを中間ファイルの出現順にソートする。
            StreamUtility.SortStream(newShaderParamAnim, newStreams);
        }

        // 指定された original_hint を持つ param_anim を取得します。
        private static param_animType GetParamAnim(
            shader_param_mat_animType matAnim,
            string originalHint)
        {
            foreach (var paramAnim in matAnim.param_anim_array.param_anim)
            {
                if (paramAnim.original_hint == originalHint)
                {
                    return paramAnim;
                }
            }
            return null;
        }

        // カラーアニメーションをマージします。
        internal static void MergeColorAnim(
            param_animType destParamAnim,
            List<G3dStream> newStreams,
            original_color_animType srcColorAnim,
            List<G3dStream> oldStreams,
            shader_param_anim_infoType shaderAnimInfo)
        {
            var paramAnimTargets = new List<param_anim_targetType>();
            for (int i = 0; i < srcColorAnim.original_color_anim_target.Length; i++)
            {
                var colorAnimTarget =
                    srcColorAnim.original_color_anim_target[i];
                var paramAnimTarget =
                    Array.Find(destParamAnim.param_anim_target,
                        animTarget => animTarget.component_index == (int)colorAnimTarget.target);

                // original_color_anim_targetType がカーブを持つ場合は、
                // そのカーブが参照するストリームをコピーする。
                int streamIndex = -1;
                if (colorAnimTarget.Curve != null)
                {
                    streamIndex = newStreams.Count;
                    newStreams.Add(newStreams[colorAnimTarget.Curve.stream_index]);
                }

                // ターゲットが存在していなければ新規作成する。
                if (paramAnimTarget == null)
                {
                    paramAnimTarget = new param_anim_targetType();
                    paramAnimTarget.component_index = (int)colorAnimTarget.target;
                }

                // base_value は必ずコピーする
                paramAnimTarget.base_value = colorAnimTarget.base_value;

                if (colorAnimTarget.Curve != null)
                {
                    // カーブをコピーし、量子化分析する
                    var curve = IfMergeUtility.CopyCurve(colorAnimTarget.Curve);
                    IfAnimationAssignUtility.Quantize(curve, paramAnimTarget.component_index, destParamAnim.type, shaderAnimInfo, newStreams[streamIndex]);
                    // カーブ間でストリームを共有することはないので、
                    // コピーしたカーブに個別のストリーム番号を割り当てる。
                    curve.stream_index = streamIndex;
                    paramAnimTarget.Curve = curve;
                }
                else
                {
                    // マージファイル側のカラーアニメーションにカーブが存在していなくて
                    // DCC ファイルのパラメーターアニメーションにカーブが存在している場合は
                    // パラメーターアニメーションのカーブを削除する。
                    paramAnimTarget.Curve = null;
                }

                paramAnimTargets.Add(paramAnimTarget);
            }
            destParamAnim.param_anim_target = paramAnimTargets.ToArray();
        }

        // テクスチャ SRT アニメーションをマージします。
        internal static void MergeTexsrtAnim(
            param_animType destParamAnim,
            List<G3dStream> newStreams,
            original_texsrt_animType srcTexsrtAnim,
            List<G3dStream> oldStreams,
            shader_param_anim_infoType shaderAnimInfo)
        {
            var paramAnimTargets = new List<param_anim_targetType>();
            // モード値のパラメーターをマージする
            {
                var modeParamTarget =
                    Array.Find(destParamAnim.param_anim_target,
                        animTarget => animTarget.component_index == 0);
                if (modeParamTarget == null)
                {
                    modeParamTarget = new param_anim_targetType { component_index = 0 };
                }

                // base_value は必ずコピーする
                modeParamTarget.base_value = (float)srcTexsrtAnim.mode;
                // original_texsrt_animType からコピーする場合はカーブを削除する
                modeParamTarget.Curve = null;

                paramAnimTargets.Add(modeParamTarget);
            }

            // その他パラメーターをマージする
            int[] componentIndexTable = { 1, 2, 3, 4, 5 };
            for (int i = 0; i < srcTexsrtAnim.original_texsrt_anim_target.Length; i++)
            {
                original_texsrt_anim_targetType texsrtAnimTarget =
                    srcTexsrtAnim.original_texsrt_anim_target[i];
                param_anim_targetType paramAnimTarget =
                    Array.Find(destParamAnim.param_anim_target,
                        animTarget => animTarget.component_index == componentIndexTable[(int)texsrtAnimTarget.target]);

                // original_texsrt_anim_targetType がカーブを持つ場合は、
                // そのカーブが参照するストリームをコピーする。
                int streamIndex = -1;
                if (texsrtAnimTarget.Curve != null)
                {
                    streamIndex = newStreams.Count;
                    newStreams.Add(newStreams[texsrtAnimTarget.Curve.stream_index]);
                }

                // ターゲットが存在していなければ新規作成する。
                if (paramAnimTarget == null)
                {
                    paramAnimTarget = new param_anim_targetType { component_index = componentIndexTable[(int)texsrtAnimTarget.target] };
                    // component_index="0" はモード値であるため、original_texsrt_anim_target_targetType
                    // に 1 を足して component_index とする。
                }

                // base_value は必ずコピーする
                paramAnimTarget.base_value = texsrtAnimTarget.base_value;

                // カーブをコピーする
                if (texsrtAnimTarget.Curve != null)
                {
                    // カーブをコピーし、量子化分析する
                    var curve = IfMergeUtility.CopyCurve(texsrtAnimTarget.Curve);
                    IfAnimationAssignUtility.Quantize(curve, paramAnimTarget.component_index, destParamAnim.type, shaderAnimInfo, newStreams[streamIndex]);
                    // カーブ間でストリームを共有することはないので、
                    // コピーしたカーブに個別のストリーム番号を割り当てる。
                    curve.stream_index = streamIndex;
                    paramAnimTarget.Curve = curve;
                }
                else
                {
                    // マージファイル側のテクスチャ SRT アニメーションにカーブが存在していなくて
                    // DCC ファイルのパラメーターアニメーションにカーブが存在している場合は
                    // パラメーターアニメーションのカーブを削除する。
                    paramAnimTarget.Curve = null;
                }

                paramAnimTargets.Add(paramAnimTarget);
            }
            destParamAnim.param_anim_target = paramAnimTargets.ToArray();
        }
    }
}
