﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シーンアニメーションマージャ
    public static class IfSceneAnimMerger
    {
        // マージ
        public static void Merge(IfSceneAnimMergeContext context)
        {
            scene_animType newSceneAnim = context.NewSceneAnim;
            List<G3dStream> newStreams = context.NewStreams;
            scene_animType oldSceneAnim = context.OldSceneAnim;
            List<G3dStream> oldStreams = context.OldStreams;

            // <user_data_array> <tool_data> <user_tool_data> <comment>
            IfMergeUtility.MergeRootObject(
                newSceneAnim, oldSceneAnim,
                newStreams, oldStreams);

            // quantize_tolerance_～ をマージする
            newSceneAnim.scene_anim_info.quantize_tolerance_rotate =
                oldSceneAnim.scene_anim_info.quantize_tolerance_rotate;
            newSceneAnim.scene_anim_info.quantize_tolerance_translate =
                oldSceneAnim.scene_anim_info.quantize_tolerance_translate;

            // カメラアニメーションをマージする
            MergeCameraAnim(
                context.CameraAnimTable,
                newSceneAnim, newStreams,
                oldSceneAnim, oldStreams);

            // ライトアニメーションをマージする
            MergeLightAnim(
                context.LightAnimTable,
                newSceneAnim, newStreams,
                oldSceneAnim, oldStreams);

            // フォグアニメーションをマージする
            MergeFogAnim(
                context.FogAnimTable,
                newSceneAnim, newStreams,
                oldSceneAnim, oldStreams);

            // ストリームを中間ファイルの出現順にソートする
            StreamUtility.SortStream(newSceneAnim, newStreams);
        }

        // カメラアニメーションのマージ
        public static void MergeCameraAnim(
            Dictionary<camera_animType, camera_animType> cameraAnimTable,
            scene_animType newSceneAnim,
            List<G3dStream> newStreams,
            scene_animType oldSceneAnim,
            List<G3dStream> oldStreams)
        {
            if (oldSceneAnim.camera_anim_array == null ||
                oldSceneAnim.camera_anim_array.camera_anim == null)
            {
                return;
            }

            if (newSceneAnim.camera_anim_array == null)
            {
                newSceneAnim.camera_anim_array =
                    new camera_anim_arrayType();
                newSceneAnim.camera_anim_array.camera_anim =
                    new camera_animType[0];
            }

            List<camera_animType> newCameraAnim =
                new List<camera_animType>(newSceneAnim.camera_anim_array.camera_anim);

            // アニメーションをマージする
            foreach (camera_animType oldCameraAnim in
                oldSceneAnim.camera_anim_array.camera_anim)
            {
                if (!cameraAnimTable.ContainsKey(oldCameraAnim))
                {
                    // 引数ファイルに存在していないアニメーションのみマージファイルからコピーする
                    foreach (camera_anim_targetType cameraTarget in
                        oldCameraAnim.camera_anim_target ?? Enumerable.Empty<camera_anim_targetType>())
                    {
                        // カーブのストリームを追加しストリームインデックスを書き換える
                        if (cameraTarget.Curve == null) { continue; }

                        int streamIndex = cameraTarget.Curve.stream_index;
                        cameraTarget.Curve.stream_index = newStreams.Count;
                        newStreams.Add(oldStreams[streamIndex]);
                    }

                    // ユーザーデータはストリームも含めてコピーを行う
                    oldCameraAnim.user_data_array =
                        IfMergeUtility.CopyUserDataArray(oldCameraAnim.user_data_array,
                                                         newStreams,
                                                         oldStreams);

                    newCameraAnim.Add(oldCameraAnim);
                }
                else
                {
                    camera_animType newAnim = cameraAnimTable[oldCameraAnim];
                    IfMergeUtility.MergeComment(newAnim.comment, oldCameraAnim.comment);

                    user_data_arrayType newUserData = newAnim.user_data_array;
                    IfMergeUtility.MergeUserDataArray(ref newUserData,
                                                      oldCameraAnim.user_data_array,
                                                      newStreams, oldStreams);
                    newAnim.user_data_array = newUserData;
                }
            }

            // アニメーションを置きかえる
            newSceneAnim.camera_anim_array.length = newCameraAnim.Count;
            newSceneAnim.camera_anim_array.camera_anim = newCameraAnim.ToArray();
        }

        // ライトアニメーションのマージ
        public static void MergeLightAnim(
            Dictionary<light_animType, light_animType> lightAnimTable,
            scene_animType newSceneAnim,
            List<G3dStream> newStreams,
            scene_animType oldSceneAnim,
            List<G3dStream> oldStreams)
        {
            if (oldSceneAnim.light_anim_array == null ||
                oldSceneAnim.light_anim_array.light_anim == null)
            {
                return;
            }

            if (newSceneAnim.light_anim_array == null)
            {
                newSceneAnim.light_anim_array =
                    new light_anim_arrayType();
                newSceneAnim.light_anim_array.light_anim =
                    new light_animType[0];
            }

            List<light_animType> newLightAnim =
                new List<light_animType>(newSceneAnim.light_anim_array.light_anim);

            // アニメーションをマージする
            foreach (light_animType oldLightAnim in
                oldSceneAnim.light_anim_array.light_anim)
            {
                if (!lightAnimTable.ContainsKey(oldLightAnim))
                {
                    // 引数ファイルに存在していないアニメーションのみマージファイルからコピーする
                    foreach (light_anim_targetType lightTarget in
                        oldLightAnim.light_anim_target ?? Enumerable.Empty<light_anim_targetType>())
                    {
                        // カーブのストリームを追加しストリームインデックスを書き換える
                        if (lightTarget.Curve == null) { continue; }

                        int streamIndex = lightTarget.Curve.stream_index;
                        lightTarget.Curve.stream_index = newStreams.Count;
                        newStreams.Add(oldStreams[streamIndex]);
                    }

                    // ユーザーデータはストリームも含めてコピーを行う
                    oldLightAnim.user_data_array =
                        IfMergeUtility.CopyUserDataArray(oldLightAnim.user_data_array,
                                                         newStreams,
                                                         oldStreams);

                    newLightAnim.Add(oldLightAnim);
                }
                else
                {
                    light_animType newAnim = lightAnimTable[oldLightAnim];
                    IfMergeUtility.MergeComment(newAnim.comment, oldLightAnim.comment);

                    user_data_arrayType newUserData = newAnim.user_data_array;
                    IfMergeUtility.MergeUserDataArray(ref newUserData,
                                                      oldLightAnim.user_data_array,
                                                      newStreams, oldStreams);
                    newAnim.user_data_array = newUserData;
                }
            }

            // アニメーションを置きかえる
            newSceneAnim.light_anim_array.length = newLightAnim.Count;
            newSceneAnim.light_anim_array.light_anim = newLightAnim.ToArray();
        }

        // フォグアニメーションのマージ
        public static void MergeFogAnim(
            Dictionary<fog_animType, fog_animType> fogAnimTable,
            scene_animType newSceneAnim,
            List<G3dStream> newStreams,
            scene_animType oldSceneAnim,
            List<G3dStream> oldStreams)
        {
            if (oldSceneAnim.fog_anim_array == null ||
                oldSceneAnim.fog_anim_array.fog_anim == null)
            {
                return;
            }

            if (newSceneAnim.fog_anim_array == null)
            {
                newSceneAnim.fog_anim_array =
                    new fog_anim_arrayType();
                newSceneAnim.fog_anim_array.fog_anim =
                    new fog_animType[0];
            }

            List<fog_animType> newFogAnim =
                new List<fog_animType>(newSceneAnim.fog_anim_array.fog_anim);

            // アニメーションをマージする
            foreach (fog_animType oldFogAnim in
                oldSceneAnim.fog_anim_array.fog_anim)
            {
                if (!fogAnimTable.ContainsKey(oldFogAnim))
                {
                    // 引数ファイルに存在していないアニメーションのみマージファイルからコピーする
                    foreach (fog_anim_targetType fogTarget in
                        oldFogAnim.fog_anim_target ?? Enumerable.Empty<fog_anim_targetType>())
                    {
                        // カーブのストリームを追加しストリームインデックスを書き換える
                        if (fogTarget.Curve == null) { continue; }

                        int streamIndex = fogTarget.Curve.stream_index;
                        fogTarget.Curve.stream_index = newStreams.Count;
                        newStreams.Add(oldStreams[streamIndex]);
                    }

                    // ユーザーデータはストリームも含めてコピーを行う
                    oldFogAnim.user_data_array =
                        IfMergeUtility.CopyUserDataArray(oldFogAnim.user_data_array,
                                                         newStreams,
                                                         oldStreams);

                    newFogAnim.Add(oldFogAnim);
                }
                else
                {
                    fog_animType newAnim = fogAnimTable[oldFogAnim];
                    IfMergeUtility.MergeComment(newAnim.comment, oldFogAnim.comment);

                    user_data_arrayType newUserData = newAnim.user_data_array;
                    IfMergeUtility.MergeUserDataArray(ref newUserData,
                                                      oldFogAnim.user_data_array,
                                                      newStreams, oldStreams);
                    newAnim.user_data_array = newUserData;
                }
            }

            // アニメーションを置きかえる
            newSceneAnim.fog_anim_array.length = newFogAnim.Count;
            newSceneAnim.fog_anim_array.fog_anim = newFogAnim.ToArray();
        }
    }
}
