﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;

namespace nw.g3d.iflib
{
    // テキスト中間ファイルフォーマッタユーティリティ
    internal static class IfTextFormatterUtility
    {
        // 閉じ括弧の除去
        internal static string RemoveBracket(string line)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(line.EndsWith(">"));
            return line.Remove(line.Length - 1);
        }

        // スラッシュ閉じ括弧の除去
        internal static string RemoveSlashBracket(string line)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(line.EndsWith(" />"));
            return line.Remove(line.Length - 3);
        }

        //---------------------------------------------------------------------
        // process_log_array のフォーマット
        internal static void Format_process_log_array(
            TextReader rd, TextWriter wt, string baseIndent = "")
        {
            WriteLineWithIndent(wt, $"<process_log_array {rd.ReadLine()}", baseIndent);

            while (true)
            {
                string process_log = rd.ReadLine();
                if (process_log != "<process_log")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(process_log == "</process_log_array>");
                    WriteLineWithIndent(wt, $"{process_log}", baseIndent);
                    break;
                }
                WriteLineWithIndent(wt, $"\t{process_log} {rd.ReadLine()} {rd.ReadLine()}", baseIndent);
                string argument = rd.ReadLine();
                if (argument.EndsWith(" />"))
                {
                    WriteLineWithIndent(wt, $"\t\t{IfTextFormatterUtility.RemoveSlashBracket(argument)}", baseIndent);
                    WriteLineWithIndent(wt, "\t/>", baseIndent);
                }
                else
                {
                    WriteLineWithIndent(wt, $"\t\t{IfTextFormatterUtility.RemoveBracket(argument)}", baseIndent);
                    WriteLineWithIndent(wt, "\t>", baseIndent);
                    while (true)
                    {
                        string close_process_log = rd.ReadLine();
                        WriteLineWithIndent(wt, close_process_log, baseIndent);
                        if (close_process_log.EndsWith("</process_log>")) { break; }
                    }
                }
            }
        }

        private static void WriteLineWithIndent(TextWriter wt, string text, string indent)
        {
            wt.WriteLine($"{indent}{text}");
        }

        //---------------------------------------------------------------------
        // user_data_array のフォーマット
        internal static void Format_user_data_array(
            TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<user_data_array {1}", indent, rd.ReadLine());

            while (true)
            {
                // user_data
                string user_data = rd.ReadLine();
                if (user_data != "<user_data")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(user_data == "</user_data_array>");
                    wt.WriteLine("{0}{1}", indent, user_data);
                    break;
                }
                wt.WriteLine("{0}\t{1} {2} {3}",
                    indent, user_data, rd.ReadLine(), rd.ReadLine());

                string data = rd.ReadLine();
                switch (data)
                {
                    case "<user_int":
                        Format_user_int(rd, wt, indent);
                        break;
                    case "<user_float":
                        Format_user_float(rd, wt, indent);
                        break;
                    case "<user_string":
                        Format_user_string(rd, wt, indent, "string");
                        break;
                    case "<user_wstring":
                        Format_user_string(rd, wt, indent, "wstring");
                        break;
                    case "<user_stream":
                        wt.WriteLine("{0}\t\t<user_stream {1} {2}",
                            indent, rd.ReadLine(), rd.ReadLine());
                        break;
                    default:
                        Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected data {data}");
                        break;
                }

                string close_user_data = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_user_data == "</user_data>");
                wt.WriteLine("{0}\t{1}", indent, close_user_data);
            }
        }

        // user_int のフォーマット
        private static void Format_user_int(TextReader rd, TextWriter wt, string indent)
        {
            string count = rd.ReadLine();
            wt.WriteLine("{0}\t\t<user_int {1}", indent, count);
            if (count.EndsWith(" />") || count.EndsWith("</user_int>")) { return; }
            while (true)
            {
                string close_user_int = rd.ReadLine();
                wt.WriteLine(close_user_int);
                if (close_user_int.EndsWith("</user_int>")) { return; }
            }
        }

        // user_float のフォーマット
        private static void Format_user_float(TextReader rd, TextWriter wt, string indent)
        {
            string count = rd.ReadLine();
            wt.WriteLine("{0}\t\t<user_float {1}", indent, count);
            if (count.EndsWith(" />") || count.EndsWith("</user_float>")) { return; }
            while (true)
            {
                string close_user_float = rd.ReadLine();
                wt.WriteLine(close_user_float);
                if (close_user_float.EndsWith("</user_float>")) { return; }
            }
        }

        // user_string のフォーマット
        private static void Format_user_string(
            TextReader rd, TextWriter wt, string indent, string stringType)
        {
            string count = rd.ReadLine();
            wt.WriteLine("{0}\t\t<user_{1} {2}", indent, stringType, count);
            if (count.EndsWith(" />")) { return; }

            while (true)
            {
                string string_elem = rd.ReadLine();
                if (!string_elem.StartsWith("<" + stringType))
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(string_elem == "</user_" + stringType + ">");
                    wt.WriteLine("{0}\t\t{1}", indent, string_elem);
                    break;
                }
                wt.WriteLine("{0}\t\t\t{1}", indent, string_elem);
                if (string_elem.EndsWith(" />") ||
                    string_elem.EndsWith("</" + stringType + ">")) { continue; }
                while (true)
                {
                    string data = rd.ReadLine();
                    wt.WriteLine(data);
                    if (data.EndsWith(" />") ||
                        data.EndsWith("</" + stringType + ">")) { break; }
                }
            }
        }

        //---------------------------------------------------------------------
        // comment のフォーマット
        internal static void FormatComment(
            TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine($"{indent}<comment {rd.ReadLine()} {rd.ReadLine()} {rd.ReadLine()}");
        }

        //---------------------------------------------------------------------
        // tool_data のフォーマット
        internal static void FormatToolData(
            TextReader rd, TextWriter wt, string indent)
        {
            FormatData("tool_data", rd, wt, indent);
        }

        //---------------------------------------------------------------------
        // user_tool_data のフォーマット
        internal static void FormatUserToolData(
            TextReader rd, TextWriter wt, string indent)
        {
            FormatData("user_tool_data", rd, wt, indent);
        }

        private static void FormatData(string name, TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<{1}>", indent, name);

            int indentCount = 0;
            bool isText = false;
            while (true)
            {
                string line = rd.ReadLine();
                if (line == "</" + name + ">")
                {
                    wt.WriteLine("{0}{1}", indent, line);
                    break;
                }

                line = line.TrimStart(null);

                if (line.StartsWith("<"))
                {
                    if (!line.StartsWith("</"))
                    {
                        ++indentCount;
                    }

                    wt.Write("{0}", indent);
                    for (int i = 0; i < indentCount; ++i)
                    {
                        wt.Write("\t");
                    }

                    // 要素
                    isText = false;
                    if (line.EndsWith(">"))
                    {
                        wt.WriteLine("{0}", line);

                        if (line.Contains("</") || line.EndsWith("/>"))
                        {
                            --indentCount;
                        }
                        else
                        {
                            isText = true;
                        }
                    }
                    else
                    {
                        wt.Write("{0}", line);
                    }
                }
                else
                {
                    if (isText)
                    {
                        // TEXT
                        wt.Write("{0}", indent);
                        for (int i = 0; i < indentCount; ++i)
                        {
                            wt.Write("\t");
                        }

                        wt.WriteLine("\t{0}", line);
                    }
                    else
                    {
                        // 属性
                        if (line.EndsWith(">"))
                        {
                            wt.WriteLine(" {0}", line);

                            if (line.Contains("</") || line.EndsWith("/>"))
                            {
                                --indentCount;
                            }
                            else
                            {
                                isText = true;
                            }
                        }
                        else
                        {
                            wt.Write(" {0}", line);
                        }
                    }
                }
            }
        }

        //---------------------------------------------------------------------
        // アニメーションカーブのフォーマット
        internal static void FormatCurve(TextReader rd, TextWriter wt, string indent)
        {
            string curve = rd.ReadLine();
            switch (curve)
            {
                case "<hermite_curve":
                    Format_hermite_curve(rd, wt, indent);
                    break;
                case "<linear_curve":
                    Format_linear_curve(rd, wt, indent);
                    break;
                case "<step_curve":
                    Format_step_curve(rd, wt, indent);
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected curve {curve}");
                    break;
            }
        }

        // hermite_curve のフォーマット
        internal static void Format_hermite_curve(TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<hermite_curve {1} {2} {3}",
                indent, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1}", indent, rd.ReadLine());
            wt.WriteLine("{0}\t{1}",
                indent, IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("{0}/>", indent);
        }

        // linear_curve のフォーマット
        internal static void Format_linear_curve(TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<linear_curve {1} {2} {3}",
                indent, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1}", indent, rd.ReadLine());
            wt.WriteLine("{0}\t{1}",
                indent, IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("{0}/>", indent);
        }

        // step_curve のフォーマット
        internal static void Format_step_curve(TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<step_curve {1} {2} {3}",
                indent, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1} {2}", indent, rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("{0}\t{1}", indent, rd.ReadLine());
            wt.WriteLine("{0}\t{1}", indent,
                IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("{0}/>", indent);
        }

        //---------------------------------------------------------------------
        // macro_array のフォーマット
        internal static void Format_macro_array(TextReader rd, TextWriter wt, string indent)
        {
            wt.WriteLine("{0}<macro_array {1}", indent, rd.ReadLine());
            while (true)
            {
                // macro
                string macro = rd.ReadLine();
                if (macro != "<macro")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(macro == "</macro_array>");
                    wt.WriteLine("{0}{1}", indent, macro);
                    break;
                }
                wt.WriteLine("{0}\t{1} {2} {3} {4}",
                    indent, macro, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }

        //---------------------------------------------------------------------
        // stream のフォーマット
        internal static void Format_stream(TextReader rd, TextWriter wt, string indent="")
        {
            // stream_array
            WriteLineWithIndent(wt, $"<stream_array {rd.ReadLine()}", indent);

            while (true)
            {
                // stream
                string stream = rd.ReadLine();
                if (stream != "<stream")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(stream == "</stream_array>");
                    WriteLineWithIndent(wt, $"{stream}", indent);
                    break;
                }

                string stream_index = rd.ReadLine();
                string type = rd.ReadLine();
                string size = rd.ReadLine();
                string column = rd.ReadLine();
                WriteLineWithIndent(wt, $"\t{stream} {stream_index} {type} {size} {column}", indent);

                if (column.EndsWith(" />") || column.EndsWith("</stream>")) { continue; }

                while (true)
                {
                    string close_stream = rd.ReadLine();
                    WriteLineWithIndent(wt, close_stream, indent);
                    if (close_stream.EndsWith("</stream>")) { break; }
                }
            }
        }
    }
}
