﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using nw.g3d.nw4f_3dif;
using System.Xml.Schema;

namespace nw.g3d.iflib
{
    // テキスト中間ファイルフォーマッタ
    public static class IfTextFormatter
    {
        // テキスト中間ファイルをバイト列にフォーマットします
        // データ列をフォーマットする場合は、先に IfDataFormatter.Format() を呼びます
        public static byte[] FormatBytes(nw4f_3difType nw4f_3dif, string xsdBasePath)
        {
            return Encoding.UTF8.GetBytes(FormatString(nw4f_3dif, xsdBasePath).ToString());
        }

        /// <summary>
        /// テキスト中間ファイルを文字列にフォーマットします
        /// データ列をフォーマットする場合は、先に IfDataFormatter.Format() を呼びます
        /// </summary>
        /// <param name="nw4f_3dif">フォーマットするデータを指定します。</param>
        /// <param name="xsdBasePath">各バージョンの xsd が含まれるルートフォルダーを指定します。</param>
        /// <returns></returns>
        public static StringBuilder FormatString(nw4f_3difType nw4f_3dif, string xsdBasePath)
        {
            // 書き出し時に配列ヒント更新とデータフォーマットを強制する
            // データがフォーマットされていないと中間ファイルのフォーマットが失敗する
            nw4f_3dif.UpdateArrayHint();
            IfDataFormatter.Format(nw4f_3dif);

            // 加工しやすいようにアトリビュート単位で改行する
            XmlWriterSettings writerSettings = new XmlWriterSettings();
            writerSettings.Indent = true;
            writerSettings.IndentChars = string.Empty;
            writerSettings.NewLineOnAttributes = true;

            // 文字列へシリアライズする
            MemoryStream memStream = new MemoryStream();
            using (XmlWriter xmlWriter = XmlWriter.Create(memStream, writerSettings))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                serializer.Serialize(xmlWriter, nw4f_3dif);
            }
            memStream.Close();

            byte[] fileImage = memStream.ToArray();

            if (!string.IsNullOrEmpty(xsdBasePath))
            {
                ValidateXmlSchema(fileImage, xsdBasePath);
            }

            // 文字列をフォーマットする
            string source = Encoding.UTF8.GetString(fileImage);
            StringBuilder builder = new StringBuilder();
            using (StringReader rd = new StringReader(source))
            using (StringWriter wt = new StringWriter(builder))
            {
                Format(rd, wt);
            }
            return builder;
        }

        //---------------------------------------------------------------------

        private static void ValidateXmlSchema(byte[] fileImage, string xsdBasePath)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.True(!string.IsNullOrEmpty(xsdBasePath));
            string version = IfReadUtility.GetVersion(fileImage);
            XmlSchema schema = IfTextReadUtility.GetXsd(xsdBasePath, version);
            ValidateXmlSchema(fileImage, schema);
        }

        private static void ValidateXmlSchema(byte[] fileImage, XmlSchema schema)
        {
            XmlReaderSettings readerSettings = new XmlReaderSettings();
            readerSettings.CloseInput = true;
            readerSettings.ValidationType = ValidationType.Schema;
            readerSettings.Schemas.Add(schema);

            // チェック
            using (XmlReader ifReader = XmlReader.Create(
                new MemoryStream(fileImage), readerSettings))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                try
                {
                    nw4f_3difType file = (nw4f_3difType)serializer.Deserialize(ifReader);
                }
                catch (System.Exception exception)
                {
                    StringBuilder message = new StringBuilder();
                    message.AppendLine(exception.Message);
                    message.AppendLine(exception.InnerException?.Message??string.Empty);
                    string xmlText = System.Text.Encoding.Default.GetString(fileImage);
                    string[] lines = xmlText.Split(new string[] { System.Environment.NewLine }, System.StringSplitOptions.None);
                    int lineNumber = 1;
                    foreach (var line in lines)
                    {
                        message.AppendLine($"{lineNumber}: {line}");
                        ++lineNumber;
                    }

                    throw new System.Exception($"Invalid xml format. {message.ToString()}");
                }
            }
        }

        // フォーマット
        private static void Format(TextReader rd, TextWriter wt)
        {
            // XML 宣言
            string xmlDecl = rd.ReadLine();
            wt.WriteLine(xmlDecl);

            // nw4f_3dif
            string nw4f_3dif = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(nw4f_3dif.StartsWith("<nw4f_3dif"));
            wt.WriteLine("<nw4f_3dif {0}", rd.ReadLine());

            // file_info
            string file_info = rd.ReadLine();
            string file_type;
            if (file_info == "<file_info>")
            {
                FormatFileInfo(rd, wt);
                file_type = rd.ReadLine();
            }
            else
            {
                file_type = file_info;
            }

            file_type = file_type.Trim(new char[] { '<', '>' });

            // ファイル種別で振り分け
            switch (file_type)
            {
                case G3dConstant.ModelElementName:
                    IfTextModelFormatter.Format(rd, wt);
                    break;
                case G3dConstant.MaterialElementName:
                    IfTextModelFormatter.FormatMaterial(rd, wt);
                    break;
                case G3dConstant.TextureElementName:
                    IfTextTextureFormatter.Format(rd, wt);
                    break;
                case G3dConstant.SkeletalAnimElementName:
                    IfTextSkeletalAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.MaterialAnimElementName:
                    IfTextMaterialAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.ShaderParamAnimElementName:
                    IfTextShaderParamAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.TexPatternAnimElementName:
                    IfTextTexPatternAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.BoneVisibilityAnimElementName:
                    IfTextBoneVisibilityAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.MatVisibilityAnimElementName:
                    IfTextMatVisibilityAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.ShapeAnimElementName:
                    IfTextShapeAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.SceneAnimElementName:
                    IfTextSceneAnimFormatter.Format(rd, wt);
                    break;
                case G3dConstant.ShaderConfigElementName:
                    IfTextShaderConfigFormatter.Format(rd, wt);
                    break;
                case G3dConstant.ShaderDefinitionElementName:
                    IfTextShaderDefinitionFormatter.Format(rd, wt);
                    break;
                case G3dConstant.ShaderVariationElementName:
                    IfTextShaderVariationFormatter.Format(rd, wt);
                    break;
                case G3dConstant.CombinerShaderElementName:
                    IfTextCombinerShaderFormatter.Format(rd, wt);
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected file type {file_type}");
                    break;
            }

            string close_nw4f_3dif = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_nw4f_3dif == "</nw4f_3dif>", close_nw4f_3dif);
            wt.WriteLine(close_nw4f_3dif);
        }

        // ファイル情報のフォーマット
        private static void FormatFileInfo(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("<file_info>");

            string create = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(create == "<create");
            wt.WriteLine($"\t{create} {rd.ReadLine()} {rd.ReadLine()}");
            wt.WriteLine("\t\t{0}", IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("\t/>");

            string modify = rd.ReadLine();
            string close_file_info;
            if (modify == "<modify")
            {
                wt.WriteLine($"\t{modify} {rd.ReadLine()} {rd.ReadLine()}");
                close_file_info = rd.ReadLine();
            }
            else
            {
                close_file_info = modify;
            }
            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_file_info == "</file_info>");
            wt.WriteLine("{0}", close_file_info);
        }
    }
}
