﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Xml;
using System.Xml.Serialization;
namespace nw.g3d.iflib
{
    // バイナリ中間ファイルフォーマッタ
    public static class IfBinaryFormatter
    {
        #region G3dStream
        //---------------------------------------------------------------------
        // G3dStream
        // ストリーム付きバイナリ中間ファイルをフォーマットします
        public static byte[] FormatStream(
            nw4f_3difType nw4f_3dif, List<G3dStream> stream_array, string xsdBasePath)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!G3dStreamUtility.HasStreamArray(nw4f_3dif));

            // テキストフォーマットを施す
            byte[] textData = IfTextFormatter.FormatBytes(nw4f_3dif, xsdBasePath);

            // ストリームが存在しない場合は、テキストと同一
            if (stream_array.Count == 0) { return textData; }

            MemoryStream result = new MemoryStream();
            using (BinaryWriter resultWriter = new BinaryWriter(result))
            {
                // テキスト領域の書き込み
                resultWriter.Write(textData, 0, textData.Length);

                // null 終端とパディング
                resultWriter.Write((byte)0);
                WritePadding(resultWriter);

                // ストリームデータのフォーマット
                IfBinaryFormatter.FormatStream(resultWriter, stream_array);
            }
            result.Close();
            return result.ToArray();
        }

        //---------------------------------------------------------------------
        // ストリーム付きバイナリ中間ファイルをフォーマットします
        // Xml 部分のインデント調整を省略します。
        public static byte[] FormatStreamFast(
            nw4f_3difType nw4f_3dif, List<G3dStream> stream_array)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!G3dStreamUtility.HasStreamArray(nw4f_3dif));

            //// テキストフォーマットを施す
            //byte[] textData = IfTextFormatter.FormatBytes(nw4f_3dif);
            // 文字列へシリアライズする
            var writerSettings = new XmlWriterSettings()
            {
                Encoding = new UTF8Encoding(true),            // BOM付き
                Indent = false,
                CloseOutput = false,
            };
            MemoryStream memStream = new MemoryStream();
            using (XmlWriter xmlWriter = XmlWriter.Create(memStream, writerSettings))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                serializer.Serialize(xmlWriter, nw4f_3dif);
            }
            var textLength = memStream.Length;

            // ストリームが存在しない場合は、テキストと同一
            if (stream_array.Count == 0) { return memStream.ToArray(); }

            //MemoryStream result = new MemoryStream();
            using (BinaryWriter resultWriter = new BinaryWriter(memStream))
            {
                resultWriter.Seek(0, SeekOrigin.End);
                // テキスト領域の書き込み
                //resultWriter.Write(textData, 0, textData.Length);

                // null 終端とパディング
                resultWriter.Write((byte)0);
                WritePadding(resultWriter);

                // ストリームデータのフォーマット
                IfBinaryFormatter.FormatStream(resultWriter, stream_array);
            }
            memStream.Close();
            return memStream.ToArray();
        }

        // ストリームデータのフォーマット
        internal static void FormatStream(
            BinaryWriter resultWriter, List<G3dStream> stream_array)
        {
            // ストリームデータ列の準備
            int streamCount = stream_array.Count;
            MemoryStream[] streams = new MemoryStream[streamCount];
            int[] streamSizes = new int[streamCount];

            G3dParallel.For(0, streamCount, delegate(int i)
            {
                streams[i] = new MemoryStream();
                using (BinaryWriter wt = new BinaryWriter(streams[i]))
                {
                    stream_array[i].Write(wt);
                    WritePadding(wt);
                    streamSizes[i] = (int)streams[i].Position;
                }
            });

            // バイナリヘッダの書き込み
            resultWriter.Write(G3dConstant.StreamArrayChunkID);
            resultWriter.Write(streamCount);

            // バイナリチャンク先頭からのオフセット書き込み
            // バイナリヘッダ、ストリーム数、（オフセット＋サイズ）の配列
            int streamAddr = RoundUp(8 + 4 + (4 + 4) * streamCount);
            for (int i = 0; i < streamCount; i++)
            {
                resultWriter.Write(streamAddr);
                resultWriter.Write(streamSizes[i]);
                streamAddr += streamSizes[i];
            }
            WritePadding(resultWriter);

            // ストリームデータ列の統合
            for (int i = 0; i < streamCount; i++)
            {
                byte[] streamArray = streams[i].ToArray();
                resultWriter.Write(streamArray, 0, streamArray.Length);
            }
        }
        #endregion G3dStream

        #region IntermediateFileStream
        //---------------------------------------------------------------------
        // IntermediateFileStream
        // ストリーム付きバイナリ中間ファイルをフォーマットします
        public static byte[] FormatStream(
            nw4f_3difType nw4f_3dif, List<IntermediateFileStream> stream_array, string xsdBasePath)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!G3dStreamUtility.HasStreamArray(nw4f_3dif));

            // テキストフォーマットを施す
            byte[] textData = IfTextFormatter.FormatBytes(nw4f_3dif, xsdBasePath);

            // ストリームが存在しない場合は、テキストと同一
            if (stream_array.Count == 0) { return textData; }

            MemoryStream result = new MemoryStream();
            using (BinaryWriter resultWriter = new BinaryWriter(result))
            {
                // テキスト領域の書き込み
                resultWriter.Write(textData, 0, textData.Length);

                // null 終端とパディング
                resultWriter.Write((byte)0);
                WritePadding(resultWriter);

                // ストリームデータのフォーマット
                IfBinaryFormatter.FormatStream(resultWriter, stream_array);
            }
            result.Close();
            return result.ToArray();
        }

        //---------------------------------------------------------------------
        // ストリーム付きバイナリ中間ファイルをフォーマットします
        // Xml 部分のインデント調整を省略します。
        public static byte[] FormatStreamFast(
            nw4f_3difType nw4f_3dif, List<IntermediateFileStream> stream_array)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!G3dStreamUtility.HasStreamArray(nw4f_3dif));

            //// テキストフォーマットを施す
            //byte[] textData = IfTextFormatter.FormatBytes(nw4f_3dif);
            // 文字列へシリアライズする
            var writerSettings = new XmlWriterSettings()
            {
                Encoding = new UTF8Encoding(true),            // BOM付き
                Indent = false,
                CloseOutput = false,
            };
            MemoryStream memStream = new MemoryStream();
            using (XmlWriter xmlWriter = XmlWriter.Create(memStream, writerSettings))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                serializer.Serialize(xmlWriter, nw4f_3dif);
            }
            var textLength = memStream.Length;

            // ストリームが存在しない場合は、テキストと同一
            if (stream_array.Count == 0) { return memStream.ToArray(); }

            //MemoryStream result = new MemoryStream();
            using (BinaryWriter resultWriter = new BinaryWriter(memStream))
            {
                resultWriter.Seek(0, SeekOrigin.End);
                // テキスト領域の書き込み
                //resultWriter.Write(textData, 0, textData.Length);

                // null 終端とパディング
                resultWriter.Write((byte)0);
                WritePadding(resultWriter);

                // ストリームデータのフォーマット
                IfBinaryFormatter.FormatStream(resultWriter, stream_array);
            }
            memStream.Close();
            return memStream.ToArray();
        }

        // ストリームデータのフォーマット
        internal static void FormatStream(
            BinaryWriter resultWriter, List<IntermediateFileStream> stream_array)
        {
            // ストリームデータ列の準備
            int streamCount = stream_array.Count;
            //MemoryStream[] streams = new MemoryStream[streamCount];
            var streams = new byte[streamCount][];
            int[] streamSizes = new int[streamCount];

            G3dParallel.For(0, streamCount, delegate(int i)
            {
                streams[i] = stream_array[i].WriteBytes();
                streamSizes[i] = streams[i].Length;
            });

            // バイナリヘッダの書き込み
            resultWriter.Write(G3dConstant.StreamArrayChunkID);
            resultWriter.Write(streamCount);

            // バイナリチャンク先頭からのオフセット書き込み
            // バイナリヘッダ、ストリーム数、（オフセット＋サイズ）の配列
            int streamAddr = RoundUp(8 + 4 + (4 + 4) * streamCount);
            for (int i = 0; i < streamCount; i++)
            {
                resultWriter.Write(streamAddr);
                resultWriter.Write(streamSizes[i]);
                streamAddr += streamSizes[i];
            }
            WritePadding(resultWriter);

            // ストリームデータ列の統合
            for (int i = 0; i < streamCount; i++)
            {
                resultWriter.Write(streams[i], 0, streams[i].Length);
            }
        }
        #endregion IntermediateFileStream

        //---------------------------------------------------------------------
        // アライメント繰上げ
        private static int RoundUp(int position)
        {
            int aligment = G3dConstant.BinaryAlignment;
            return (position + (aligment - 1)) & ~(aligment - 1);
        }

        // パディング書き込み
        public static void WritePadding(BinaryWriter writer)
        {
            int position = (int)writer.Seek(0, SeekOrigin.Current);
            int count = RoundUp(position) - position;
            for (int i = 0; i < count; i++) { writer.Write((byte)0); }
        }
    }
}
