﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using IronPython.Hosting;
using IronPython.Runtime;
using Microsoft.Scripting.Hosting;
using Microsoft.Scripting;

namespace nw.g3d.iflib
{
    // フィルタエンジン
    public class IfFilterScript
    {
        // コンストラクタ
        public IfFilterScript(string scriptPath, string argument)
        {
            this.ScriptPath = scriptPath;
            Nintendo.Foundation.Contracts.Assertion.Operation.True((this.ScriptPath != null) && File.Exists(this.ScriptPath));
            Nintendo.Foundation.Contracts.Assertion.Operation.True(Path.GetExtension(this.ScriptPath) == ".py");

            this.Argument = argument;
            if (this.Argument == null) { this.Argument = string.Empty; }
        }

        //---------------------------------------------------------------------
        // 初期化
        public void Initialize()
        {
            if (this.IsInitialized) { return; }

            this.Engine = IfScriptingUtility.CreateEngine();
            this.Source = this.Engine.CreateScriptSourceFromFile(
                this.ScriptPath, Encoding.UTF8);
            this.Scope = this.Engine.CreateScope();
            try
            {
                this.Source.Execute(this.Scope);
            }
            catch (SyntaxErrorException exp)
            {
                throw new Exception(
                    string.Format("SyntaxError {0} l.{1}  {2}",
                        exp.SourcePath, exp.Line, exp.Message));
            }

            this.Scope.SetVariable("ScriptArgument", this.Argument);
            this.Scope.SetVariable("ScriptDictionary", this.Dictionary);
            this.Scope.SetVariable("Result", true);

            this.HasPreFilter = HasFunction(this.Scope, "PreFilter");
            this.HasFilter = HasFunction(this.Scope, "Filter");
            this.HasPostFilter = HasFunction(this.Scope, "PostFilter");

            this.IsInitialized = true;
        }

        // 関数が定義されているか
        private bool HasFunction(ScriptScope scope, string functionName)
        {
            object variable;
            scope.TryGetVariable(functionName, out variable);
            PythonFunction function = variable as PythonFunction;
            return (function != null);
        }

        //---------------------------------------------------------------------
        // プレフィルタ
        public bool PreFilter(List<IfFilterTarget> targets)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(this.IsInitialized);
            if (!this.HasPreFilter) { return true; }

            this.Scope.SetVariable("prefilter_targets", targets);
            try
            {
                this.Engine.Execute("PreFilter(prefilter_targets)", this.Scope);
            }
            catch (Exception e)
            {
                throw new Exception(this.Engine.GetService<ExceptionOperations>().FormatException(e));
            }
            foreach (IfFilterTarget target in targets) { target.Unload(); }

            return this.Scope.GetVariable<bool>("Result");
        }

        //---------------------------------------------------------------------
        // ポストフィルタ
        public bool PostFilter(List<IfFilterTarget> targets)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(this.IsInitialized);
            if (!this.HasPostFilter) { return true; }

            this.Scope.SetVariable("postfilter_targets", targets);
            try
            {
                this.Engine.Execute("PostFilter(postfilter_targets)", this.Scope);
            }
            catch (Exception e)
            {
                throw new Exception(this.Engine.GetService<ExceptionOperations>().FormatException(e));
            }
            foreach (IfFilterTarget target in targets) { target.Unload(); }

            return this.Scope.GetVariable<bool>("Result");
        }

        //---------------------------------------------------------------------
        public readonly string ScriptPath;
        public readonly string Argument;

        public readonly PythonDictionary Dictionary = new PythonDictionary();

        public bool HasPreFilter { get; private set; }
        public bool HasFilter { get; private set; }
        public bool HasPostFilter { get; private set; }

        private ScriptEngine Engine;
        private ScriptSource Source;
        private ScriptScope Scope;
        private bool IsInitialized;
    }
}
