﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // スケルタルアニメーションの合体ボーン圧縮
    public class IfSkeletalAnimBoneUniteCompressor : IfSkeletalAnimCompressor
    {
        // コンストラクタ
        public IfSkeletalAnimBoneUniteCompressor() :
            base("IfSkeletalAnimBoneUniteCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_unite"; }
        }

        // 圧縮
        protected override void Compress()
        {
            skeletal_animType skeletalAnim = this.Target;
            List<G3dStream> streams = this.Streams;

            bone_animType[] boneAnims = skeletalAnim.bone_anim_array.bone_anim;
            int boneAnimCount = boneAnims.Length;
            BoneAnimInfo[] boneAnimInfos = new BoneAnimInfo[boneAnimCount];

            // ボーンアニメーション情報の初期化
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i] = new BoneAnimInfo(boneAnims[i], skeletalAnim.skeletal_anim_info); }
            BoneAnimInfo.Setup(boneAnimInfos);

            // 削除可能なボーンの洗い出し
            // ルートノード(boneInfos[0])は削除しないので開始インデックスが 1 となる
            for (int i = 1; i < boneAnimCount; i++)
            {
                bone_animType boneAnim = boneAnims[i];
                BoneAnimInfo boneAnimInfo =    boneAnimInfos[i];
                if (boneAnim.compress_enable)
                {
                    // binarize_scale /    binarize_rotate / binarize_translate が false のボーンアニメーションは
                    // 圧縮できないのでエラーとする。
                    if (!GetBinarizeFlag(boneAnim))
                    {
                        IfStrings.Throw("SkeletalAnimCompress_Error_InvalidBinarizeFlag", boneAnim.name);
                    }

                    boneAnimInfo.IsReservedToRemove    = true;
                }
            }

            // アニメーションが付いているボーンが削除されるか確認する
            // アニメーションのマージが行われると元々アニメーションを持たないボーンに
            // カーブが設定されてしまうので正しく表示できなくなる
            // そのためこの確認はアニメーションのマージを行う前に実行する必要がある
            SkeletalAnimCompressUtility.CheckAnimationRemoved(boneAnimInfos);

            // <bone_anim> の rigid_body, matrix_index 属性を再設定する
            BoneAnimInfo rootBoneAnim = boneAnimInfos[0];
            foreach (BoneAnimInfo boneAnimInfo in boneAnimInfos)
            {
                bone_animType boneAnim = boneAnimInfo.BoneAnim;
                // スキニングで利用されるボーンが存在しているという事はリジッドスキンか
                // スムーススキンのメッシュが存在している。その場合はルートボーンに
                // メッシュが移動するので、ルートボーンの rigid_body 属性を true に設定する。
                if (boneAnim.matrix_index[0] != -1 || boneAnim.matrix_index[1] != -1)
                {
                    rootBoneAnim.BoneAnim.rigid_body = true;
                    boneAnim.matrix_index[0] = boneAnim.matrix_index[1] = -1;
                }
                // リジッドボディが参照するボーンが削除される場合、そのボーンより上位階層で
                // 削除されないボーンにメッシュが移動するので、そのボーンの rigid_body 属性
                // を true に設定する。
                if (boneAnimInfo.IsReservedToRemove && boneAnim.rigid_body)
                {
                    BoneAnimInfo uniteTo = boneAnimInfo.GetIncompressibleAncestorBone();
                    uniteTo.BoneAnim.rigid_body = true;
                }
            }
            rootBoneAnim.BoneAnim.matrix_index[0] =
            rootBoneAnim.BoneAnim.matrix_index[1] = -1;

            // 親が削除されたら、自身のSRT値を新しい親の座標系で定義しなおす。
            var animationItems = new List<SkeletalAnimCompressUtility.SetAnimationItem>();
            foreach (BoneAnimInfo boneAnimInfo in boneAnimInfos)
            {
                if (boneAnimInfo.IsReservedToRemove)
                {
                    continue;
                }

                BoneAnimInfo parent = boneAnimInfo.GetIncompressibleAncestorBone();
                if (boneAnimInfo.Parent != parent)
                {
                    animationItems.AddRange(
                        SkeletalAnimCompressUtility.MergeAnimation(
                            skeletalAnim,
                            streams,
                            parent,
                            boneAnimInfo));
                    // 親ノードとの合成後は正しく扱えないので。
                    boneAnimInfo.BoneAnim.scale_compensate = false;
                }
            }
            // SetAnimationは重いので、並列実行する
            G3dParallel.ForEach(animationItems, item =>
            {
                item.Stream = SkeletalAnimCompressUtility.SetAnimation(item.Target, item.FullValues, item.CurveTolerance, item.IsLoop, item.IsAngle);
            });

            foreach (var item in animationItems)
            {
                if (item.Stream == null)
                {
                    continue;
                }
                var streamIndex = streams.Count;
                streams.Add(item.Stream);
                Nintendo.Foundation.Contracts.Assertion.Operation.True(item.Target.Item is hermite_curveType);
                var curve = item.Target.Item as hermite_curveType;
                curve.stream_index = streamIndex;
            }

            // 新しいボーンのインデックスを設定する
            SkeletalAnimCompressUtility.SetNewIndex(rootBoneAnim);
            // アニメーションの削除
            SkeletalAnimCompressUtility.RemoveBoneAnims(this.Target, boneAnimInfos);
            // 不要になったストリームを削除する
            StreamUtility.SortStream(this.Target, this.Streams);
        }
    }
}
