﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // スケルタルアニメーションの刈り込みボーン圧縮
    public class IfSkeletalAnimBoneCullCompressor : IfSkeletalAnimCompressor
    {
        // コンストラクタ
        public IfSkeletalAnimBoneCullCompressor() :
            base("IfSkeletalAnimBoneCullCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_cull"; }
        }

        // 圧縮
        protected override void Compress()
        {
            CompressBoneCull(this.Target);

            // 不要になったストリームを削除する
            StreamUtility.SortStream(this.Target, this.Streams);
        }

        internal static void CompressBoneCull(skeletal_animType target)
        {
            bone_animType[] bone_anims = target.bone_anim_array.bone_anim;
            int boneAnimCount = bone_anims.Length;
            BoneAnimInfo[] boneAnimInfos = new BoneAnimInfo[boneAnimCount];
            BoneAnimInfo[] parent;

            // ボーンアニメーション情報の初期化
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i] = new BoneAnimInfo(bone_anims[i], target.skeletal_anim_info); }
            BoneAnimInfo.Setup(boneAnimInfos);

            // 削除可能なボーンの洗い出し
            bool update = true;
            while (update)
            {
                update = false;
                for (int i = 0; i < boneAnimCount; i++)
                {
                    bone_animType bone_anim = bone_anims[i];
                    BoneAnimInfo boneAnimInfo = boneAnimInfos[i];
                    if (boneAnimInfo.IsReservedToRemove) { continue; }
                    if (CheckRemovable(boneAnimInfo))
                    {
                        boneAnimInfo.IsReservedToRemove = true;
                        update = true;
                    }
                }
            }

            // 以下の条件に当てはまる場合はルートボーンを削除する。
            // - ボーン名が "nw4f_root" である
            // - ルートボーンの削除されない子供の数が 1 つだけ
            BoneAnimInfo rootAnimInfo = boneAnimInfos[0];
            SkeletalAnimCompressUtility.GetThinOutBoneRef(boneAnimInfos, out parent);
            BoneAnimInfo[] rootChild = Array.FindAll(
                parent, delegate (BoneAnimInfo bone) { return (bone == rootAnimInfo); });
            if (rootAnimInfo.BoneAnim.bone_name == G3dConstant.RootBoneName &&
                (rootChild.Length <= 1))
            {
                rootAnimInfo.IsReservedToRemove = true;
            }

            // ボーン圧縮後のルートボーンを見つける。
            if (rootAnimInfo.IsReservedToRemove)
            {
                int idx = Array.FindIndex(
                    parent, delegate (BoneAnimInfo bone) { return (bone == rootAnimInfo); });
                // 全てのノードが削除される場合は idx が -1 になる。
                // フォーマッタがボーンが存在していない場合に対応していないので、ルートノードのみ残す。
                if (idx >= 0)
                {
                    rootAnimInfo = boneAnimInfos[idx];
                }
                else
                {
                    rootAnimInfo.IsReservedToRemove = false;
                }
            }

            // アニメーションが付いているボーンが削除されるか確認する
            SkeletalAnimCompressUtility.CheckBinalizeAnimationRemoved(boneAnimInfos);
            // 新しいアニメーションのインデックスを設定する
            SkeletalAnimCompressUtility.SetNewIndex(rootAnimInfo);
            // アニメーションの削除
            SkeletalAnimCompressUtility.RemoveBoneAnims(target, boneAnimInfos);
        }

        // 削除可能かチェック
        private static bool CheckRemovable(BoneAnimInfo bone_animInfo)
        {
            bone_animType bone_anim = bone_animInfo.BoneAnim;

            // 圧縮禁止
            if (!bone_anim.compress_enable) { return false; }

            // 描画に利用されている
            if (bone_animInfo.RenderMatrix) { return false; }

            // <renderset> bone_anim_index は気にしなくてよい？

            // 子がいたら削除しない
            if (bone_animInfo.HasChild()) { return false; }

            return true;
        }
    }
}
