﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using nw.g3d.nw4f_3dif;
using System.Collections.Generic;

namespace nw.g3d.iflib
{
    public class IfRigidBodyToRigidSkinning : IfModelCompressor
    {
        public IfRigidBodyToRigidSkinning()
            : base("IfRigidBodyToRigidSkinning_Log")
        {
        }

        public override string Process
        {
            get
            {
                return "rigid_body_to_rigid_skinning";
            }
        }

        public override int GetCount()
        {
            if (this.Target.shape_array == null)
            {
                return 0;
            }
            return this.Target.shape_array.shape.Count(x => x.shape_info.vertex_skinning_count == 0);
        }

        public override string GetResult()
        {
            return string.Format("shape[{0}]", this.PreCount - this.PostCount);
        }

        public override string ToString()
        {
            return IfStrings.Get(
                this.ToStringId, this.Stopwatch.ElapsedMilliseconds,
                (this.PreCount > 0) ? (100f * this.PostCount / this.PreCount) : 0,
                this.PreCount - this.PostCount);
        }

        protected override void Compress()
        {
            ConvertRigidBodyToRigidSkinning(this.Target, this.Streams);
        }

        internal static void ConvertRigidBodyToRigidSkinning(modelType target, List<G3dStream> streams)
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            var model = target;
            if (model.shape_array != null)
            {
                foreach (shapeType shape in model.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfRigidBodyToRigidSkinning_Error_UnitedModelUnsupported");
                    }
                }
            }

            // vertex_array か shape_array が省略されているときは圧縮を行わない
            if (model.shape_array == null ||
                model.shape_array.shape == null) { return; }
            if (model.vertex_array == null ||
                model.vertex_array.vertex == null) { return; }
            // 初期化
            var shapes = model.shape_array.shape;
            var isSorted = IfModelSortUtility.IsShapeSorted(model);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelShapeCompressor_Error_NotSorted",
                    shapes[isSorted].name, shapes[isSorted + 1].name);
            }
            var shapeCount = shapes.Length;

            var bones = model.skeleton.bone_array.bone;
            var boneCount = bones.Length;
            var boneInfos = new BoneInfo[boneCount];
            for (var i = 0; i < boneCount; i++)
            {
                boneInfos[i] = new BoneInfo(bones[i], model.skeleton.skeleton_info);
            }
            BoneInfo.Setup(boneInfos);

            // 既に存在するリジッドスキニングに使われているボーンをカウント
            // 新規inv_model_matrixのインデクス用
            var mtxPaletteCount = 0;
            foreach (var bone in bones)
            {
                mtxPaletteCount = Math.Max(mtxPaletteCount, bone.matrix_index[0] + 1);
                mtxPaletteCount = Math.Max(mtxPaletteCount, bone.matrix_index[1] + 1);
            }

            var vertices = model.vertex_array.vertex;

            for (var i = 0; i < shapeCount; i++)
            {
                var shape = shapes[i];
                var shapeInfo = shape.shape_info;

                // リジッドボディの時のみ処理
                if (shapeInfo.vertex_skinning_count != 0)
                {
                    continue;
                }

                // シェイプが参照するボーンとルートボーン
                var rootbone = bones[0];
                var rootboneInfo = boneInfos[0];
                boneType shapebone = null;
                BoneInfo shapeboneInfo = null;

                for (var j = 0; j < boneCount; j++)
                {
                    if (bones[j].name == shapeInfo.bone_name)
                    {
                        shapebone = bones[j];
                        shapeboneInfo = boneInfos[j];
                        break;
                    }
                }

                if (shapebone == null)
                {
                    continue;
                }

                // ボーンがリジッドスキニング用に行列パレットを生成していない場合はここで追加する
                if (shapebone.matrix_index[1] == -1)
                {
                    shapebone.matrix_index[1] = mtxPaletteCount++;
                }

                var vertex = vertices[shapeInfo.vertex_index];
                var vtxAttribs = vertex.vtx_attrib_array.vtx_attrib.ToList();

                // 既にブレンドインデックスが存在する場合は何もしない（多分ありえない）
                if (vtxAttribs.Any(x => x.name == "_i0" || x.hint == "blendindex0"))
                {
                    continue;
                }
                // ブレンドインデックスの vtx_attrib を追加
                var attrib = new vtx_attribType
                {
                    name = "_i0",
                    hint = "blendindex0",
                    type = vtx_attrib_typeType.@uint,
                    attrib_index = vtxAttribs.Count,
                    index_hint = vtxAttribs.Count,
                    count = vtxAttribs[0].count,
                    quantize_type = vtx_attrib_quantize_typeType.none,
                    stream_index = streams.Count,
                };
                vtxAttribs.Add(attrib);

                // input buffer を追加
                var buffer = vertex.vtx_buffer_array.vtx_buffer[0];
                var inputs = buffer.input_array.input.ToList();
                var input = new inputType
                {
                    attrib_index = attrib.attrib_index,
                    index = inputs.Count,
                    index_hint = inputs.Count
                };
                inputs.Add(input);

                // stream を追加
                var data = new int[attrib.count];
                var mtxIndex = shapebone.matrix_index[1];
                for (var j = 0; j < attrib.count; j++)
                {
                    data[j] = mtxIndex;
                }
                var stream = new G3dStream { type = stream_typeType.@int };
                var streamType = new streamType
                {
                    count = attrib.count,
                    column = 1,
                    stream_index = streams.Count,
                    index_hint = streams.Count,
                    type = stream_typeType.@int
                };
                stream.SetText(streamType);
                stream.IntData.Clear();
                stream.IntData.AddRange(data);
                streams.Add(stream);

                // リジッドボディからリジッドスキニングにする
                shapeInfo.vertex_skinning_count = 1;
                shapebone.rigid_body = false;
                // <inv_model_matrix> を生成する
                var btm =
                    ModelCompressUtility.GetBoneTransform(model, shapeboneInfo);
                btm.Invert();
                var invstring = string.Empty;
                for (var row = 0; row < 3; row++)
                {
                    invstring += string.Format(
                        "{0}{1} {2} {3} {4}",
                        invstring.Any() ? " " : string.Empty,
                        btm[row, 0],
                        btm[row, 1],
                        btm[row, 2],
                        btm[row, 3]);
                }
                shapebone.inv_model_matrix = invstring;
                // ボーン圧縮が有効になっているときは、シェイプの bone_name を最初に見つかった圧縮不可のボーンの名前にする
                // (シェイプ圧縮されるようにしつつ、親ボーンを圧縮不可にした場合にボーンビジビリティアニメーションが効くようにする)
                if (shapebone.compress_enable)
                {
                    boneType parentBone = FindFirstCompressDisabledParentBone(bones, shape.shape_info.bone_name);
                    if (parentBone == null)
                    {
                        shapeInfo.bone_name = rootbone.name;
                    }
                    else
                    {
                        shapeInfo.bone_name = parentBone.name;
                    }
                }

                vertex.vtx_attrib_array.vtx_attrib = vtxAttribs.ToArray();
                buffer.input_array.input = inputs.ToArray();
            }

            // シェイプをソートする
            ModelCompressUtility.SortShape(model);
            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(model, streams);
        }

        private static boneType FindFirstCompressDisabledParentBone(boneType[] bones, string parentBoneName)
        {
            foreach (var bone in bones)
            {
                if (bone.name == parentBoneName)
                {
                    if (bone.compress_enable == false)
                    {
                        return bone;
                    }

                    return FindFirstCompressDisabledParentBone(bones, bone.parent_name);
                }
            }

            return null;
        }
    }
}
