﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Microsoft.Scripting.Utils;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルのシェイプ圧縮
    public class IfModelShapeCompressor : IfModelCompressor
    {
        private bool CompressShapeIgnoreSkinningCount;

        // コンストラクタ
        public IfModelShapeCompressor(bool compressShapeIgnoreSkinningCount) :
            base("IfModelShapeCompressor_Log", compressShapeIgnoreSkinningCount ? "--ignore-skinning-count" : string.Empty)
        {
            CompressShapeIgnoreSkinningCount = compressShapeIgnoreSkinningCount;
        }

        // プロセス
        public override string Process
        {
            get { return "compress_shape"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            if (this.Target.shape_array != null &&
                this.Target.shape_array.shape != null)
            {
                return this.Target.shape_array.shape.Length;
            }
            else
            {
                return 0;
            }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Format(
                "shape[{0}/{1}({2:f0}%)]",
                this.PostCount, this.PreCount, 100f * this.PostCount / this.PreCount);
        }

        // 圧縮
        protected override void Compress()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (this.Target.shape_array != null)
            {
                foreach (shapeType shape in this.Target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            // vertex_array か shape_array が省略されているときは圧縮を行わない
            if (this.Target.shape_array == null ||
                this.Target.shape_array.shape == null) { return; }
            if (this.Target.vertex_array == null ||
                this.Target.vertex_array.vertex == null) { return; }

            /// TODO: 制約条件チェック
            /// vertex と shape が同数で、マッチングしていること
            /// shape.shape_index == shape.shape_info.vertex_index
            /// もしくは制約で無くす

            // 初期化
            shapeType[] shapes = this.Target.shape_array.shape;
            int isSorted = IfModelSortUtility.IsShapeSorted(this.Target);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelShapeCompressor_Error_NotSorted",
                    shapes[isSorted].name, shapes[isSorted + 1].name);
            }
            int shapeCount = shapes.Length;
            CompShapeInfo[] compShapeInfos = new CompShapeInfo[shapeCount];
            int[] shapeCounts = new int[shapeCount];

            vertexType[] vertices = this.Target.vertex_array.vertex;
            for (int i = 0; i < shapeCount; i++)
            {
                vertexType vertex = vertices[shapes[i].shape_info.vertex_index];
                compShapeInfos[i] = new CompShapeInfo(shapes[i], i, vertex);
                shapeCounts[i] = shapes[i].shape_info.original_shape_count;
            }

            // 削除可能なシェイプの調査
            for (int i = 0; i < shapeCount; i++)
            {
                if (compShapeInfos[i].IsReservedToBeRemoved) { continue; }
                for (int j = i + 1; j < shapeCount; j++)
                {
                    if (compShapeInfos[j].IsReservedToBeRemoved) { continue; }
                    if (CanCompress(compShapeInfos[i], compShapeInfos[j]))
                    {
                        compShapeInfos[j].ReplaceIndex = i;
                        compShapeInfos[j].IsReservedToBeRemoved = true;
                        compShapeInfos[j].ReplaceVertexIndex = compShapeInfos[i].Vertex.vertex_index;

                        compShapeInfos[i].TotalIndexCount += compShapeInfos[j].TotalIndexCount;
                        compShapeInfos[j].TotalIndexCount = 0;

                        compShapeInfos[i].TotalVertexCount += compShapeInfos[j].TotalVertexCount;
                        compShapeInfos[j].TotalVertexCount = 0;

                        // 圧縮シェイプ数を加算
                        shapeCounts[i] += shapeCounts[j];
                    }
                }
            }

            // 新しいインデックスの割り振り
            int newIndex = 0;
            for (int i = 0; i < shapeCount; i++)
            {
                CompShapeInfo compShapeInfo = compShapeInfos[i];
                if (!compShapeInfo.IsReservedToBeRemoved)
                {
                    compShapeInfo.ReplaceIndex = newIndex;
                    compShapeInfo.ReplaceVertexIndex = newIndex;
                    newIndex++;
                }
                else
                {
                    compShapeInfo.MergeCompShapeInfo = compShapeInfos[compShapeInfo.ReplaceIndex];
                    compShapeInfo.ReplaceIndex =
                        compShapeInfos[compShapeInfo.ReplaceIndex].ReplaceIndex;
                    compShapeInfo.ReplaceVertexIndex =
                        compShapeInfos[compShapeInfo.ReplaceVertexIndex].ReplaceVertexIndex;
                }
            }

            // マージと新しい配列の構築
            shapeType[] newShapes = new shapeType[newIndex];
            vertexType[] newVertices = new vertexType[newIndex];
            for (int i = 0; i < shapeCount; i++)
            {
                CompShapeInfo compShapeInfo = compShapeInfos[i];
                if (compShapeInfo.IsReservedToBeRemoved)
                {
                    Merge(compShapeInfo.MergeCompShapeInfo, compShapeInfo);
                }
                else
                {
                    newShapes[compShapeInfo.ReplaceIndex] = compShapeInfo.Shape;
                    compShapeInfo.Shape.shape_info.vertex_index = compShapeInfo.ReplaceIndex;
                    newVertices[compShapeInfo.ReplaceVertexIndex] = compShapeInfo.Vertex;

                    // 削除前の shape 数を shape_info.original_shape_count に記録します。
                    compShapeInfo.Shape.shape_info.original_shape_count = shapeCounts[i];
                }
            }

            this.Target.shape_array.length = newShapes.Length;
            this.Target.shape_array.shape = newShapes;
            this.Target.vertex_array.length = newVertices.Length;
            this.Target.vertex_array.vertex = newVertices;

            // シェイプをソートする
            ModelCompressUtility.SortShape(this.Target);
            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(this.Target, this.Streams);

            if (this.CompressShapeIgnoreSkinningCount)
            {
                // マテリアル違いだが、元々連続していたシェイプのスキニング方式が異なってしまう場合があるので、
                // スキニング方式を揃える(連続するシェイプに生じる隙間をなくす)
                AlignSkinningTypeForContinuousShape(this.Target, this.Streams);

                // リジッドスキニングがスムーススキニングに変換されている場合があるので、
                // マトリックスインデックスを割り当て直す
                ReassignBoneMatrixIndex(this.Target, this.Streams);
            }
        }

        /// <summary>
        /// 連続するシェイプのスキニング方式を揃えます。
        /// </summary>
        internal static void AlignSkinningTypeForContinuousShape(modelType model, List<G3dStream> streams)
        {
            if (model.shape_array == null)
            {
                return;
            }

            for (int shapeIndex0 = 0; shapeIndex0 < model.shape_array.shape.Length; ++shapeIndex0)
            {
                for (int shapeIndex1 = shapeIndex0 + 1; shapeIndex1 < model.shape_array.shape.Length; ++shapeIndex1)
                {
                    shapeType shape0 = model.shape_array.shape[shapeIndex0];
                    shapeType shape1 = model.shape_array.shape[shapeIndex1];
                    bool isContinuousShape = !string.IsNullOrEmpty(shape0.shape_info.original_bone_name) &&
                        shape0.shape_info.original_bone_name == shape1.shape_info.original_bone_name;
                    if (!isContinuousShape)
                    {
                        // 連続していないシェイプなので修正の必要なし
                        continue;
                    }

                    if (shape0.shape_info.vertex_skinning_count == shape1.shape_info.vertex_skinning_count)
                    {
                        // スキニング方式が同じなので修正の必要なし
                        continue;
                    }

                    if (shape0.shape_info.vertex_skinning_count == 0 ||
                        shape1.shape_info.vertex_skinning_count == 0)
                    {
                        // 連続するシェイプがリジッドボディとスキニングシェイプで異なる事は通常有り得ないので
                        // とりあえず無視しておく
                        continue;
                    }

                    shapeType targetShape = shape0;
                    shapeType sourceShape = shape1;
                    if (shape0.shape_info.vertex_skinning_count > shape1.shape_info.vertex_skinning_count)
                    {
                        targetShape = shape1;
                        sourceShape = shape0;
                    }

                    AlignSkinningCount(targetShape, sourceShape, model, streams);
                }
            }
        }

        internal static void AlignSkinningCount(
            shapeType targetShape,
            shapeType sourceShape,
            modelType model,
            List<G3dStream> streams)
        {
            int dstSkinningCount = targetShape.shape_info.vertex_skinning_count;
            int srcSkinningCount = sourceShape.shape_info.vertex_skinning_count;
            var newSkinningCount = Math.Max(dstSkinningCount, srcSkinningCount);

            int oldSkinningCount = targetShape.shape_info.vertex_skinning_count;
            targetShape.shape_info.vertex_skinning_count = newSkinningCount;

            int skinningStreamsLastColumn;
            var skinningStreamsCount = Math.DivRem(newSkinningCount - 1, 4, out skinningStreamsLastColumn) + 1;
            skinningStreamsLastColumn += 1;

            vertexType targetVertex = model.vertex_array.vertex[targetShape.shape_info.vertex_index];
            vtx_attribType[] targetAttrs = model.vertex_array.vertex[targetShape.shape_info.vertex_index].vtx_attrib_array.vtx_attrib;
            vtx_attribType[] sourceAttrs = model.vertex_array.vertex[sourceShape.shape_info.vertex_index].vtx_attrib_array.vtx_attrib;
            List<vtx_attribType> newAttrs = targetAttrs.ToList();

            vertexType sourceVertex = model.vertex_array.vertex[sourceShape.shape_info.vertex_index];

            var blendIndexAttr = targetAttrs.FirstOrDefault(x => x.name == "_i0");
            Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(blendIndexAttr);
            var blendIndexStreamLength = blendIndexAttr.count;

            for (int skinningStreamIndex = 0; skinningStreamIndex < skinningStreamsCount; ++skinningStreamIndex)
            {
                var skinningIndexName = string.Format("_i{0}", skinningStreamIndex);
                var skinningWeightName = string.Format("_w{0}", skinningStreamIndex);
                var isLastStream = (skinningStreamIndex == (skinningStreamsCount - 1));
                var column = isLastStream ? skinningStreamsLastColumn : 4;

                vtx_attribType skinningIndexAttr = targetAttrs.FirstOrDefault(attr => attr.name == skinningIndexName);

                var dstIndexVtxAttr = targetAttrs.FirstOrDefault(x => x.name == skinningIndexName);
                var dstWeightVtxAttr = targetAttrs.FirstOrDefault(x => x.name == skinningWeightName);
                var srcIndexVtxAttr = sourceAttrs.FirstOrDefault(x => x.name == skinningIndexName);
                var srcWeightVtxAttr = sourceAttrs.FirstOrDefault(x => x.name == skinningWeightName);
                int bufferIndex;
                int inputIndex;
                if (dstIndexVtxAttr == null)
                {
                    GetBufferAndInputIndex(sourceVertex, srcIndexVtxAttr, out bufferIndex, out inputIndex);
                    AddBlendIndexVertexAndStream(targetVertex, streams, skinningStreamIndex, blendIndexStreamLength, column, bufferIndex, inputIndex);
                }
                else
                {
                    var stream = streams[dstIndexVtxAttr.stream_index];
                    if (stream.column < column)
                    {
                        IncreaseBlendIndexWeightStreamColumn(stream, dstIndexVtxAttr, stream.column, column);
                    }
                }

                if (dstWeightVtxAttr == null)
                {
                    GetBufferAndInputIndex(sourceVertex, srcWeightVtxAttr, out bufferIndex, out inputIndex);
                    AddWeightVertexAndStream(targetVertex, streams, skinningStreamIndex, blendIndexStreamLength, column, bufferIndex, inputIndex);
                }
                else
                {
                    var stream = streams[dstWeightVtxAttr.stream_index];
                    if (stream.column < column)
                    {
                        IncreaseBlendIndexWeightStreamColumn(stream, dstWeightVtxAttr, stream.column, column);
                    }
                }
            }

            // リジッドスキニングからスムーススキニングに変換する場合は位置座標や法線もモデル空間に座標変換
            if (newSkinningCount > 1 && dstSkinningCount == 1)
            {
                TransformVerticesToModelLocalFromBoneLocal(targetVertex.vtx_attrib_array, model, streams);
            }
        }

        // 圧縮可能か
        private bool CanCompress(CompShapeInfo lhs, CompShapeInfo rhs)
        {
            shape_infoType lhsInfo = lhs.Shape.shape_info;
            shape_infoType rhsInfo = rhs.Shape.shape_info;

            if (lhsInfo.mat_name != rhsInfo.mat_name) { return false; }
            if (lhsInfo.bone_name != rhsInfo.bone_name) { return false; }

            if (CompressShapeIgnoreSkinningCount)
            {
                // 片方がリジッドボディでもう片方がスキニングの場合は圧縮しない
                if ((lhsInfo.vertex_skinning_count == 0 && rhsInfo.vertex_skinning_count >= 1) ||
                    (lhsInfo.vertex_skinning_count >= 1 && rhsInfo.vertex_skinning_count == 0))
                {
                    return false;
                }
            }
            else
            {
                if (lhsInfo.vertex_skinning_count != rhsInfo.vertex_skinning_count)
                {
                    return false;
                }
            }

            // <key_shape_array> が存在している場合は圧縮しない
            shapeType lhsShape = lhs.Shape;
            shapeType rhsShape = rhs.Shape;
            bool hasKeyShape =
                ((lhsShape.key_shape_array != null && lhsShape.key_shape_array.length > 0) ||
                 (rhsShape.key_shape_array != null && rhsShape.key_shape_array.length > 0));
            if (hasKeyShape) { return false; }

            meshType lhsMesh = lhs.Shape.mesh_array.mesh[0];
            meshType rhsMesh = rhs.Shape.mesh_array.mesh[0];

            if (lhsMesh.mode != rhsMesh.mode) { return false; }
            // index 65534 積算
            // TODO: INT_MAX でチェックを入れる？
            //if ((lhs.TotalIndexCount + rhs.TotalIndexCount) > 65534) { return false; }

            return CanCompressVertex(lhs, rhs);
        }

        private static bool IsValidSkinningNameNum(vtx_attribType vtxAttrib)
        {
            int num;
            if (!int.TryParse(vtxAttrib.name.Substring(2), out num))
            {
                return false;
            }
            return num >= 0 && num <= 31;
        }

        private static bool IsSkinningIndex(vtx_attribType vtxAttrib)
        {
            return vtxAttrib.name.StartsWith("_i") && IsValidSkinningNameNum(vtxAttrib);
        }

        private static bool IsSkinningWeight(vtx_attribType vtxAttrib)
        {
            return vtxAttrib.name.StartsWith("_w") && IsValidSkinningNameNum(vtxAttrib);
        }

        private static bool IsSkinning(vtx_attribType vtxAttrib)
        {
            return IsSkinningIndex(vtxAttrib) || IsSkinningWeight(vtxAttrib);
        }

        // 頂点データが圧縮可能か
        private bool CanCompressVertex(CompShapeInfo lhs, CompShapeInfo rhs)
        {
            vtx_attribType[] lhsVtxAttrs;
            vtx_attribType[] rhsVtxAttrs;
            if (CompressShapeIgnoreSkinningCount)
            {
                lhsVtxAttrs = lhs.Vertex.vtx_attrib_array.vtx_attrib.Where(x => !IsSkinning(x)).ToArray();
                rhsVtxAttrs = rhs.Vertex.vtx_attrib_array.vtx_attrib.Where(x => !IsSkinning(x)).ToArray();
            }
            else
            {
                lhsVtxAttrs = lhs.Vertex.vtx_attrib_array.vtx_attrib;
                rhsVtxAttrs = rhs.Vertex.vtx_attrib_array.vtx_attrib;
            }
            if (lhsVtxAttrs.Length != rhsVtxAttrs.Length)
            {
                return false;
            }
            for (int i = 0; i < lhsVtxAttrs.Length; i++)
            {
                if (lhsVtxAttrs[i].name != rhsVtxAttrs[i].name)
                {
                    return false;
                }
                if (lhsVtxAttrs[i].hint != rhsVtxAttrs[i].hint)
                {
                    return false;
                }
                if (lhsVtxAttrs[i].type != rhsVtxAttrs[i].type)
                {
                    return false;
                }

                // vertex 65534 積算
                // TODO: INT_MAX でチェックを入れる？
                //if ((lhs.TotalVertexCount + rhs.TotalVertexCount) > 65534) { return false; }
            }

            // vtx_buffer_array が省略されている場合は圧縮しない
            if (lhs.Vertex.vtx_buffer_array == null ||
                lhs.Vertex.vtx_buffer_array.vtx_buffer == null)
            {
                return false;
            }
            if (rhs.Vertex.vtx_buffer_array == null ||
                rhs.Vertex.vtx_buffer_array.vtx_buffer == null)
            {
                return false;
            }

            vtx_bufferType[] lhsVtxBuffs = lhs.Vertex.vtx_buffer_array.vtx_buffer;
            vtx_bufferType[] rhsVtxBuffs = rhs.Vertex.vtx_buffer_array.vtx_buffer;
            if (lhsVtxBuffs.Length != rhsVtxBuffs.Length)
            {
                return false;
            }
            for (int i = 0; i < lhsVtxBuffs.Length; i++)
            {
                inputType[] lhsInputs;
                inputType[] rhsInputs;
                if (CompressShapeIgnoreSkinningCount)
                {
                    lhsInputs = lhsVtxBuffs[i].input_array.input.Where(x => lhsVtxAttrs.Any(y => y.attrib_index == x.attrib_index)).ToArray();
                    rhsInputs = rhsVtxBuffs[i].input_array.input.Where(x => rhsVtxAttrs.Any(y => y.attrib_index == x.attrib_index)).ToArray();
                }
                else
                {
                    lhsInputs = lhsVtxBuffs[i].input_array.input;
                    rhsInputs = rhsVtxBuffs[i].input_array.input;
                }
                if (lhsInputs.Length != rhsInputs.Length)
                {
                    return false;
                }
                for (int j = 0; j < lhsInputs.Length; j++)
                {
                    if (lhsInputs[j].attrib_index != rhsInputs[j].attrib_index)
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        private static void AddBlendIndexVertexAndStream(
            vertexType vertex, List<G3dStream> streams, int streamNum, int count, int column, int bufferIndex, int inputIndex)
        {
            int attribIndex;
            int streamIndex;
            var vtxAttrs = AddVertexAndInput(vertex, streams, bufferIndex, inputIndex, out attribIndex, out streamIndex);

            var vtxAttr = new vtx_attribType
            {
                name = string.Format("_i{0}", streamNum),
                hint = string.Format("blendindex{0}", streamNum),
                attrib_index = attribIndex,
                stream_index = streamIndex,
                count = count
            };
            switch (column)
            {
            case 1:
                vtxAttr.type = vtx_attrib_typeType.@uint;
                break;
            case 2:
                vtxAttr.type = vtx_attrib_typeType.uint2;
                break;
            case 3:
                vtxAttr.type = vtx_attrib_typeType.uint3;
                break;
            case 4:
                vtxAttr.type = vtx_attrib_typeType.uint4;
                break;
            default:
                Nintendo.Foundation.Contracts.Assertion.Operation.True(column >= 0 && column <= 4);
                break;
            }
            vtxAttrs.Add(vtxAttr);
            vertex.vtx_attrib_array.vtx_attrib = vtxAttrs.ToArray();
            vertex.vtx_attrib_array.UpdateHint();

            var stream = new G3dStream
            {
                column = column,
                type = stream_typeType.@int
            };
            var addBlendIndexData = new int[count * column].Select(x => InvalidBlendIndex);
            stream.IntData.AddRange(addBlendIndexData);
            streams.Add(stream);
        }

        private void AddIndexVertexAndStream(CompShapeInfo info, int streamNum, int count, int column, int bufferIndex, int inputIndex)
        {
            int attribIndex;
            int streamIndex;
            var vtxAttrs = AddVertexAndInput(info, bufferIndex, inputIndex, out attribIndex, out streamIndex);

            var vtxAttr = new vtx_attribType
            {
                name = string.Format("_i{0}", streamNum),
                hint = string.Format("blendindex{0}", streamNum),
                attrib_index = attribIndex,
                stream_index = streamIndex,
                count = count
            };
            switch (column)
            {
                case 1:
                    vtxAttr.type = vtx_attrib_typeType.@uint;
                    break;
                case 2:
                    vtxAttr.type = vtx_attrib_typeType.uint2;
                    break;
                case 3:
                    vtxAttr.type = vtx_attrib_typeType.uint3;
                    break;
                case 4:
                    vtxAttr.type = vtx_attrib_typeType.uint4;
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(column >= 0 && column <= 4);
                    break;
            }
            vtxAttrs.Add(vtxAttr);
            info.Vertex.vtx_attrib_array.vtx_attrib = vtxAttrs.ToArray();
            info.Vertex.vtx_attrib_array.UpdateHint();

            var stream = new G3dStream
            {
                column = column,
                type = stream_typeType.@int
            };
            var addBlendIndexData = new int[count * column].Select(x => InvalidBlendIndex);
            stream.IntData.AddRange(addBlendIndexData);
            Streams.Add(stream);
        }

        private static void AddWeightVertexAndStream(vertexType vertex, List<G3dStream> streams, int streamNum, int count, int column, int bufferIndex, int inputIndex)
        {
            int attribIndex;
            int streamIndex;
            var vtxAttrs = AddVertexAndInput(vertex, streams, bufferIndex, inputIndex, out attribIndex, out streamIndex);

            var vtxAttr = new vtx_attribType
            {
                name = string.Format("_w{0}", streamNum),
                hint = string.Format("blendweight{0}", streamNum),
                attrib_index = attribIndex,
                stream_index = streamIndex,
                count = count
            };
            switch (column)
            {
            case 1:
                vtxAttr.type = vtx_attrib_typeType.@float;
                break;
            case 2:
                vtxAttr.type = vtx_attrib_typeType.float2;
                break;
            case 3:
                vtxAttr.type = vtx_attrib_typeType.float3;
                break;
            case 4:
                vtxAttr.type = vtx_attrib_typeType.float4;
                break;
            default:
                Nintendo.Foundation.Contracts.Assertion.Operation.True(column >= 0 && column <= 4);
                break;
            }
            vtxAttrs.Add(vtxAttr);
            vertex.vtx_attrib_array.vtx_attrib = vtxAttrs.ToArray();
            vertex.vtx_attrib_array.UpdateHint();

            var stream = new G3dStream
            {
                column = column,
                type = stream_typeType.@float
            };
            // リジッドボーンに新しくウェイトを追加する場合は最初のウェイトに1を設定
            if (streamNum == 0)
            {
                var perVertex = new float[column];
                perVertex[0] = 1.0f;
                stream.FloatData.Capacity = count * column;
                for (var i = 0; i < count; i++)
                {
                    stream.FloatData.AddRange(perVertex);
                }
            }
            else
            {
                stream.FloatData.AddRange(new float[count * column]);
            }

            streams.Add(stream);
        }

        private void AddWeightVertexAndStream(CompShapeInfo info, int streamNum, int count, int column, int bufferIndex, int inputIndex)
        {
            AddWeightVertexAndStream(info.Vertex, this.Streams, streamNum, count, column, bufferIndex, inputIndex);
        }

        private static List<vtx_attribType> AddVertexAndInput(
            vertexType vertex, List<G3dStream> streams, int bufferIndex, int inputIndex, out int attribIndex,
            out int streamIndex)
        {
            var buffers = vertex.vtx_buffer_array.vtx_buffer.ToList();
            var buffer = buffers.FirstOrDefault(x => x.index == bufferIndex);
            if (buffer == null)
            {
                buffer = new vtx_bufferType
                {
                    index = bufferIndex,
                    input_array = new input_arrayType { input = new inputType[0] }
                };
                buffers.Add(buffer);
            }
            var vtxInputs = buffer.input_array.input.ToList();
            var vtxAttrs = vertex.vtx_attrib_array.vtx_attrib.ToList();
            attribIndex = vtxAttrs.Count;
            streamIndex = streams.Count;

            var vtxInput = new inputType { attrib_index = attribIndex };
            Ensure.Operation.True(inputIndex >= 0);
            if (inputIndex < vtxInputs.Count)
            {
                vtxInputs.Insert(inputIndex, vtxInput);
            }
            else
            {
                vtxInputs.Add(vtxInput);
            }

            buffer.input_array.input = vtxInputs.ToArray();
            buffer.input_array.UpdateHint();

            vertex.vtx_buffer_array.vtx_buffer = buffers.ToArray();
            vertex.vtx_buffer_array.UpdateHint();
            return vtxAttrs;
        }

        private List<vtx_attribType> AddVertexAndInput(CompShapeInfo info, int bufferIndex, int inputIndex, out int attribIndex,
            out int streamIndex)
        {
            return AddVertexAndInput(info.Vertex, this.Streams, bufferIndex, inputIndex, out attribIndex, out streamIndex);
        }

        // マージ
        private void Merge(CompShapeInfo dst, CompShapeInfo src)
        {
            // original_bone_name の結合
            {
                List<string> originalBoneNames = new List<string>();
                string[] destOriginalBoneNames = dst.Shape.shape_info.original_bone_name.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                originalBoneNames.AddRange(destOriginalBoneNames);
                string[] sourceOriginalBoneNames = src.Shape.shape_info.original_bone_name.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                originalBoneNames.AddRange(sourceOriginalBoneNames);
                string[] originalBoneNameArray = originalBoneNames.Distinct().ToArray();
                Array.Sort<string>(originalBoneNameArray);
                dst.Shape.shape_info.original_bone_name = string.Join(",", originalBoneNameArray);
            }

            // original_material_name の結合
            {
                List<string> originalMaterialNames = new List<string>();
                string[] destOriginalMaterialNames = dst.Shape.shape_info.original_material_name.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                originalMaterialNames.AddRange(destOriginalMaterialNames);
                string[] sourceOriginalMaterialNames = src.Shape.shape_info.original_material_name.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                originalMaterialNames.AddRange(sourceOriginalMaterialNames);
                string[] originalMaterialNameArray = originalMaterialNames.Distinct().ToArray();
                Array.Sort<string>(originalMaterialNameArray);
                dst.Shape.shape_info.original_material_name = string.Join(",", originalMaterialNameArray);
            }

            // メッシュのマージ
            meshType dstMesh = dst.Shape.mesh_array.mesh[0];
            meshType srcMesh = src.Shape.mesh_array.mesh[0];
            var dstSkinningCount = dst.Shape.shape_info.vertex_skinning_count;
            var srcSkinningCount = src.Shape.shape_info.vertex_skinning_count;

            var dstVtxAttrArray = dst.Vertex.vtx_attrib_array;
            var srcVtxAttrArray = src.Vertex.vtx_attrib_array;

            // スキニングカウントが一致しない場合はblendindex, blendweightのストリームを修正する
            // ストリームの数が足りない場合は追加する
            if (dstSkinningCount > 0 && srcSkinningCount > 0 && dstSkinningCount != srcSkinningCount)
            {
                var skinningCount = Math.Max(dstSkinningCount, srcSkinningCount);
                dst.Shape.shape_info.vertex_skinning_count = skinningCount;
                int skinningStreamsLastColumn;
                var skinningStreamsCount = Math.DivRem(skinningCount - 1, 4, out skinningStreamsLastColumn) + 1;
                skinningStreamsLastColumn += 1;

                // スキニングのインデクスは必ず存在するはず
                var dstBoneIndexAttrib = dstVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == "_i0");
                Nintendo.Foundation.Contracts.Assertion.Operation.True(dstBoneIndexAttrib != null);
                var dstBoneIndexCount = dstBoneIndexAttrib.count;
                var srcBoneIndexAttrib = srcVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == "_i0");
                Nintendo.Foundation.Contracts.Assertion.Operation.True(srcBoneIndexAttrib != null);
                var srcBoneIndexCount = srcBoneIndexAttrib.count;

                for (var i = 0; i < skinningStreamsCount; i++)
                {
                    var skinningIndexName = string.Format("_i{0}", i);
                    var skinningWeightName = string.Format("_w{0}", i);
                    var isLastStream = (i == (skinningStreamsCount - 1));
                    var column = isLastStream ? skinningStreamsLastColumn : 4;
                    var dstIndexVtxAttr = dstVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == skinningIndexName);
                    var dstWeightVtxAttr = dstVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == skinningWeightName);
                    var srcIndexVtxAttr = srcVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == skinningIndexName);
                    var srcWeightVtxAttr = srcVtxAttrArray.vtx_attrib.FirstOrDefault(x => x.name == skinningWeightName);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(srcIndexVtxAttr != null || dstIndexVtxAttr != null);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(srcWeightVtxAttr != null || dstWeightVtxAttr != null);
                    int bufferIndex;
                    int inputIndex;
                    if (dstIndexVtxAttr == null)
                    {
                        GetBufferAndInputIndex(src, srcIndexVtxAttr, out bufferIndex, out inputIndex);
                        AddIndexVertexAndStream(dst, i, dstBoneIndexCount, column, bufferIndex, inputIndex);
                    }
                    else
                    {
                        var stream = Streams[dstIndexVtxAttr.stream_index];
                        if (stream.column < column)
                        {
                            IncreaseBlendIndexWeightStreamColumn(stream, dstIndexVtxAttr, stream.column, column);
                        }
                    }

                    if (dstWeightVtxAttr == null)
                    {
                        GetBufferAndInputIndex(src, srcWeightVtxAttr, out bufferIndex, out inputIndex);
                        AddWeightVertexAndStream(dst, i, dstBoneIndexCount, column, bufferIndex, inputIndex);
                    }
                    else
                    {
                        var stream = Streams[dstWeightVtxAttr.stream_index];
                        if (stream.column < column)
                        {
                            IncreaseBlendIndexWeightStreamColumn(stream, dstWeightVtxAttr, stream.column, column);
                        }
                    }

                    if (srcIndexVtxAttr == null)
                    {
                        GetBufferAndInputIndex(dst, dstIndexVtxAttr, out bufferIndex, out inputIndex);
                        AddIndexVertexAndStream(src, i, srcBoneIndexCount, column, bufferIndex, inputIndex);
                    }
                    else
                    {
                        var stream = Streams[srcIndexVtxAttr.stream_index];
                        if (stream.column < column)
                        {
                            IncreaseBlendIndexWeightStreamColumn(stream, srcIndexVtxAttr, stream.column, column);
                        }
                    }
                    if (srcWeightVtxAttr == null)
                    {
                        GetBufferAndInputIndex(dst, dstWeightVtxAttr, out bufferIndex, out inputIndex);
                        AddWeightVertexAndStream(src, i, srcBoneIndexCount, column, bufferIndex, inputIndex);
                    }
                    else
                    {
                        var stream = Streams[srcWeightVtxAttr.stream_index];
                        if (stream.column < column)
                        {
                            IncreaseBlendIndexWeightStreamColumn(stream, srcWeightVtxAttr, stream.column, column);
                        }
                    }
                }
            }

            dstMesh.count += srcMesh.count;
            // インデックスのオフセット取得のため、頂点データのマージ前にシェイプをマージする
            int offset = dst.Vertex.vtx_attrib_array.vtx_attrib[0].count;
            G3dStream dstIdxStream = this.Streams[dstMesh.stream_index];
            Nintendo.Foundation.Contracts.Assertion.Operation.True(dstIdxStream.type == stream_typeType.@int);
            G3dStream srcIdxStream = this.Streams[srcMesh.stream_index];
            Nintendo.Foundation.Contracts.Assertion.Operation.True(srcIdxStream.type == stream_typeType.@int);
            foreach (int index in srcIdxStream.IntData)
            {
                dstIdxStream.IntData.Add(index + offset);
            }

            // サブメッシュのマージ
            if (dstMesh.submesh_array.submesh.Length > 1)
            {
                IfStrings.Throw("IfModelShapeCompressor_Error_MultiSubmesh", dst.Shape.name);
            }
            dstMesh.submesh_array.submesh[0].count += srcMesh.submesh_array.submesh[0].count;

            // リジッドスキニングからスムーススキニングに変換する場合は位置座標や法線もモデル空間に座標変換する必要がある
            {
                int skinningCount = Math.Max(dstSkinningCount, srcSkinningCount);
                if (skinningCount > 1)
                {
                    if (dstSkinningCount == 1)
                    {
                        TransformVerticesToModelLocalFromBoneLocal(dstVtxAttrArray);
                    }
                    else if (srcSkinningCount == 1)
                    {
                        TransformVerticesToModelLocalFromBoneLocal(srcVtxAttrArray);
                    }
                }
            }

            // 頂点データのマージ
            for (var i = 0; i < srcVtxAttrArray.vtx_attrib.Length; i++)
            {
                // ストリームのマージ
                var srcVtxAttrib = srcVtxAttrArray.vtx_attrib[i];
                var srcStreamIndex = srcVtxAttrib.stream_index;

                var dstVtxAttrib = dstVtxAttrArray.vtx_attrib.FirstOrDefault(x => IsSameVertexType(x, srcVtxAttrib));
                Nintendo.Foundation.Contracts.Assertion.Operation.True(dstVtxAttrib != null);
                var dstStreamIndex = dstVtxAttrib.stream_index;

                var dstVtxStream = this.Streams[dstStreamIndex];
                var srcVtxStream = this.Streams[srcStreamIndex];
                Nintendo.Foundation.Contracts.Assertion.Operation.True(dstVtxStream.type == srcVtxStream.type);

                // 頂点数の更新
                dstVtxAttrArray.vtx_attrib[i].count += srcVtxAttrib.count;
                // 圧縮されたシェイプの量子化精度はnoneにする
                dstVtxAttrArray.vtx_attrib[i].quantize_type = vtx_attrib_quantize_typeType.none;

                switch (dstVtxStream.type)
                {
                    case stream_typeType.@float:
                        dstVtxStream.FloatData.AddRange(srcVtxStream.FloatData);
                        break;
                    case stream_typeType.@int:
                        dstVtxStream.IntData.AddRange(srcVtxStream.IntData);
                        break;
                    default:
                        Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {dstVtxStream.type}");
                        break;
                }
            }
        }


        private static void TransformVerticesToModelLocalFromBoneLocal(
            vtx_attrib_arrayType vertexAttribArray, modelType target, List<G3dStream> streams)
        {
            // 存在するはずなので例外処理しない
            var blendIndexAttr = vertexAttribArray.vtx_attrib.First(x => x.name.StartsWith("_i"));
            var blendWeightAttr = vertexAttribArray.vtx_attrib.First(x => x.name.StartsWith("_w"));

            var posAttr = vertexAttribArray.vtx_attrib.FirstOrDefault(x => x.name.StartsWith("_p"));
            var normalAttr = vertexAttribArray.vtx_attrib.FirstOrDefault(x => x.name.StartsWith("_n"));
            var tangentAttr = vertexAttribArray.vtx_attrib.FirstOrDefault(x => x.name.StartsWith("_t"));
            var binormalAttr = vertexAttribArray.vtx_attrib.FirstOrDefault(x => x.name.StartsWith("_b"));

            var blendIndexStream = streams[blendIndexAttr.stream_index];
            var blendWeightStream = streams[blendWeightAttr.stream_index];
            int skinningCount = blendIndexStream.column;
            int vertexCount = blendIndexAttr.count;

            BoneInfo[] boneInfos = BoneInfo.CreateAndSetup(target);
            for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
            {
                for (int skinningIndex = 0; skinningIndex < skinningCount; ++skinningIndex)
                {
                    float weight = blendWeightStream.FloatData[vertexIndex * skinningCount + skinningIndex];
                    if (weight == 0.0f)
                    {
                        continue;
                    }

                    int matrixIndex = blendIndexStream.IntData[vertexIndex * skinningCount + skinningIndex];
                    var relatedBone = target.skeleton.bone_array.bone.FirstOrDefault(
                        bone => bone.matrix_index[1] == matrixIndex);
                    Nintendo.Foundation.Contracts.Ensure.Operation.NotNull(relatedBone);

                    var relatedBoneInfo = boneInfos.FirstOrDefault(x => x.Bone == relatedBone);
                    Nintendo.Foundation.Contracts.Ensure.Operation.NotNull(relatedBoneInfo);

                    Matrix44 transformMatrix = ModelCompressUtility.GetBoneTransform(target, relatedBoneInfo);

                    if (posAttr != null)
                    {
                        TransformVertex(streams[posAttr.stream_index], vertexIndex, transformMatrix, false);
                    }

                    if (normalAttr != null)
                    {
                        TransformVertex(
                            streams[normalAttr.stream_index], vertexIndex,
                            ModelCompressUtility.GetNormalTransform(transformMatrix), true);
                    }

                    if (tangentAttr != null)
                    {
                        TransformVertex(
                            streams[tangentAttr.stream_index], vertexIndex,
                            ModelCompressUtility.GetTangentBinormalTransform(transformMatrix), true);
                    }

                    if (binormalAttr != null)
                    {
                        TransformVertex(
                            streams[binormalAttr.stream_index], vertexIndex,
                            ModelCompressUtility.GetTangentBinormalTransform(transformMatrix), true);
                    }

                    // 元々リジッドスキニングであればウェイトのあるコンポーネントは１つだけなのでループを抜ける
                    break;
                }
            }
        }

        private void TransformVerticesToModelLocalFromBoneLocal(vtx_attrib_arrayType vertexAttribArray)
        {
            TransformVerticesToModelLocalFromBoneLocal(vertexAttribArray, this.Target, this.Streams);
        }

        private static void TransformVertex(
            G3dStream transformStream, int vertexIndex, Matrix44 transformMatrix, bool normalize)
        {
            Nintendo.Foundation.Contracts.Ensure.Operation.NotNull(transformStream.FloatData);
            Nintendo.Foundation.Contracts.Ensure.Operation.True(transformStream.column >= 3);

            int index = vertexIndex * transformStream.column;
            Vector3 vec = new Vector3(
                transformStream.FloatData[index],
                transformStream.FloatData[index + 1],
                transformStream.FloatData[index + 2]);

            Vector3 transformedVec = transformMatrix * vec;

            if (normalize)
            {
                transformedVec.Normalize();
            }

            transformStream.FloatData[index] = transformedVec.X;
            transformStream.FloatData[index + 1] = transformedVec.Y;
            transformStream.FloatData[index + 2] = transformedVec.Z;
        }

        private static void GetBufferAndInputIndex(
            vertexType src, vtx_attribType srcIndexVtxAttr, out int bufferIndex,
            out int inputIndex)
        {
            var buffer =
                src.vtx_buffer_array.vtx_buffer.FirstOrDefault(
                    x =>
                        x.input_array.input.Any(input => input.attrib_index == srcIndexVtxAttr.attrib_index));
            Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(buffer);
            bufferIndex = buffer.index;
            inputIndex = buffer.input_array.input.FindIndex(x => x.attrib_index == srcIndexVtxAttr.attrib_index);
        }

        private static void GetBufferAndInputIndex(
            CompShapeInfo src, vtx_attribType srcIndexVtxAttr, out int bufferIndex,
            out int inputIndex)
        {
            GetBufferAndInputIndex(src.Vertex, srcIndexVtxAttr, out bufferIndex, out inputIndex);
        }

        private bool IsSameVertexType(vtx_attribType lhs, vtx_attribType rhs)
        {
            return lhs.name == rhs.name && lhs.hint == rhs.hint && lhs.type == rhs.type;
        }

        private static void IncreaseBlendIndexWeightStreamColumn(G3dStream stream, vtx_attribType vtxAttrib, int srcColumn, int dstColumn)
        {
            switch (stream.type)
            {
                case stream_typeType.@float:
                    IncreaseBlendWeightStreamDataColumn(stream.FloatData, vtxAttrib, srcColumn, dstColumn);
                    break;
                case stream_typeType.@int:
                    IncreaseBlendIndexStreamDataColumn(stream.IntData, vtxAttrib, srcColumn, dstColumn);
                    break;
            }

            stream.column = dstColumn;
        }

        private const int InvalidBlendIndex = -1;

        private static void IncreaseBlendIndexStreamDataColumn(List<int> oldData, vtx_attribType vtxAttrib, int srcColumn, int dstColumn)
        {
            var valueCount = vtxAttrib.count;
            var newData = new int[valueCount * dstColumn];
            var oldIndex = 0;
            var newIndex = 0;
            for (var j = 0; j < valueCount; j++)
            {
                for (var k = 0; k < srcColumn; k++)
                {
                    newData[newIndex] = oldData[oldIndex];
                    oldIndex++;
                    newIndex++;
                }
                for (var k = srcColumn; k < dstColumn; k++)
                {
                    newData[newIndex] = InvalidBlendIndex;
                    newIndex++;
                }
            }

            switch (dstColumn)
            {
                case 1:
                    vtxAttrib.type = vtx_attrib_typeType.@uint;
                    break;
                case 2:
                    vtxAttrib.type = vtx_attrib_typeType.uint2;
                    break;
                case 3:
                    vtxAttrib.type = vtx_attrib_typeType.uint3;
                    break;
                case 4:
                    vtxAttrib.type = vtx_attrib_typeType.uint4;
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(dstColumn <= 4);
                    break;
            }
            oldData.Clear();
            oldData.AddRange(newData);
        }

        private static void IncreaseBlendWeightStreamDataColumn(List<float> oldData, vtx_attribType vtxAttrib, int srcColumn, int dstColumn)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(dstColumn > srcColumn);
            var valueCount = vtxAttrib.count;
            var newData = new float[valueCount * dstColumn];
            var oldIndex = 0;
            var newIndex = 0;
            for (var j = 0; j < valueCount; j++)
            {
                for (var k = 0; k < srcColumn; k++)
                {
                    newData[newIndex] = oldData[oldIndex];
                    oldIndex++;
                    newIndex++;
                }
                for (var k = srcColumn; k < dstColumn; k++)
                {
                    newData[newIndex] = 0.0f;
                    newIndex++;
                }
            }

            switch (dstColumn)
            {
                case 1:
                    vtxAttrib.type = vtx_attrib_typeType.@float;
                    break;
                case 2:
                    vtxAttrib.type = vtx_attrib_typeType.float2;
                    break;
                case 3:
                    vtxAttrib.type = vtx_attrib_typeType.float3;
                    break;
                case 4:
                    vtxAttrib.type = vtx_attrib_typeType.float4;
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(dstColumn <= 4);
                    break;
            }

            oldData.Clear();
            oldData.AddRange(newData);
        }

        // シェイプ情報
        private class CompShapeInfo : ShapeInfo
        {
            public CompShapeInfo(shapeType shape, int index,
                vertexType vertex) : base(shape)
            {
                //this.Shape = shape;
                this.Index = index;
                this.ReplaceIndex = index;
                this.Vertex = vertex;
                this.ReplaceVertexIndex = vertex.vertex_index;
                this.TotalIndexCount = shape.mesh_array.mesh[0].count;
                int vertexCount = vertex.vtx_attrib_array.vtx_attrib[0].count;
                this.TotalVertexCount = vertexCount;
            }

            public override string ToString()
            {
                string basicInfo = string.Format("{0} {1,2}→{2,2}",
                    !this.IsReservedToBeRemoved ? "○" : "×",
                    this.Index,
                    this.ReplaceIndex);
                string vertexInfo = string.Format(
                    "Vtx {0,2}→{1,2} Idx[{2,5}→{3,5}] Vtx[{4,5}→{5,5}]",
                    this.Vertex.vertex_index,
                    this.ReplaceVertexIndex,
                    this.Shape.mesh_array.mesh[0].count,
                    this.TotalIndexCount,
                    this.Vertex.vtx_attrib_array.vtx_attrib[0].count,
                    this.TotalVertexCount);

                return string.Format(
                    "{0} {1}",
                    basicInfo,
                    vertexInfo);
            }

            //public readonly shapeType Shape;
            public readonly int Index;
            public int ReplaceIndex { get; set; }
            public CompShapeInfo MergeCompShapeInfo { get; set; }
            public readonly vertexType Vertex;
            public bool IsReservedToBeRemoved { get; set; }
            public int ReplaceVertexIndex { get; set; }
            public int TotalIndexCount { get; set; }
            public int TotalVertexCount { get; set; }
        }

        private enum SkinningType
        {
            None,
            SmoothSkinning,
            RigidSkinning,
        }

        private class BoneMatrixIndexInfo
        {
            public BoneMatrixIndexInfo(boneType bone)
            {
                this.Bone = bone;
                this.OriginalSmoothSkinningMatrixIndex = bone.matrix_index[0];
                this.OriginalRigidSkinningMatrixIndex = bone.matrix_index[1];
            }

            public boneType Bone { get; }

            public int OriginalRigidSkinningMatrixIndex { get; set; } = -1;
            public int OriginalSmoothSkinningMatrixIndex { get; set; } = -1;

            public int RigidSkinningMatrixIndex { get; set; } = -1;
            public int SmoothSkinningMatrixIndex { get; set; } = -1;

            public int RigidSkinningCount { get; set; } = 0;

            public int SmoothSkinningCount { get; set; } = 0;

            public bool IsUsedInRigidSkinning { get { return RigidSkinningCount > 0; } }

            public bool IsUsedInSmoothSkinning { get { return SmoothSkinningCount > 0; } }
            public bool IsOriginalUsedInRigidSkinning { get { return OriginalRigidSkinningMatrixIndex >= 0; } }

            public bool IsOriginalUsedInSmoothSkinning { get { return OriginalSmoothSkinningMatrixIndex >= 0; } }

            public bool IsUsedInSkinning { get { return IsUsedInRigidSkinning || IsUsedInSmoothSkinning; } }
            public bool IsOriginalUsedInSkinning { get { return IsOriginalUsedInRigidSkinning || IsOriginalUsedInSmoothSkinning; } }
        }

        private class MatrixIndexCandidate
        {
            public int RigidSkinningMatrixIndex { get; set; } = -1;
            public int SmoothSkinningMatrixIndex { get; set; } = -1;
        }

        private static BoneMatrixIndexInfo[] CreateBoneMatrixIndexInfo(modelType model, List<G3dStream> streams)
        {
            int boneCount = model.skeleton.bone_array.bone.Length;
            BoneMatrixIndexInfo[] boneMatrixInfos = new BoneMatrixIndexInfo[boneCount];
            for (int boneIndex = 0; boneIndex < boneCount; ++boneIndex)
            {
                boneType bone = model.skeleton.bone_array.bone[boneIndex];
                boneMatrixInfos[boneIndex] = new BoneMatrixIndexInfo(bone);
            }

            // 各ボーンがスキニングされているかどうかを判定し、記録
            foreach (var shape in model.shape_array.shape)
            {
                vertexType vertex = model.vertex_array.vertex[shape.shape_info.vertex_index];
                var skinningIndexAttrs = vertex.vtx_attrib_array.vtx_attrib.Where(attr => IsSkinningIndex(attr));

                int firstValidMatrixIndex = -1;
                List<int> relatedMatrixIndices = new List<int>();
                foreach (vtx_attribType skinningIndexAttr in skinningIndexAttrs)
                {
                    // ストリームに含まれているマトリックスインデックスを列挙し、仮値に割り当てるための値を取得する
                    G3dStream stream = streams[skinningIndexAttr.stream_index];
                    for (int streamElemIndex = 0; streamElemIndex < stream.IntData.Count; ++streamElemIndex)
                    {
                        int matrixIndex = stream.IntData[streamElemIndex];
                        if (matrixIndex != InvalidBlendIndex && !relatedMatrixIndices.Contains(matrixIndex))
                        {
                            relatedMatrixIndices.Add(matrixIndex);
                            if (firstValidMatrixIndex == -1)
                            {
                                firstValidMatrixIndex = matrixIndex;
                            }
                        }
                    }
                }

                foreach (vtx_attribType skinningIndexAttr in skinningIndexAttrs)
                {
                    SkinningType skinningType = SkinningType.None;
                    vtx_attribType skinningWeightAttr = vertex.vtx_attrib_array.vtx_attrib.FirstOrDefault(attr => IsSkinningWeight(attr));
                    if (skinningWeightAttr != null)
                    {
                        skinningType = SkinningType.SmoothSkinning;
                    }
                    else
                    {
                        skinningType = SkinningType.RigidSkinning;
                    }

                    // ストリームに含まれているマトリックスインデックスを列挙
                    G3dStream stream = streams[skinningIndexAttr.stream_index];

                    // --ignoring-skinning-count 使用時に仮で埋められたウェイト 0 の頂点のブレンドインデックスを既存の行列インデックスに置き換える
                    Nintendo.Foundation.Contracts.Assertion.True<Exception>(firstValidMatrixIndex != -1);
                    for (int streamElemIndex = 0; streamElemIndex < stream.IntData.Count; ++streamElemIndex)
                    {
                        if (stream.IntData[streamElemIndex] == InvalidBlendIndex)
                        {
                            stream.IntData[streamElemIndex] = firstValidMatrixIndex;
                        }
                    }

                    // 列挙したマトリックスインデックスを持つボーンをカウント
                    for (int boneIndex = 0; boneIndex < model.skeleton.bone_array.bone.Length; ++boneIndex)
                    {
                        boneType bone = model.skeleton.bone_array.bone[boneIndex];
                        bool isRelatedBone = relatedMatrixIndices.Contains(bone.matrix_index[0]) || relatedMatrixIndices.Contains(bone.matrix_index[1]);
                        if (isRelatedBone)
                        {
                            BoneMatrixIndexInfo boneMatrixRef = boneMatrixInfos[boneIndex];
                            switch (skinningType)
                            {
                                case SkinningType.RigidSkinning:
                                    {
                                        boneMatrixRef.RigidSkinningCount++;
                                    }
                                    continue;
                                case SkinningType.SmoothSkinning:
                                    {
                                        boneMatrixRef.SmoothSkinningCount++;
                                    }
                                    continue;
                                case SkinningType.None:
                                default:
                                    continue;
                            }
                        }
                    }
                }
            }

            return boneMatrixInfos;
        }

        private static void AssignBoneMatrixIndex(BoneMatrixIndexInfo[] boneMatrixInfos)
        {
            // -1 で初期化
            foreach (var boneInfo in boneMatrixInfos)
            {
                boneInfo.Bone.matrix_index[0] = -1;
                boneInfo.Bone.matrix_index[1] = -1;
            }

            // マトリックスパレットはスムーススキニング、リジッドスキニングの順番に詰められていて
            // ボーンのインデックス順に行列インデックスが決まる前提
            // ここの仕様が DCC ツール側の実装と食い違うと破綻するので仕様変更時は注意

            // スムーススキニング分の割り当て
            int matrixIndex = 0;
            foreach (var boneInfo in boneMatrixInfos)
            {
                boneType bone = boneInfo.Bone;
                if (boneInfo.IsUsedInSmoothSkinning)
                {
                    boneInfo.SmoothSkinningMatrixIndex = matrixIndex;
                    bone.matrix_index[0] = matrixIndex;
                    ++matrixIndex;
                }
            }

            // リジッドスキニング分の割り当て
            foreach (var boneInfo in boneMatrixInfos)
            {
                boneType bone = boneInfo.Bone;
                if (boneInfo.IsUsedInRigidSkinning)
                {
                    boneInfo.RigidSkinningMatrixIndex = matrixIndex;
                    bone.matrix_index[1] = matrixIndex;
                    ++matrixIndex;
                }
            }
        }

        private static Dictionary<int, MatrixIndexCandidate> CreateBlendIndexConvertTable(BoneMatrixIndexInfo[] boneMatrixInfos)
        {
            Dictionary<int, MatrixIndexCandidate> blendIndexConvertTable = new Dictionary<int, MatrixIndexCandidate>();
            {
                foreach (var boneInfo in boneMatrixInfos)
                {
                    if (boneInfo.IsOriginalUsedInSkinning)
                    {
                        MatrixIndexCandidate matrixIndexDest = new MatrixIndexCandidate();
                        if (boneInfo.IsUsedInSmoothSkinning)
                        {
                            matrixIndexDest.SmoothSkinningMatrixIndex = boneInfo.SmoothSkinningMatrixIndex;
                        }

                        if (boneInfo.IsUsedInRigidSkinning)
                        {
                            matrixIndexDest.RigidSkinningMatrixIndex = boneInfo.RigidSkinningMatrixIndex;
                        }

                        if (boneInfo.IsOriginalUsedInSmoothSkinning)
                        {
                            blendIndexConvertTable.Add(boneInfo.OriginalSmoothSkinningMatrixIndex, matrixIndexDest);
                        }

                        if (boneInfo.IsOriginalUsedInRigidSkinning)
                        {
                            blendIndexConvertTable.Add(boneInfo.OriginalRigidSkinningMatrixIndex, matrixIndexDest);
                        }
                    }
                }
            }

            return blendIndexConvertTable;
        }

        private static MatrixIndexCandidate TryGetBlendIndexCandidate(int oldBlendIndex, Dictionary<int, MatrixIndexCandidate> blendIndexConvertTable)
        {
            MatrixIndexCandidate candidate;
            try
            {
                blendIndexConvertTable.TryGetValue(oldBlendIndex, out candidate);
            }
            catch (KeyNotFoundException)
            {
                throw new Exception($"Internal Error: key {oldBlendIndex} was not found in converting blend index.");
            }
            catch (Exception)
            {
                throw;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(candidate != null);

            return candidate;
        }

        private static vtx_attribType FindReleatedSkinWeightAttr(string blendIndexAttrName, vtx_attribType[] vertexAttributes)
        {
            foreach (var weightAttr in vertexAttributes.Where(attr => IsSkinningWeight(attr)))
            {
                // _i#、_w# という名前である前提
                char indexNumChar = blendIndexAttrName[2];
                char weightNumChar = weightAttr.name[2];
                if (indexNumChar == weightNumChar)
                {
                    return weightAttr;
                }
            }

            return null;
        }

        private static void ConvertBlendIndices(modelType model, List<G3dStream> streams, Dictionary<int, MatrixIndexCandidate> blendIndexConvertTable)
        {
            foreach (var vertex in model.vertex_array.vertex)
            {
                var skinningIndexAttrs = vertex.vtx_attrib_array.vtx_attrib.Where(attr => IsSkinningIndex(attr));
                foreach (vtx_attribType skinningIndexAttr in skinningIndexAttrs)
                {
                    vtx_attribType skinningWeightAttr = FindReleatedSkinWeightAttr(skinningIndexAttr.name, vertex.vtx_attrib_array.vtx_attrib);
                    G3dStream blendIndexStream = streams[skinningIndexAttr.stream_index];
                    if (skinningWeightAttr != null)
                    {
                        // スムーススキニング
                        G3dStream blendWeightStream = streams[skinningWeightAttr.stream_index];
                        for (int streamElemIndex = 0; streamElemIndex < blendIndexStream.IntData.Count; ++streamElemIndex)
                        {
                            int blendIndex = blendIndexStream.IntData[streamElemIndex];
                            MatrixIndexCandidate convertedBlendIndex = TryGetBlendIndexCandidate(blendIndex, blendIndexConvertTable);
                            blendIndexStream.IntData[streamElemIndex] = convertedBlendIndex.SmoothSkinningMatrixIndex;
                        }
                    }
                    else
                    {
                        // リジッドスキニング
                        for (int streamElemIndex = 0; streamElemIndex < blendIndexStream.IntData.Count; ++streamElemIndex)
                        {
                            int blendIndex = blendIndexStream.IntData[streamElemIndex];
                            MatrixIndexCandidate convertedBlendIndex = TryGetBlendIndexCandidate(blendIndex, blendIndexConvertTable);
                            blendIndexStream.IntData[streamElemIndex] = convertedBlendIndex.RigidSkinningMatrixIndex;
                        }
                    }
                }
            }
        }

        internal static void ReassignBoneMatrixIndex(modelType model, List<G3dStream> streams)
        {
            // ボーンがスキニングに使用されているかどうかの情報を収集
            BoneMatrixIndexInfo[] boneInfo = CreateBoneMatrixIndexInfo(model, streams);

            // 収集した情報に基づき、ボーンのマトリックスインデックスの再割り当て
            AssignBoneMatrixIndex(boneInfo);

            // 再割り当て後のマトリックスインデックスにストリームを追従させるためにブレンドインデックスのストリームを変換
            Dictionary<int, MatrixIndexCandidate> blendIndexConvertTable = CreateBlendIndexConvertTable(boneInfo);
            ConvertBlendIndices(model, streams, blendIndexConvertTable);
        }
    }
}
