﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルの合体ボーン圧縮
    public class IfModelBoneUniteCompressor : IfModelCompressor
    {
        // コンストラクタ
        public IfModelBoneUniteCompressor() :
            base("IfModelBoneUniteCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_unite"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            return this.Target.skeleton.bone_array.bone.Length;
        }

        // 圧縮
        protected override void Compress()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (this.Target.shape_array != null)
            {
                foreach (shapeType shape in this.Target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            modelType model = this.Target;
            List<G3dStream> streams = this.Streams;

            boneType[] bones = model.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];

            shapeType[] shapes;
            int shapeCount;
            ShapeInfo[] shapeInfos;
            if (this.Target.shape_array != null &&
                this.Target.shape_array.shape != null)
            {
                shapes = model.shape_array.shape;
                shapeCount = shapes.Length;
                shapeInfos = new ShapeInfo[shapeCount];
            }
            else
            {
                shapes = null;
                shapeCount = 0;
                shapeInfos = null;
            }

            // ボーンのソートチェック
            int isSorted = IfModelSortUtility.IsBoneSorted(this.Target);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelBoneCompressor_Error_NotSorted",
                    bones[isSorted].name, bones[isSorted + 1].name);
            }

            // ボーン情報の初期化
            Dictionary<string, BoneInfo> boneDict = new Dictionary<string, BoneInfo>();
            for (int i = 0; i < boneCount; i++)
            {
                boneInfos[i] = new BoneInfo(bones[i], model.skeleton.skeleton_info);
                boneDict.Add(boneInfos[i].Bone.name, boneInfos[i]);
            }
            BoneInfo.Setup(boneInfos);

            // シェープ情報の初期化
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i] = new ShapeInfo(shapes[i]); }
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i].Setup(boneInfos); }

            // 削除可能なボーンの洗い出し
            // ルートノード(boneInfos[0])は削除しないので開始インデックスが 1 となる
            for (int i = 1; i < boneCount; i++)
            {
                boneType bone = bones[i];
                BoneInfo boneInfo = boneInfos[i];
                if (CheckRemovable(bone))
                {
                    boneInfo.IsReservedToRemove = true;
                }
            }

            // 新しいボーンに出力順のインデックスを設定する
            ModelCompressUtility.SetNewIndex(boneInfos[0]);

            // 行列パレットを作成するために、パレットとして使用されるボーンを収集する。
            BonePalette bonePltt = ModelCompressUtility.CreateBonePalette(boneInfos, model);

            // リクエストホルダーを作成する
            TransformRequestHolder reqHolder =
                new TransformRequestHolder(model, streams, bonePltt);

            // Shape の移動、およびエンベロープのプリミティブ化によって必要な頂点座標と
            // 法線ベクトルの変換を収集する。
            if (shapeInfos != null)
            {
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    BoneInfo uniteTo;
                    if (!shapeInfo.Bone.IsReservedToRemove)
                    {
                        // 圧縮禁止なので Shape の移動は Skin のみ。
                        // RigidBody はボーン自身に動かす。
                        uniteTo = shapeInfo.Bone;
                    }
                    else
                    {
                        // 圧縮可能なので、上位階層の圧縮禁止ノードに Shape を移動させる。
                        uniteTo = shapeInfo.Bone.GetIncompressibleAncestorBone();
                    }

                    if (shapeInfo.SkinningMode != ShapeInfo.ShapeSkinningMode.RigidBody)
                    {
                        // スキンはルートノードへ移動

                        reqHolder.InsertTransformRequest(boneInfos[0], shapeInfo);
                    }
                    else
                    {
                        reqHolder.InsertTransformRequest(uniteTo, shapeInfo);
                    }
                }
            }

            // 頂点座標を変換する
            reqHolder.ApplyTransform();

            // <shape>を変換：
            // ・エンベロープをリジッドボディ化。
            if (shapeInfos != null)
            {
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    BoneInfo uniteTo;
                    if (!shapeInfo.Bone.IsReservedToRemove)
                    {
                        // 圧縮禁止なので Shape の移動は Skin のみ。
                        // RigidBody はボーン自身に動かす。
                        uniteTo = shapeInfo.Bone;
                    }
                    else
                    {
                        // 圧縮可能なので、上位階層の圧縮禁止ノードに Shape を移動させる。
                        uniteTo = shapeInfo.Bone.GetIncompressibleAncestorBone();
                    }

                    if (shapeInfo.SkinningMode != ShapeInfo.ShapeSkinningMode.RigidBody)
                    {
                        // スキンはルートノードへ移動
                        ModelCompressUtility.ConvertShape(model, streams, boneInfos[0], shapeInfo);
                    }
                    else
                    {
                        ModelCompressUtility.ConvertShape(model, streams, uniteTo, shapeInfo);
                    }
                }

                // <shape>を上位階層の非圧縮ボーンに移動させる。
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    shapeType shape = shapeInfo.Shape;
                    string boneName = shape.shape_info.bone_name;
                    if (!boneDict.ContainsKey(boneName))
                    {
                        // 参照しているボーンが見つからない場合はエラー
                        IfStrings.Throw("ModelCompressUtility_Error_BoneNotFound",
                            shape.name, boneName);
                    }

                    BoneInfo boneInfo = boneDict[boneName];
                    if (!boneInfo.IsReservedToRemove)
                    {
                        // 圧縮禁止のボーンはエンベロープのみを移動。
                        if (shapeInfo.SkinningMode != ShapeInfo.ShapeSkinningMode.RigidBody)
                        {
                            shapeInfo.Bone = boneInfos[0];
                            shape.shape_info.bone_name = boneInfos[0].Bone.name;
                        }
                    }
                    else
                    {
                        // 圧縮されるボーンの単純プリミティブは上階層の非圧縮ボーンへ移動。
                        // エンベロープはルートボーンへ移動。
                        if (shapeInfo.SkinningMode != ShapeInfo.ShapeSkinningMode.RigidBody)
                        {
                            shapeInfo.Bone = boneInfos[0];
                            shape.shape_info.bone_name = boneInfos[0].Bone.name;
                        }
                        else
                        {
                            BoneInfo uniteTo = boneInfo.GetIncompressibleAncestorBone();
                            shapeInfo.Bone = uniteTo;
                            shape.shape_info.bone_name = uniteTo.Bone.name;
                        }
                    }
                }
            }

            // キーシェイプを削除する
            ModelCompressUtility.RemoveKeyShape(model);

            // 削除される <bone> の座標変換を下位ノードにマージ
            for (int iBone = 1; iBone < boneInfos.Length; iBone++)
            {
                // 削除されるボーンについては処理する必要がない
                BoneInfo bi = boneInfos[iBone];
                if (bi.IsReservedToRemove)
                {
                    continue;
                }
                // 変換を合成する階層の上位ボーンを求める。
                BoneInfo uniteTo = bi.GetIncompressibleAncestorBone();
                // 直接の親が削除されない場合は変換を合成する必要がない
                if (uniteTo == bi.Parent)
                {
                    continue;
                }

                // uniteTo が参照するボーンから bi の親までのボーンが削除されるので、
                // その削除されるボーンの変換を合成して bi が参照するボーンに設定する。
                Matrix44 boneTransform =
                    ModelCompressUtility.GetBoneTransform(model, bi, uniteTo);
                Transform trans = new Transform();

                // 求めた変換を SRT の要素に分解してボーンに設定する
                boneTransform.GetTransform(trans);
                bi.Bone.translate = trans.Translate.ToArray();
                bi.Bone.scale = trans.Scale.ToArray();
                if (bi.SkeletonInfo.rotate_mode == skeleton_info_rotate_modeType.euler_xyz)
                {
                    // boneType.rotate は４要素必用なので、オイラー各で回転を保持する場合は
                    // 必ず配列の [3] に 1.0 を設定しておく。
                    Vector3 euler = EulerXyz.ToDegree(trans.Rotate);
                    bi.Bone.rotate = new float[] { euler.X, euler.Y, euler.Z, 1.0f };
                }
                else
                {
                    Quaternion quat = EulerXyz.ToQuaternion(trans.Rotate);
                    bi.Bone.rotate = quat.ToArray();
                }
                // 親ノードとの合成後は正しく扱えないので scale_compensate は無効にしておく。
                bi.Bone.scale_compensate = false;
            }

            // rigid_body フラグとスキニング関連情報の更新。
            // 一度全てのボーンの rigid_body フラグを false にし、
            // 使用されているボーンのみ true に設定し直す。
            foreach (BoneInfo boneInfo in boneInfos)
            {
                boneInfo.Bone.rigid_body = false;
                // スキン関連の情報を削除しておく
                boneInfo.Bone.matrix_index[0] =
                boneInfo.Bone.matrix_index[1] = -1;
                boneInfo.Bone.inv_model_matrix = null;
            }
            if (shapeInfos != null)
            {
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    BoneInfo boneInfo = shapeInfo.Bone;
                    boneInfo.Bone.rigid_body = true;
                }
            }

            // ボーンの削除
            ModelCompressUtility.RemoveBones(model, boneInfos);
            // シェイプをソートする
            ModelCompressUtility.SortShape(model);
            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(model, streams);
        }

        // 削除可能かチェック
        private bool CheckRemovable(boneType bone)
        {
            // 圧縮禁止
            if (!bone.compress_enable) { return false; }

            return true;
        }
    }
}
