﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using System.IO;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using nw.g3d.iflib;

namespace nw.g3d.ifcvtr
{
    // obj コンバータ
    internal class ObjConverter : Converter
    {
        private string g3dToolRoot = null;
        private CommandLineOptions options = null;

        // コンストラクタ
        internal ObjConverter(string sourcePath, CommandLineOptions param, string g3dToolRoot)
            : base(sourcePath)
        {
            this.g3dToolRoot = g3dToolRoot;

            if (!string.IsNullOrEmpty(param.Output))
            {
                if (G3dPath.IsModelPath(param.Output))
                {
                    this.OutputFilePath = param.Output;
                }
                else
                {
                    this.OutputFilePath = Path.ChangeExtension(
                        param.Output, G3dPath.ModelBinaryExtension);
                }
            }
            else
            {
                this.OutputFilePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.ModelBinaryExtension);
            }

            this.options = param;
            this.InvertUvVertical = !param.DisableInvertUv;
        }

        //=====================================================================
        // 変換
        internal override void Convert()
        {
            ObjContext context = Load();
            {
                ObjIfBuilder builder = new ObjIfBuilder();
                builder.Build(context);
            }
            Save(context);

            RegisterSubConverter(context);
        }

        //=====================================================================
        // 読み込み
        private ObjContext Load()
        {
            var context = new ObjContext(
                this.SourceFilePath, this.OutputFilePath, this.options.Magnify, this.InvertUvVertical, this.options.ProjectRoot);

            // obj のパース、パースを複数種設けるならここで
            ObjParser objParser = new ObjParser();
            objParser.Parse(context);

            // mtl のパース
            MtlParser mtlParser = new MtlParser();
            foreach (string mtllib in context.Mtllibs)
            {
                mtlParser.Parse(context, mtllib);
            }

            return context;
        }

        //=====================================================================
        // 保存
        private void Save(ObjContext context)
        {
            string savePath = this.OutputFilePath;
            if (savePath != null)
            {
                if (!G3dPath.IsModelPath(savePath))
                {
                    savePath = Path.ChangeExtension(
                        savePath, G3dPath.ModelBinaryExtension);
                }
            }
            else
            {
                savePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.ModelBinaryExtension);
            }
            IfWriteUtility.WriteIntermediateFile(context.File, savePath, G3dToolUtility.GetXsdBasePath());
        }

        //=====================================================================
        // サブコンバータの登録
        private void RegisterSubConverter(ObjContext context)
        {
            foreach (ObjMaterial material in context.Materials)
            {
                RegisterTexture(context, material, material.AlbedoTexture);
                RegisterTexture(context, material, material.OpacityTexture);
                RegisterTexture(context, material, material.EmissionTexture);
                RegisterTexture(context, material, material.SpecularTexture);
            }
        }

        // テクスチャの登録
        private void RegisterTexture(ObjContext context, ObjMaterial material, string path)
        {
            if (this.options.DisableTexture)
            {
                return;
            }

            if (path == null) { return; }

            // name が重複する場合は登録しない
            string name = Path.GetFileNameWithoutExtension(path);
            int index = this.SubConverters.FindIndex(delegate(Converter match)
            {
                return Path.GetFileNameWithoutExtension(match.SourceFilePath) == name;
            });
            if (index != -1) { return; }

            string sourcePath = Path.Combine(
                Path.GetDirectoryName(material.FilePath), path);
            string outputPath = Path.Combine(Path.GetDirectoryName(context.OutputPath),
                "textures\\", name + G3dPath.TextureBinaryExtension);

            this.SubConverters.Add(new ImageConverter(sourcePath, outputPath, this.options, this.g3dToolRoot));
        }

        internal bool InvertUvVertical { get; set; } = false;
    }
}
