﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;
using nw.g3d.nw4f_3dif;
using System.Reflection;
using nw.g3d.iflib;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;

namespace nw.g3d.ifcvtr
{
    // 画像コンバータ
    internal class ImageConverter : Converter
    {
        private string g3dToolRoot = null;
        private CommandLineOptions options = null;

        // コンストラクタ
        internal ImageConverter(string sourcePath, CommandLineOptions param, string g3dToolRoot)
            : base(sourcePath)
        {
            this.g3dToolRoot = g3dToolRoot;
            if (!string.IsNullOrEmpty(param.Output))
            {
                if (G3dPath.IsTexturePath(param.Output))
                {
                    this.OutputFilePath = param.Output;
                }
                else
                {
                    this.OutputFilePath = Path.ChangeExtension(param.Output,
                        G3dPath.TextureBinaryExtension);
                }
            }
            else
            {
                this.OutputFilePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.TextureBinaryExtension);
            }
            this.options = param;
        }

        // コンストラクタ
        internal ImageConverter(
            string sourceFilePath,
            string outputFilePath,
            CommandLineOptions param,
            string g3dToolRoot)
            : base(sourceFilePath, outputFilePath)
        {
            this.options = param;
            this.g3dToolRoot = g3dToolRoot;
        }

        //=====================================================================
        // 変換
        internal override void Convert()
        {
            string outputPath = Path.GetFullPath(this.OutputFilePath);

            string tmpFile = null;
            string tmpFolder = null;
            string tmpTgaPath = null;
            try
            {
                // テンポラリフォルダの構築
                tmpFile = Path.GetTempFileName();
                tmpFolder = Path.Combine(Path.GetDirectoryName(tmpFile),
                    Path.GetFileNameWithoutExtension(tmpFile));
                Directory.CreateDirectory(tmpFolder);

                // 出力先ディレクトリの作成
                string outputFolder = Path.GetDirectoryName(outputPath);
                if (!Directory.Exists(outputFolder))
                {
                    Directory.CreateDirectory(outputFolder);
                }

                if (System.IO.Path.GetExtension(this.SourceFilePath) != ".tga")
                {
                    // イメージのロード
                    string srcPath = Path.GetFullPath(this.SourceFilePath);
                    bool hasAlpha = false;
                    using (FileStream stream = File.OpenRead(srcPath))
                    {
                        Bitmap bitmap = new Bitmap(stream);
                        hasAlpha = (bitmap.PixelFormat &
                            (PixelFormat.Alpha | PixelFormat.PAlpha)) != 0;

                        // TGA で保存
                        tmpTgaPath = Path.GetFullPath(Path.Combine(tmpFolder,
                            Path.GetFileNameWithoutExtension(this.SourceFilePath) + ".tga"));
                        SaveTga(bitmap, tmpTgaPath);
                    }

                    // テクスチャコンバータで変換
                    Convert(tmpTgaPath, outputPath, hasAlpha, this.options.ProjectRoot);
                }
                else
                {
                    // tga は Bitmap が対応していないので直変換
                    bool hasAlpha = true;
                    Convert(this.SourceFilePath, outputPath, hasAlpha, this.options.ProjectRoot);
                }
            }
            finally
            {
                if (tmpTgaPath != null) { File.Delete(tmpTgaPath); }
                if (tmpFolder != null) { Directory.Delete(tmpFolder, true); }
                if (tmpFile != null) { File.Delete(tmpFile); }
            }
        }

        //=====================================================================
        // TGA で保存
        private void SaveTga(Bitmap bitmap, string tgaPath)
        {
            // 元画像の取得
            bool hasAlpha = (bitmap.PixelFormat &
                (PixelFormat.Alpha | PixelFormat.PAlpha)) != 0;
            PixelFormat pixelFormat =
                hasAlpha ? PixelFormat.Format32bppArgb : PixelFormat.Format24bppRgb;
            BitmapData bitmapData = bitmap.LockBits(
                new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                ImageLockMode.ReadOnly, pixelFormat);
            IntPtr addr = bitmapData.Scan0;

            MemoryStream stream = new MemoryStream();
            using (BinaryWriter writer = new BinaryWriter(stream))
            {
                // ID Length
                writer.Write((byte)0);
                // Color Map Type
                writer.Write((byte)0);
                // Image Field (Uncompress)
                writer.Write((byte)2);
                // Color map First Entry Index
                writer.Write((short)0);
                // Color map Length
                writer.Write((short)0);
                // Color map Entry Size
                writer.Write((byte)0);
                // X-origin
                writer.Write((short)0);
                // Y-origin
                writer.Write((short)0);
                // Width
                writer.Write((short)bitmap.Width);
                // Height
                writer.Write((short)bitmap.Height);
                // Pixel Depth
                writer.Write((byte)(hasAlpha ? 32 : 24));
                // Image Descriptor
                writer.Write((byte)32);

                if (hasAlpha)
                {
                    for (int i = 0; i < bitmapData.Height; i++)
                    {
                        int offset = bitmapData.Stride * i;
                        for (int j = 0; j < bitmapData.Width; j++)
                        {
                            writer.Write(Marshal.ReadInt32(addr, offset + j * 4));
                        }
                    }
                }
                else
                {
                    for (int i = 0; i < bitmapData.Height; i++)
                    {
                        int offset = bitmapData.Stride * i;
                        for (int j = 0; j < bitmapData.Width; j++)
                        {
                            writer.Write(Marshal.ReadInt16(addr, offset + j * 3));
                            writer.Write(Marshal.ReadByte(addr, offset + j * 3 + 2));
                        }
                    }
                }
            }
            stream.Close();
            bitmap.UnlockBits(bitmapData);
            File.WriteAllBytes(tgaPath, stream.ToArray());
        }

        //=====================================================================
        // 変換
        private void Convert(string tgaPath, string outputPath, bool hasAlpha, string projectRoot)
        {
            var useProjectRoot = !string.IsNullOrEmpty(projectRoot);
            var tmpFile = Path.GetTempFileName() + ".ftxb";
            try
            {
                ProcessExecutor processExecutor = new ProcessExecutor();

                string texConverterPath = Path.Combine(
                    this.g3dToolRoot,
                    "3dTextureConverter.exe");
                Ensure.Operation.True(System.IO.File.Exists(texConverterPath), $"{texConverterPath} was not found");

                StringBuilder args = new StringBuilder();
                string format = (this.options.TextureFormatSrgb ? "srgb_" : "unorm_") + (hasAlpha ? "bc3" : "bc1");
                args.Append($"\"{tgaPath}\" -o \"{tmpFile}\" -f {format} -s --gpu-encoding {this.options.GpuEncoding}");
                if (this.options.DisableOriginalImage)
                {
                    args.Append(" --disable-original-image");
                }

                int exitCode = processExecutor.Execute(texConverterPath, args.ToString());
                if (exitCode != 0)
                {
                    string errorMessage = processExecutor.LogMessage + "\n" + processExecutor.ErrorMessage;
                    Debug.WriteLine(errorMessage);
                    Strings.Throw("Error_TextureConverterError", errorMessage);
                }

                // src_pathを修正
                {
                    var file = IfReadUtility.ReadIntermediateFile(tmpFile, G3dToolUtility.GetXsdBasePath());
                    var fullPath = Path.GetFullPath(SourceFilePath);
                    var srcPath = fullPath;
                    if (useProjectRoot)
                    {
                        srcPath = ObjIfBuilder.MakeRelativePath(Path.GetFullPath(projectRoot), srcPath);
                    }

                    file.FileInfo.SetCreationInfo(srcPath.Replace('\\', '/'));
                    var texture = file.GetRootEntity<Texture>();
                    if (texture.OriginalImages.Count == 1)
                    {
                        var original = texture.OriginalImages.First();
                        var originalPath = ObjIfBuilder.MakeRelativePath(Path.GetDirectoryName(outputPath), fullPath).Replace('\\', '/');
                        original.OriginalPath = Path.IsPathRooted(originalPath) ? Path.GetFileName(fullPath) : originalPath;
                    }
                    IfWriteUtility.WriteIntermediateFile(file, outputPath, G3dToolUtility.GetXsdBasePath());
                }
            }
            finally
            {
                if (tmpFile != null)
                {
                    File.Delete(tmpFile);
                }
            }
        }
    }
}
