﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using nw.g3d.toollib;
using Nintendo.G3dTool.Entities;

namespace nw.g3d.ifcvtr
{
    internal class IfModelConverter : Converter
    {
        private string xsdBasePath = null;
        private CommandLineOptions options = null;

        // コンストラクタ
        internal IfModelConverter(string sourcePath, CommandLineOptions param, string inputXsdBasePath)
            : base(sourcePath)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.True(!string.IsNullOrEmpty(inputXsdBasePath));
            this.xsdBasePath = inputXsdBasePath;

            if (!string.IsNullOrEmpty(param.Output))
            {
                if (G3dPath.IsModelPath(param.Output))
                {
                    this.OutputFilePath = param.Output;
                }
                else
                {
                    this.OutputFilePath = Path.ChangeExtension(param.Output,
                        G3dPath.ModelBinaryExtension);
                }
            }
            else
            {
                this.OutputFilePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.ModelBinaryExtension);
            }

            this.options = param;
        }

        //=====================================================================
        // 変換
        internal override void Convert()
        {
            string outputPath = Path.GetFullPath(this.OutputFilePath);

            // モデル中間ファイルのロード
            string srcPath = Path.GetFullPath(this.SourceFilePath);
            var file = IfReadUtility.ReadIntermediateFile(srcPath, xsdBasePath);
            Model readModel = file.GetRootEntity<Model>();

            // Magnify を適用する
            ApplyMagnify(readModel);

            // 書き出す
            IfWriteUtility.WriteIntermediateFile(file, outputPath, xsdBasePath);
        }

        /**
         * model に Magnify を適用します。
         */
        private void ApplyMagnify(Model model)
        {
            // 頂点位置情報を更新
            foreach (var vertex in model.Vertexes)
            {
                foreach (var vtxAttrib in vertex.VtxAttribs)
                {
                    if (!vtxAttrib.Name.StartsWith("_p"))
                    {
                        continue;
                    }

                    StreamFloat stream = vtxAttrib.Stream as StreamFloat;
                    for (int i = 0; i < stream.Count; ++i)
                    {
                        stream.Values[i] *= this.options.Magnify;
                    }

                    // 値を変更しているため、量子化精度を無効にする。
                    vtxAttrib.QuantizeType = vtx_attrib_quantize_typeType.none;
                }
            }

            // ボーンの位置情報を更新
            {
                foreach (var bone in model.Skeleton.EnumerateBones())
                {
                    bone.Translate.X *= this.options.Magnify;
                    bone.Translate.Y *= this.options.Magnify;
                    bone.Translate.Z *= this.options.Magnify;
                }
            }
        }
    }
}
