﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using System.IO;

namespace nw.g3d.ifassign
{
    internal class UpdateShaderParamAnimationUtility : AssignUtility
    {
        // コンストラクタ
        internal UpdateShaderParamAnimationUtility(g3difassignParams.UpdateAnimationSubCommand programOption)
            : base(programOption)
        {
            this.ShaderPath = programOption.Shader;
            if (!File.Exists(this.ShaderPath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_Assign_ShaderFileNotFound, this.ShaderPath));
            }
            if (!G3dPath.IsShaderDefinitionPath(this.ShaderPath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_Assign_InvalidFileType, this.ShaderPath));
            }
            if (!G3dPath.IsBinaryPath(this.ShaderPath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_Assign_NotBinaryFileType, this.ShaderPath));
            }

            this.ShadingModel = programOption.ShadingModel;

            this.Test = programOption.Test;

            Input = programOption.Path;
            Output = programOption.Output;
        }

        //=====================================================================
        // 処理
        internal override void Process()
        {
            // シェーダ定義ファイルの読み込み
            var shaderDefinition = g3difassign.GetShaderDefinitionFromPath(this.ShaderPath, this.ShaderDefinitionStreams);

            // シェーディングモデル
            shading_modelType shading_model = null;
            if (shaderDefinition.shading_model_array != null)
            {
                shading_model = shaderDefinition.shading_model_array.shading_model.FirstOrDefault(x => x.name == this.ShadingModel);
            }
            if (shading_model == null)
            {
                throw new Exception(string.Format(Resources.StringResource.Error_ShadingModelNotFoundInShaderDefinition, this.ShadingModel, this.ShaderPath));
            }
            if (!shading_model.material_shader)
            {
                throw new Exception(string.Format(Resources.StringResource.Error_ShadingModelNotForMaterial_Animation, this.ShadingModel));
            }

            MessageBuilder.Append(DeprecatedMessage(shading_model));

            string sourcePath = Input;
            var isShaderParamAnim = G3dPath.IsShaderParamAnimGroupPath(sourcePath);
            var isMaterialAnim = G3dPath.IsMaterialAnimPath(sourcePath);
            // パスのチェック
            if (!isShaderParamAnim && !isMaterialAnim)
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileIsNotAnimation, sourcePath));
            }
            if (!File.Exists(sourcePath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileNotFound, sourcePath));
            }

            // 出力パスの設定
            string outputPath = sourcePath;
            if (Output != null)
            {
                outputPath = Output;

                // もっと厳密にチェックもできけるけど複雑なケースもあるので簡易チェック
                string inExt = Path.GetExtension(sourcePath).ToLower();
                string outExt = Path.GetExtension(outputPath).ToLower();
                if (outExt.Length == 0 ||
                    !"ab".Contains(outExt[outExt.Length - 1]) ||
                    inExt.Substring(0, inExt.Length - 1) != outExt.Substring(0, outExt.Length - 1))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_OutputFileExt, outputPath));
                }
            }

            // "{sourcePath} に対する処理を行います。"
            MessageBuilder.AppendLine(string.Format(Resources.StringResource.UpdateExecute, sourcePath));

            // アニメーション読み込み
            List<G3dStream> streams = new List<G3dStream>();
            nw4f_3difType nw4f_3dif = IfReadUtility.Read(streams, sourcePath, g3difassign.XsdBasePath);

            if (isShaderParamAnim)
            {
                var shader_param_anim = (shader_param_animType)nw4f_3dif.Item;

                // マテリアルごとに処理
                var shader_param_mat_anims = shader_param_anim.shader_param_mat_anim_array != null ?
                    shader_param_anim.shader_param_mat_anim_array.shader_param_mat_anim :
                    Enumerable.Empty<shader_param_mat_animType>();
                bool hasError = false;
                foreach (var shader_param_mat_anim in shader_param_mat_anims)
                {
                    var errors = IfAnimationAssignUtility.IsConsistentWithShadingModel(
                        shader_param_mat_anim,
                        shading_model).ToArray();

                    if (errors.Length > 0)
                    {
                        hasError = true;
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.UpdateMaterialAnim, shader_param_mat_anim.mat_name));
                        foreach (var error in errors)
                        {
                            MessageBuilder.AppendLine(error.Item1);

                            // 変換
                            IfAnimationAssignUtility.Convert(error.Item2, error.Item3, error.Item4);
                        }
                    }
                }

                if (hasError)
                {
                    // ストリームのソート(や削除等)を行う。
                    nw.g3d.iflib.StreamUtility.SortStream(shader_param_anim, streams);
                }
            }
            else
            {
                var material_anim = (material_animType)nw4f_3dif.Item;

                // マテリアルごとに処理
                var per_material_anims = material_anim.per_material_anim_array != null ?
                    material_anim.per_material_anim_array.per_material_anim :
                    Enumerable.Empty<per_material_animType>();

                bool hasError = false;
                foreach (var shader_param_mat_anim in per_material_anims)
                {
                    var errors = IfAnimationAssignUtility.IsConsistentWithShadingModel(
                        shader_param_mat_anim,
                        shading_model).ToArray();

                    if (errors.Length > 0)
                    {
                        hasError = true;
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.UpdateMaterialAnim, shader_param_mat_anim.mat_name));
                        foreach (var error in errors)
                        {
                            MessageBuilder.AppendLine(error.Item1);

                            // 変換
                            IfAnimationAssignUtility.Convert(error.Item2, error.Item3, error.Item4);
                        }
                    }
                }

                if (hasError)
                {
                    // ストリームのソート(や削除等)を行う。
                    nw.g3d.iflib.StreamUtility.SortStream(material_anim, streams);
                }
            }

            // 中間ファイル出力
            if (!this.DisableFileInfo) { IfFileLogUtility.SetModify(nw4f_3dif); }
            if (!Test)
            {
                IfWriteUtility.Write(nw4f_3dif, streams, outputPath);
            }
        }

        private readonly string ShaderPath;
        private readonly string ShadingModel;
        private readonly bool Test;
        private readonly string Input;
        private readonly string Output;
        private List<G3dStream> ShaderDefinitionStreams = new List<G3dStream>();
    }
}
