﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using System.IO;
using System.Xml;
using nw.g3d.iflib.nw3de;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;

namespace nw.g3d.ifassign
{
    internal class AssignSearchPathUtility : AssignUtility
    {
        // コンストラクタ
        internal AssignSearchPathUtility(g3difassignParams.AssignSearchPathSubCommand programOption)
            : base(programOption)
        {
            if (programOption.SearchPaths != null && programOption.SearchPathsOld != null)
            {
                throw new Exception(Resources.StringResource.Error_SearchPathsAndSearchPath);
            }
            SearchPaths = (programOption.SearchPaths ?? Enumerable.Empty<string>())
                .Concat(programOption.SearchPathsOld ?? Enumerable.Empty<string>()).ToArray();
            Recursive = programOption.Recursive;
            Input = programOption.Path;
            Output = programOption.Output;
        }

        //=====================================================================
        // 処理
        internal override void Process()
        {
            // データ
            nw3de_SearchPath nw3de_SearchPath = null;
            if (SearchPaths.Any())
            {
                nw3de_SearchPath = new nw3de_SearchPath()
                {
                    SearchPaths = SearchPaths.Select(x => new nw3de_SearchPath.SearchPath() {
                        Path = x.Replace('\\', '/'), // 中間ファイルの仕様に合わせるために \ は / に置き換える
                        Recursive = Recursive,
                    }).ToList(),
                };
            }

            // 引数ファイル毎
            {
                string sourcePath = Input;

                // パスのチェック
                if (!File.Exists(sourcePath))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_InputFileNotFound, sourcePath));
                }
                if (!G3dPath.IsModelPath(sourcePath) && !G3dPath.IsTexPatternAnimPath(sourcePath) && !G3dPath.IsMaterialAnimPath(sourcePath))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_InputFileIsNotModelFtpFma, sourcePath));
                }

                // 出力パスの設定
                string outputPath = sourcePath;
                if (Output != null)
                {
                    outputPath = Output;

                    // もっと厳密にチェックもできけるけど複雑なケースもあるので簡易チェック
                    string inExt = Path.GetExtension(sourcePath).ToLower();
                    string outExt = Path.GetExtension(outputPath).ToLower();
                    if (outExt.Length == 0 ||
                        !"ab".Contains(outExt[outExt.Length - 1]) ||
                        inExt.Substring(0, inExt.Length - 1) != outExt.Substring(0, outExt.Length - 1))
                    {
                        throw new Exception(string.Format(Resources.StringResource.Error_OutputFileExt, outputPath));
                    }
                }

                {
                    // ファイル読み込み
                    var file = IfReadUtility.ReadIntermediateFile(sourcePath, g3difassign.XsdBasePath);

                    var searchPath = file.ToolData.FindToolDataObject<nw3de_SearchPath>();
                    if (searchPath != null)
                    {
                        // 既存のサーチパスを削除
                        file.ToolData.RemoveToolDataObject(searchPath);
                    }

                    if (nw3de_SearchPath != null)
                    {
                        // 追加
                        file.ToolData.AddToolDataObject(nw3de_SearchPath);
                    }
                    else
                    {
                        // 削除扱いなので何もしない
                    }

                    // 中間ファイル出力
                    if (!this.DisableFileInfo) { file.FileInfo.SetModificationInfo(); }
                    IfWriteUtility.WriteIntermediateFile(file, outputPath, g3difassign.XsdBasePath);
                }
            }
        }

        private readonly string[] SearchPaths;
        private readonly bool Recursive;
        private readonly string Input;
        private readonly string Output;
    }
}
