﻿using System;
using System.IO;
using System.Linq;
using nw.g3d.ifassign.Resources;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.ComponentModel;

namespace nw.g3d.ifassign
{
    internal class ApplyParentMaterialUtility : AssignUtility
    {
        private readonly string Input;
        private readonly string Output;
        private readonly string ShaderDefinitionFolder;
        private readonly string[] ParentMaterialPaths;

        public ApplyParentMaterialUtility(g3difassignParams.ApplyParentMaterialSubCommand programOption)
            : base(programOption)
        {
            ShaderDefinitionFolder = Environment.ExpandEnvironmentVariables(programOption.ShaderPath);
            if (!Directory.Exists(ShaderDefinitionFolder))
            {
                throw new Exception(string.Format(StringResource.Error_Assign_ShaderDirectorNotFound, ShaderDefinitionFolder));
            }

            Input = programOption.Path;
            Output = programOption.Output;
            ParentMaterialPaths = programOption.ParentMaterialPaths?.SelectMany(x => x.Split(';')).ToArray();
        }

        internal override void Process()
        {
            string modelPath;
            string outputPath;
            ModelInOutPath(Input, Output, out modelPath, out outputPath);

            // "{modelPath} に対する処理を行います。"
            MessageBuilder.AppendLine(string.Format(StringResource.UpdateExecute, modelPath));

            try
            {
                IfApplyBaseMaterialUtility.ApplyBaseMaterial(modelPath, outputPath, ShaderDefinitionFolder, DisableFileInfo, g3difassign.XsdBasePath, ParentMaterialPaths);
            }
            catch (WarningException exception)
            {
                MessageBuilder.AppendLine(exception.Message);
            }
            catch (Exception)
            {
                throw;
            }
        }

        public static void ModelInOutPath(string input, string output, out string modelPath, out string outputPath)
        {
            modelPath = Path.GetFullPath(Environment.ExpandEnvironmentVariables(input));
            outputPath = modelPath;

            // パスのチェック
            if (!File.Exists(modelPath))
            {
                throw new Exception(string.Format(StringResource.Error_InputFileNotFound, modelPath));
            }
            if (!G3dPath.IsModelPath(modelPath))
            {
                throw new Exception(string.Format(StringResource.Error_InputFileIsNotModel, modelPath));
            }

            // 出力パスの設定
            if (output != null)
            {
                if (Directory.Exists(output))
                {
                    // フォルダーの場合、入力ファイルと同名で出力
                    string fileName = Path.GetFileName(modelPath);
                    outputPath = Path.Combine(output, fileName);
                }
                else
                {
                    outputPath = output;

                    // もっと厳密にチェックもできけるけど複雑なケースもあるので簡易チェック
                    string inExt = Path.GetExtension(modelPath).ToLower();
                    string outExt = Path.GetExtension(outputPath).ToLower();
                    if (outExt.Length == 0 ||
                        !"ab".Contains(outExt[outExt.Length - 1]) ||
                        inExt.Substring(0, inExt.Length - 1) != outExt.Substring(0, outExt.Length - 1))
                    {
                        throw new Exception(string.Format(StringResource.Error_OutputFileExt, outputPath));
                    }
                }
            }
        }
    }
}
