﻿namespace ShaderAssistExAddons.Modules.ParamCode
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading.Tasks;
    using _3dExcelToShaderAnnotation.Resources;
    using Excel = Microsoft.Office.Interop.Excel;
    using nw.g3d.nw4f_3dif;

    /// <summary>
    /// シェーダパラメータコード自動生成の処理を行うクラスです。
    /// </summary>
    public class MakeParamCodeExecutor
    {
        public string AutogenFileName { get; protected set; }
        public string AutogenFileDir { get; protected set; }

        private enum Record : int
        {
            Group,
            Type,
            Variable,
            Id,
            Label,
            Default,
            OtherAnnotation,
            Preprocess
        }

        private List<PageData> _pageList = new List<PageData>();
        private List<GroupData> _groupList = new List<GroupData>();
        private List<ParamCodeData> _dataList = new List<ParamCodeData>();
        private string _xlsxFilePath = string.Empty;
        private string _shaderOptionUB = string.Empty;
        private string _shaderOptionUBId = string.Empty;
        private string _materialUB = string.Empty;
        private string _materialUBId = string.Empty;
        private string _output = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MakeParamCodeExecutor()
        {
            AutogenFileName = string.Empty;
            AutogenFileDir = string.Empty;
        }

        /// <summary>
        /// コード生成を実行します。
        /// </summary>
        /// <param name="xlsxFilePath">ファイルパスです。</param>
        /// <returns>結果を返します。</returns>
        public Result Execute(string xlsxFilePath)
        {
            this._pageList.Clear();
            this._groupList.Clear();
            this._dataList.Clear();

            this._xlsxFilePath = xlsxFilePath;

            var result = new Result();

            using (var excelContex = new ExcelContext())
            {
#if !DEBUG
                try
#endif
                {
                    excelContex.Open(xlsxFilePath);

                    this.ExtractFromExcel(excelContex);

                    result.IsSuccessful = this.GenerateCode();
                }
#if !DEBUG
                catch (Exception e)
                {
                    result.IsSuccessful = false;

                    if (!string.IsNullOrEmpty(e.Message))
                    {
                        result.ErrorMessage = e.Message;
                    }
                    else
                    {
                        result.ErrorMessage = MakeParamCodeStrings.ErrorWrongFormat;
                    }
                }
#endif
            }

            return result;
        }

        /// <summary>
        /// 生成したコードを書き込みます。
        /// </summary>
        /// <param name="filePath">書き込み先パスです。</param>
        public void Write(string filePath)
        {
            Nintendo.ToolFoundation.Contracts.Assertion.Operation.NotNull(_output);
            System.IO.StreamWriter sw = new System.IO.StreamWriter(filePath, false, System.Text.Encoding.GetEncoding("UTF-8"));
            sw.Write(_output);
            sw.Close();
        }

        [Conditional("DEBUG")]
        private void DebugLogLine(string message)
        {
            Console.WriteLine(message);
        }

        //---------------------------------------------------------------------

        private void ExtractFromExcel(ExcelContext excelContex)
        {
            // 共通設定シート
            Excel.Worksheet oCommonSheet = (Excel.Worksheet)excelContex.Workbook.Sheets[1];
            {
                Excel.Range range = oCommonSheet.UsedRange;

                {
                    Excel.Range cell = range.Find("File Name", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        AutogenFileName = ((Excel.Range)range.Cells[cell.Row, cell.Column + 1]).Text.ToString();
                    }
                }
                {
                    Excel.Range cell = range.Find("Output Directory", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        AutogenFileDir = ((Excel.Range)range.Cells[cell.Row, cell.Column + 1]).Text.ToString();
                    }
                }
                {
                    Excel.Range cell = range.Find("Shader Option Uniform Block / Id", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        _shaderOptionUB = ((Excel.Range)range.Cells[cell.Row, cell.Column + 1]).Text.ToString();
                        _shaderOptionUBId = ((Excel.Range)range.Cells[cell.Row, cell.Column + 2]).Text.ToString();
                    }

                    if (_shaderOptionUB == string.Empty)
                    {
                        _shaderOptionUB = "layout(std140) uniform Options";
                    }
                    if (_shaderOptionUBId == string.Empty)
                    {
                        _shaderOptionUBId = "option";
                    }
                }
                {
                    Excel.Range cell = range.Find("Material Uniform Block / Id", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        _materialUB = ((Excel.Range)range.Cells[cell.Row, cell.Column + 1]).Text.ToString();
                        _materialUBId = ((Excel.Range)range.Cells[cell.Row, cell.Column + 2]).Text.ToString();
                    }

                    if (_materialUB == string.Empty)
                    {
                        _materialUB = "layout(std140) uniform Material";
                    }
                    if (_materialUBId == string.Empty)
                    {
                        _materialUBId = "material";
                    }
                }

                {
                    Excel.Range cell = range.Find("page", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        for (int idx = cell.Row + 1; true; ++idx)
                        {
                            string pageId = ((Excel.Range)range.Cells[idx, cell.Column]).Text.ToString();
                            if (pageId == string.Empty)
                            {
                                break;
                            }

                            PageData page = new PageData()
                            {
                                Id = pageId,
                                Label = ((Excel.Range)range.Cells[idx, cell.Column + 1]).Text.ToString()
                            };
                            _pageList.Add(page);
                        }
                    }
                }

                // TODO: Find を厳密に。
                // TODO: 重複がないかチェック。
                {
                    Excel.Range cell = range.Find("group", Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole, MatchCase: false);
                    if (cell != null)
                    {
                        for (int idx = cell.Row + 1; true; ++idx)
                        {
                            string groupId = ((Excel.Range)range.Cells[idx, cell.Column]).Text.ToString();
                            if (groupId == string.Empty)
                            {
                                break;
                            }

                            GroupData group = new GroupData()
                            {
                                Id = groupId,
                                Label = ((Excel.Range)range.Cells[idx, cell.Column + 1]).Text.ToString(),
                                Condition = ((Excel.Range)range.Cells[idx, cell.Column + 2]).Text.ToString(),
                                Page = string.Empty
                            };
                            _groupList.Add(group);
                        }
                    }
                }
            }

            // 各ページのシート
            for (int sheetIndex = 2; sheetIndex <= excelContex.Workbook.Sheets.Count; ++sheetIndex)
            {
                Excel.Worksheet pageSheet = (Excel.Worksheet)excelContex.Workbook.Sheets[sheetIndex];
                DebugLogLine($"Parsing {pageSheet.Name}");
                ExtractPageSheetData(pageSheet);
            }

            // 共通設定シートで設定されていないページを追加する。
            foreach (var data in _dataList)
            {
                GroupData groupData = _groupList.FirstOrDefault(x => x.Id == data.Group);
                if (groupData == null)
                {
                    continue;
                }

                PageData pageData = _pageList.FirstOrDefault(x => x.Id == groupData.Page);
                if (pageData == null && (groupData.Page != string.Empty && groupData.Page != oCommonSheet.Name))
                {
                    pageData = new PageData()
                    {
                        Id = groupData.Page,
                        Label = string.Empty
                    };
                    _pageList.Add(pageData);
                }
            }

            // 共通設定ページはページとしては追加しないので取り除く。
            foreach (var group in _groupList)
            {
                if (group.Page == oCommonSheet.Name)
                {
                    group.Page = string.Empty;
                }
            }
        }

        private readonly string[] NecessaryHeaderLabels = new string[]
            {
                "group",
                "type",
                "variable",
                "id",
            };

        private int FindPageSheetHeaderRowIndex(Excel.Worksheet pageSheet)
        {
            foreach (string label in NecessaryHeaderLabels)
            {
                Excel.Range cell = pageSheet.UsedRange.Find(label);
                while (cell != null)
                {
                    Excel.Range headerRow = pageSheet.UsedRange.Rows[cell.Row];
                    try
                    {
                        foreach (string findingLabel in NecessaryHeaderLabels)
                        {
                            FindColumnNo(headerRow, findingLabel, true);
                        }
                        return cell.Row;
                    }
                    catch (Exception)
                    {
                        cell = pageSheet.UsedRange.FindNext();
                        continue;
                    }
                }
            }

            return -1;
        }

        private void ExtractPageSheetData(Excel.Worksheet pageSheet)
        {
            Excel.Range range = pageSheet.UsedRange;
            Excel.Range groupRange = pageSheet.UsedRange.Find("type");
            int groupRowIndex = FindPageSheetHeaderRowIndex(pageSheet);
            Nintendo.ToolFoundation.Contracts.Ensure.Operation.True(groupRowIndex >= 0, "Title row was not found, may be missing necessary column.");

            // ヘッダは欠けていてはならない。
            Excel.Range rangeHeader = range.Rows[groupRowIndex];
            int noGroupId = FindColumnNo(rangeHeader, "group", true);
            int noType = FindColumnNo(rangeHeader, "type", true);
            int noVariable = FindColumnNo(rangeHeader, "variable", true);
            int noId = FindColumnNo(rangeHeader, "id", true);
            int noLabel = FindColumnNo(rangeHeader, "label", false);
            int noDefault = FindColumnNo(rangeHeader, "default", false);
            int noOtherAnnotation = FindColumnNo(rangeHeader, "other_annotation", false);
            int noPreprocess = FindColumnNo(rangeHeader, "preprocess", false);

            List<string> groups = new List<string>();
            string currentGroup = string.Empty;
            int order = 0;

            for (int idxRow = groupRowIndex + 1; idxRow <= range.Rows.Count; ++idxRow)
            {
                ParamCodeData data = new ParamCodeData();

                {
                    string groupId = ((Excel.Range)range.Cells[idxRow, noGroupId]).Text.ToString();

                    if (groupId != string.Empty && groupId != currentGroup)
                    {
                        groups.Add(groupId);
                        currentGroup = groupId;
                        order = 0;
                    }
                    // group が省略されている場合には、上のセルで指定されている値を引き続き用いる。

                    data.Group = currentGroup;
                    data.Order = order;
                    order += 10;
                }

                {
                    string type = ((Excel.Range)range.Cells[idxRow, noType]).Text.ToString();
                    if (type == "renderinfo")
                    {
                        data.Type = ParamCodeData.ParamType.RenderInfo;
                    }
                    else if (type == "option")
                    {
                        data.Type = ParamCodeData.ParamType.Option;
                    }
                    else if (type == "sampler")
                    {
                        data.Type = ParamCodeData.ParamType.Sampler;
                    }
                    else if (type == "ub_material")
                    {
                        data.Type = ParamCodeData.ParamType.UniformBlockMaterial;
                    }
                    else if (type == "group")
                    {
                        data.Type = ParamCodeData.ParamType.Group;
                    }
                    else if (type == string.Empty)
                    {
                        continue;
                    }
                    else
                    {
                        throw new InvalidOperationException(
                            string.Format(MakeParamCodeStrings.ErrorWrongType, pageSheet.Name, type));
                    }
                }

                data.Variable = ((Excel.Range)range.Cells[idxRow, noVariable]).Text.ToString();
                data.Id = ((Excel.Range)range.Cells[idxRow, noId]).Text.ToString();
                data.Label = (noLabel >= 0 ? ((Excel.Range)range.Cells[idxRow, noLabel]).Text.ToString() : string.Empty);
                data.Default = (noDefault >= 0 ? ((Excel.Range)range.Cells[idxRow, noDefault]).Text.ToString() : string.Empty);
                data.OtherAnnotation = (noOtherAnnotation >= 0 ? ((Excel.Range)range.Cells[idxRow, noOtherAnnotation]).Text.ToString() : string.Empty);
                data.Preprocess = (noPreprocess >= 0 ? ((Excel.Range)range.Cells[idxRow, noPreprocess]).Text.ToString() : string.Empty);

                _dataList.Add(data);
            }

            // シートにあるグループを現在のページとして設定する。
            foreach (string group in groups)
            {
                GroupData groupData = _groupList.FirstOrDefault(x => x.Id == group);
                if (groupData == null)
                {
                    groupData = new GroupData()
                    {
                        Id = group,
                        Label = string.Empty,
                        Condition = string.Empty,
                        Page = pageSheet.Name,
                    };
                    _groupList.Add(groupData);
                }
                else
                {
                    if (groupData.Page != string.Empty && groupData.Page != pageSheet.Name)
                    {
                        throw new InvalidOperationException(
                            string.Format(MakeParamCodeStrings.ErrorGroupDuplicated, pageSheet.Name, group, groupData.Page));
                    }

                    groupData.Page = pageSheet.Name;
                }
            }
        }

        private bool GenerateCode()
        {
            string strFile = string.Empty;

            string headerFile = "\r\n// Automatically generated by 3dExcelToShaderAnnotation. DO NOT edit.\r\n\r\n";
            strFile += headerFile;

            strFile += this.MakePagesSection();

            strFile += this.MakeGroupsSection();

            strFile += this.MakeRenderInfoSection();

            strFile += this.MakeOptionsSection();

            strFile += this.MakeSamplersSection();

            strFile += this.MakeUniformBlockMaterialsSection();

            _output = strFile;
            return true;
        }

        private string MakePagesSection()
        {
            StringBuilder builder = new StringBuilder();

            if (_pageList.Count > 0)
            {
                int maxIdLen = 0;
                _pageList.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
                int maxLabelLen = 0;
                _pageList.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });

                builder.Append(
                    "//------------------------------------------------------------------------------\r\n" +
                    "// pages\r\n" +
                    "//------------------------------------------------------------------------------\r\n\r\n");

                foreach (var page in _pageList)
                {
                    string str = string.Format("// @@ page={0}", GetPaddedAnnotation(page.Id, maxIdLen));

                    if (!string.IsNullOrEmpty(page.Label))
                    {
                        str += string.Format(" label={0}", GetPaddedAnnotation(page.Label, maxLabelLen));
                    }
                    str += string.Format(" order={0}", AddQuotations(_pageList.IndexOf(page).ToString()));
                    str += "\r\n";

                    builder.Append(str);
                }

                builder.Append("\r\n\r\n");
            }

            return builder.ToString();
        }

        private string MakeGroupsSection()
        {
            StringBuilder builder = new StringBuilder();

            if (_groupList.Count == 0)
            {
                return string.Empty;
            }

            int maxIdLen = 0;
            _groupList.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
            int maxPageLen = 0;
            _groupList.ForEach(x => { if (maxPageLen < LenAnnotation(x.Page)) { maxPageLen = LenAnnotation(x.Page); } });
            int maxLabelLen = 0;
            _groupList.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });

            builder.Append(
                "//------------------------------------------------------------------------------\r\n" +
                "// groups\r\n" +
                "//------------------------------------------------------------------------------\r\n\r\n");

            var groupDatas = _dataList.FindAll(x => x.Type == ParamCodeData.ParamType.Group);
            int maxGroupLen = 0;
            groupDatas.ForEach(x => { if (maxGroupLen < LenAnnotation(x.Group)) { maxGroupLen = LenAnnotation(x.Group); } });
            foreach (var group in _groupList)
            {
                string str = string.Format("// @@ group={0}", GetPaddedAnnotation(group.Id, maxIdLen));

                if (!string.IsNullOrEmpty(group.Label))
                {
                    str += string.Format(" label={0}", GetPaddedAnnotation(group.Label, maxLabelLen));
                }
                if (!string.IsNullOrEmpty(group.Page))
                {
                    str += string.Format(" page={0}", GetPaddedAnnotation(group.Page, maxPageLen));
                }
                if (!string.IsNullOrEmpty(group.Condition))
                {
                    str += string.Format(" condition={0}", AddQuotations(group.Condition));
                }
                str += string.Format(" order={0}", AddQuotations(_groupList.IndexOf(group).ToString()));

                ParamCodeData groupParamDef = _dataList.FirstOrDefault(x => x.Id == group.Id && (x.Type == ParamCodeData.ParamType.Group));
                if (groupParamDef != null && !string.IsNullOrEmpty(groupParamDef.Group))
                {
                    str += string.Format(" group={0}", GetPaddedAnnotation(groupParamDef.Group, maxGroupLen));
                }

                str += "\r\n";

                builder.Append(str);
            }

            builder.Append("\r\n\r\n");

            return builder.ToString();
        }

        private string MakeRenderInfoSection()
        {
            StringBuilder builder = new StringBuilder();

            if (!_dataList.Any(x => x.Type == ParamCodeData.ParamType.RenderInfo))
            {
                return string.Empty;
            }

            var renderInfo = _dataList.FindAll(x => x.Type == ParamCodeData.ParamType.RenderInfo);

            int maxIdLen = 0;
            renderInfo.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
            int maxGroupLen = 0;
            renderInfo.ForEach(x => { if (maxGroupLen < LenAnnotation(x.Group)) { maxGroupLen = LenAnnotation(x.Group); } });
            int maxLabelLen = 0;
            renderInfo.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });
            int maxDefaultLen = 0;
            renderInfo.ForEach(x => { if (maxDefaultLen < LenAnnotation(x.Default)) { maxDefaultLen = LenAnnotation(x.Default); } });

            builder.Append(
                "//------------------------------------------------------------------------------\r\n" +
                "// renderinfo\r\n" +
                "//------------------------------------------------------------------------------\r\n\r\n");

            foreach (var data in renderInfo)
            {
                builder.Append(
                    MakeAnnotation(data, "renderinfo", maxIdLen, maxGroupLen, maxLabelLen, maxDefaultLen));
            }

            builder.Append("\r\n\r\n");

            return builder.ToString();
        }

        private string MakeOptionsSection()
        {
            StringBuilder builder = new StringBuilder();

            if (!_dataList.Any(x => x.Type == ParamCodeData.ParamType.Option))
            {
                return string.Empty;
            }

            var shaderOptions = _dataList.FindAll(x => x.Type == ParamCodeData.ParamType.Option);

            int maxMacroLen = 0;
            int maxValueLen = 0;
            foreach (var data in shaderOptions)
            {
                string macro = string.Empty;
                string value = string.Empty;

                int splitPos = data.Variable.IndexOf(" ");
                if (splitPos >= 0)
                {
                    macro = data.Variable.Substring(0, splitPos).Trim();
                    value = data.Variable.Substring(splitPos).Trim();
                }

                if (string.IsNullOrWhiteSpace(macro) || string.IsNullOrWhiteSpace(value))
                {
                    throw new InvalidOperationException(
                        string.Format(MakeParamCodeStrings.ErrorWrongOptionVariable, data.Variable));
                }

                if (maxMacroLen < macro.Length)
                {
                    maxMacroLen = macro.Length;
                }

                if (maxValueLen < value.Length)
                {
                    maxValueLen = value.Length;
                }
            }

            int maxIdLen = 0;
            shaderOptions.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
            int maxGroupLen = 0;
            shaderOptions.ForEach(x => { if (maxGroupLen < LenAnnotation(x.Group)) { maxGroupLen = LenAnnotation(x.Group); } });
            int maxLabelLen = 0;
            shaderOptions.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });
            int maxDefaultLen = 0;
            shaderOptions.ForEach(x => { if (maxDefaultLen < LenAnnotation(x.Default)) { maxDefaultLen = LenAnnotation(x.Default); } });

            builder.Append(
                "//------------------------------------------------------------------------------\r\n" +
                "// shader options\r\n" +
                "//------------------------------------------------------------------------------\r\n\r\n");

            List<ParamCodeData> preprocessData = shaderOptions.FindAll(x => !string.IsNullOrEmpty(x.Preprocess));
            while (preprocessData.Count > 0)
            {
                ParamCodeData data = preprocessData[0];
                string condition = data.Preprocess;

                List<ParamCodeData> processed = new List<ParamCodeData>();
                List<string> dataIf = new List<string>();
                List<string> dataElse = new List<string>();

                for (int idx = 0; idx < preprocessData.Count; ++idx)
                {
                    data = preprocessData[idx];
                    if (data.Preprocess != condition)
                    {
                        continue;
                    }

                    processed.Add(data);

                    {
                        int splitPos = data.Variable.IndexOf(" ");
                        string macro = data.Variable.Substring(0, splitPos).Trim();
                        string value = data.Variable.Substring(splitPos).Trim();

                        string str = string.Format("#define {0} {1} ",
                            GetPaddedString(macro, maxMacroLen),
                            GetPaddedString(value, maxValueLen));
                        str += MakeAnnotation(data, "id", maxIdLen, maxGroupLen, maxLabelLen, maxDefaultLen);
                        dataIf.Add(str);
                    }

                    if (idx + 1 < preprocessData.Count && preprocessData[idx + 1].Preprocess == "else")
                    {
                        int splitPos = preprocessData[idx + 1].Variable.IndexOf(" ");
                        string macro = preprocessData[idx + 1].Variable.Substring(0, splitPos).Trim();
                        string value = preprocessData[idx + 1].Variable.Substring(splitPos).Trim();

                        string str = string.Format("#define {0} {1} ",
                            GetPaddedString(macro, maxMacroLen),
                            GetPaddedString(value, maxValueLen));
                        str += MakeAnnotation(preprocessData[idx + 1], "id", maxIdLen, maxGroupLen, maxLabelLen, maxDefaultLen);
                        dataElse.Add(str);
                        processed.Add(preprocessData[idx + 1]);
                    }
                }
                processed.ForEach(x => preprocessData.Remove(x));

                {
                    string str = string.Format("#if {0}\r\n", condition);
                    dataIf.ForEach(x => str += x);
                    if (dataElse.Count > 0)
                    {
                        str += "#else\r\n";
                        dataElse.ForEach(x => str += x);
                    }

                    str += "#endif\r\n";
                    builder.Append(str);
                }
            }

            foreach (var data in shaderOptions)
            {
                if (!string.IsNullOrEmpty(data.Preprocess))
                {
                    continue;
                }

                int splitPos = data.Variable.IndexOf(" ");
                string macro = data.Variable.Substring(0, splitPos).Trim();
                string value = data.Variable.Substring(splitPos).Trim();

                builder.Append(
                    string.Format("#define {0} {1} ",
                        GetPaddedString(macro, maxMacroLen),
                        GetPaddedString(value, maxValueLen)));
                builder.Append(
                    MakeAnnotation(data, "id", maxIdLen, maxGroupLen, maxLabelLen, maxDefaultLen));
            }

            builder.Append("\r\n");

            {
                List<string> optionBlock = new List<string>();
                foreach (var data in shaderOptions)
                {
                    // 列挙型 choice では option block を作成しない。
                    bool isStringChoice = false;
                    Regex reg = new Regex("choice.*?=.*?\"(?<choices>.*?)\"");
                    {
                        Match m = reg.Match(data.OtherAnnotation);
                        if (m.Success)
                        {
                            string choices = m.Groups["choices"].Value.Trim();

                            if (!choices.Contains("[") && choices != "bool")
                            {
                                int parseResult;
                                string[] choicesAndLabels = choices.Split(',');
                                foreach (string choiceAndLabel in choicesAndLabels)
                                {
                                    string[] splitChoice = choiceAndLabel.Split(':');
                                    if (!int.TryParse(splitChoice[0], out parseResult))
                                    {
                                        isStringChoice = true;
                                        break;
                                    }
                                }
                            }

                            m = m.NextMatch();
                            Nintendo.ToolFoundation.Contracts.Assertion.Operation.False(m.Success);
                        }
                    }

                    if (!isStringChoice && !string.IsNullOrEmpty(data.Id))
                    {
                        string str = string.Format("    int {0} // @@ id={1}\r\n", GetPaddedString(data.Id + ";", maxIdLen + 1), AddQuotations(data.Id));
                        if (!optionBlock.Contains(str))
                        {
                            optionBlock.Add(str);
                        }
                    }
                }

                if (optionBlock.Count > 0)
                {
                    string str = string.Format("{0}  // @@ id={1} type=\"option\"\r\n", _shaderOptionUB, AddQuotations(_shaderOptionUBId));
                    str += "{\r\n";
                    optionBlock.ForEach(x => str += x);
                    str += "};\r\n";
                    builder.Append(str);
                }
            }

            builder.Append("\r\n\r\n");

            return builder.ToString();
        }

        private string MakeSamplersSection()
        {
            StringBuilder builder = new StringBuilder();

            if (!_dataList.Any(x => x.Type == ParamCodeData.ParamType.Sampler))
            {
                return string.Empty;
            }

            var samplers = _dataList.FindAll(x => x.Type == ParamCodeData.ParamType.Sampler);

            int maxValueLen = 0;
            foreach (var data in samplers)
            {
                string[] variables = MakeVariables(data.Variable);

                if (variables.Length != 1)
                {
                    throw new InvalidOperationException(
                        string.Format(MakeParamCodeStrings.ErrorWrongVariable, data.Variable));
                }

                if (maxValueLen < LenAnnotation(variables[0]))
                {
                    maxValueLen = LenAnnotation(variables[0]);
                }
            }

            int maxIdLen = 0;
            samplers.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
            int maxGroupLen = 0;
            samplers.ForEach(x => { if (maxGroupLen < LenAnnotation(x.Group)) { maxGroupLen = LenAnnotation(x.Group); } });
            int maxLabelLen = 0;
            samplers.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });

            builder.Append(
                "//------------------------------------------------------------------------------\r\n" +
                "// samplers\r\n" +
                "//------------------------------------------------------------------------------\r\n\r\n");

            {
                List<ParamCodeData> preprocessData = samplers.FindAll(x => !string.IsNullOrEmpty(x.Preprocess));
                while (preprocessData.Count > 0)
                {
                    ParamCodeData data = preprocessData[0];
                    string condition = data.Preprocess;

                    List<ParamCodeData> processed = new List<ParamCodeData>();
                    List<string> dataIf = new List<string>();
                    List<string> dataElse = new List<string>();

                    for (int idx = 0; idx < preprocessData.Count; ++idx)
                    {
                        data = preprocessData[idx];
                        if (data.Preprocess != condition)
                        {
                            continue;
                        }

                        processed.Add(data);
                        string[] variables = MakeVariables(data.Variable);
                        dataIf.Add(
                            string.Format("uniform {0} // @@ id={1}\r\n", GetPaddedString(variables[0], maxValueLen), AddQuotations(data.Id)));

                        if (idx + 1 < preprocessData.Count && preprocessData[idx + 1].Preprocess == "else")
                        {
                            dataElse.Add(
                                string.Format("uniform {0} // @@ id={1}\r\n", GetPaddedString(variables[0], maxValueLen), AddQuotations(data.Id)));
                            processed.Add(preprocessData[idx + 1]);
                        }
                    }
                    processed.ForEach(x => preprocessData.Remove(x));

                    string str = string.Format("#if {0}\r\n", condition);
                    dataIf.ForEach(x => str += x);
                    if (dataElse.Count > 0)
                    {
                        str += "#else\r\n";
                        dataElse.ForEach(x => str += x);
                    }

                    str += "#endif\r\n";
                    builder.Append(str);
                }
            }

            foreach (var data in samplers)
            {
                if (!string.IsNullOrEmpty(data.Preprocess))
                {
                    continue;
                }

                string[] variables = MakeVariables(data.Variable);
                builder.Append(
                    string.Format("uniform {0} // @@ id={1}", GetPaddedString(variables[0], maxValueLen), AddQuotations(data.Id)));
                builder.Append("\r\n");
            }

            builder.Append("\r\n");

            {
                List<ParamCodeData> preprocessData = samplers.FindAll(x => !string.IsNullOrEmpty(x.Preprocess));
                while (preprocessData.Count > 0)
                {
                    ParamCodeData data = preprocessData[0];
                    string condition = data.Preprocess;

                    List<ParamCodeData> processed = new List<ParamCodeData>();
                    List<string> dataIf = new List<string>();
                    List<string> dataElse = new List<string>();

                    for (int idx = 0; idx < preprocessData.Count; ++idx)
                    {
                        data = preprocessData[idx];
                        if (data.Preprocess != condition)
                        {
                            continue;
                        }

                        processed.Add(data);
                        dataIf.Add(MakeAnnotation(data, "sampler_id", maxIdLen, maxGroupLen, maxLabelLen, 0));

                        if (idx + 1 < preprocessData.Count && preprocessData[idx + 1].Preprocess == "else")
                        {
                            dataElse.Add(MakeAnnotation(data, "sampler_id", maxIdLen, maxGroupLen, maxLabelLen, 0));
                            processed.Add(preprocessData[idx + 1]);
                        }
                    }
                    processed.ForEach(x => preprocessData.Remove(x));

                    string str = string.Format("#if {0}\r\n", condition);
                    dataIf.ForEach(x => str += x);
                    if (dataElse.Count > 0)
                    {
                        str += "#else\r\n";
                        dataElse.ForEach(x => str += x);
                    }

                    str += "#endif\r\n";
                    builder.Append(str);
                }
            }

            foreach (var data in samplers)
            {
                if (!string.IsNullOrEmpty(data.Preprocess))
                {
                    continue;
                }

                builder.Append(
                    MakeAnnotation(data, "sampler_id", maxIdLen, maxGroupLen, maxLabelLen, 0));
            }

            builder.Append("\r\n\r\n");

            return builder.ToString();
        }

        private string MakeUniformBlockMaterialsSection()
        {
            StringBuilder builder = new StringBuilder();

            if (!_dataList.Any(x => x.Type == ParamCodeData.ParamType.UniformBlockMaterial))
            {
                return string.Empty;
            }

            var ubMaterials = _dataList.FindAll(x => x.Type == ParamCodeData.ParamType.UniformBlockMaterial);

            int maxValueLen = 0;
            foreach (var data in ubMaterials)
            {
                string[] variables = MakeVariables(data.Variable);
                foreach (string variable in variables)
                {
                    if (maxValueLen < LenAnnotation(variable))
                    {
                        maxValueLen = LenAnnotation(variable);
                    }
                }
            }

            int maxIdLen = 0;
            ubMaterials.ForEach(x => { if (maxIdLen < LenAnnotation(x.Id)) { maxIdLen = LenAnnotation(x.Id); } });
            int maxGroupLen = 0;
            ubMaterials.ForEach(x => { if (maxGroupLen < LenAnnotation(x.Group)) { maxGroupLen = LenAnnotation(x.Group); } });
            int maxLabelLen = 0;
            ubMaterials.ForEach(x => { if (maxLabelLen < LenAnnotation(x.Label)) { maxLabelLen = LenAnnotation(x.Label); } });
            int maxDefaultLen = 0;
            ubMaterials.ForEach(x => { if (maxDefaultLen < LenAnnotation(x.Default)) { maxDefaultLen = LenAnnotation(x.Default); } });

            builder.Append(
                "//------------------------------------------------------------------------------\r\n" +
                "// uniform block for material\r\n" +
                "//------------------------------------------------------------------------------\r\n\r\n");

            {
                string str = string.Format("{0}  // @@ id={1} type=\"material\"", _materialUB, AddQuotations(_materialUBId));
                str += "\r\n{\r\n";

                for (int idx = 0; idx < ubMaterials.Count; ++idx)
                {
                    var data = ubMaterials[idx];
                    bool hasPreprocess = false;

                    if (!string.IsNullOrEmpty(data.Preprocess))
                    {
                        hasPreprocess = true;

                        if (data.Preprocess != "else")
                        {
                            str += string.Format("#if {0}\r\n", data.Preprocess);
                        }
                        else
                        {
                            str += "#else\r\n";
                        }
                    }

                    string[] variables = MakeVariables(data.Variable);
                    for (int i = 0; i < variables.Length; ++i)
                    {
                        if (i == 0 && !string.IsNullOrEmpty(data.Id))
                        {
                            str += string.Format("    {0} // @@ id={1}", GetPaddedString(variables[0], maxValueLen), AddQuotations(data.Id));
                        }
                        else
                        {
                            str += string.Format("    {0}", GetPaddedString(variables[i], maxValueLen));
                        }

                        str += "\r\n";
                    }

                    if (hasPreprocess &&
                        !(idx + 1 < ubMaterials.Count && ubMaterials[idx + 1].Preprocess == "else"))
                    {
                        str += "#endif\r\n";
                    }
                }

                str += "};\r\n";
                builder.Append(str);
            }

            builder.Append("\r\n");

            for (int idx = 0; idx < ubMaterials.Count; ++idx)
            {
                var data = ubMaterials[idx];
                bool hasPreprocess = false;

                if (!string.IsNullOrEmpty(data.Preprocess))
                {
                    hasPreprocess = true;

                    if (data.Preprocess != "else")
                    {
                        builder.Append(string.Format("#if {0}\r\n", data.Preprocess));
                    }
                    else
                    {
                        builder.Append("#else\r\n");
                    }
                }

                if (!string.IsNullOrEmpty(data.Id))
                {
                    builder.Append(
                        MakeAnnotation(data, "uniform_id", maxIdLen, maxGroupLen, maxLabelLen, maxDefaultLen));
                }

                if (hasPreprocess &&
                    !(idx + 1 < ubMaterials.Count && ubMaterials[idx + 1].Preprocess == "else"))
                {
                    builder.Append("#endif\r\n");
                }
            }

            builder.Append("\r\n\r\n");

            return builder.ToString();
        }

        private static string MakeAnnotation(ParamCodeData data, string Id, int maxIdLen, int maxGroupLen, int maxLabelLen, int maxDefaultLen)
        {
            if (string.IsNullOrEmpty(data.Id))
            {
                return "\r\n";
            }

            // TODO: OtherAnnotation に Id、Label、Default が指定されている場合に警告をだす。

            string str = string.Format("// @@ {0}={1}", Id, GetPaddedAnnotation(data.Id, maxIdLen));

            if (!string.IsNullOrEmpty(data.Group))
            {
                str += string.Format(" group={0}", GetPaddedAnnotation(data.Group, maxGroupLen));
            }
            if (!string.IsNullOrEmpty(data.Label))
            {
                str += string.Format(" label={0}", GetPaddedAnnotation(data.Label, maxLabelLen));
            }
            if (!string.IsNullOrEmpty(data.Default))
            {
                str += string.Format(" default={0}", GetPaddedAnnotation(data.Default, maxDefaultLen));
            }
            if (!string.IsNullOrEmpty(data.OtherAnnotation))
            {
                str += string.Format(" {0}", data.OtherAnnotation);
            }
            str += string.Format(" order={0}", AddQuotations(data.Order.ToString()));
            str += "\r\n";

            return str;
        }

        private static string[] MakeVariables(string str)
        {
            Nintendo.ToolFoundation.Contracts.Assertion.Argument.NotNull(str);

            List<string> result = new List<string>();
            string[] variables = str.Split(';');

            for (int i = 0; i < variables.Length; ++i)
            {
                variables[i] = variables[i].Trim();
                if (variables[i] != string.Empty)
                {
                    result.Add(variables[i] + ";");
                }
            }

            return result.ToArray();
        }

        private static int LenAnnotation(string str)
        {
            if (string.IsNullOrEmpty(str))
            {
                return 0;
            }
            return System.Text.Encoding.GetEncoding("UTF-8").GetByteCount(str);
        }

        private static string AddQuotations(string str)
        {
            if (str == null)
            {
                return null;
            }
            return "\"" + str + "\"";
        }

        private static string GetPaddedString(string str, int len)
        {
            if (str == null)
            {
                return null;
            }
            int strLen = LenAnnotation(str);
            for (int i = 0; i < len - strLen; ++i)
            {
                str += " ";
            }
            return str;
        }

        private static string GetPaddedAnnotation(string str, int len)
        {
            if (str == null)
            {
                return null;
            }
            return GetPaddedString(AddQuotations(str), len + 2);
        }

        private static int FindColumnNo(Excel.Range range, string header, bool isNecessary)
        {
            Excel.Range cell = range.Find(header, Type.Missing, Type.Missing, Excel.XlLookAt.xlWhole);
            if (cell == null)
            {
                if (isNecessary)
                {
                    throw new Exception($"Can not find header of \"{header}\"");
                }
                return -1;
            }
            return cell.Column;
        }

        ////---------------------------------------------------------------------

        public class Result
        {
            public bool IsSuccessful { get; set; }
            public string ErrorMessage { get; set; }

            public Result()
            {
                this.IsSuccessful = false;
                this.ErrorMessage = string.Empty;
            }
        }

        private class PageData
        {
            public string Id { get; set; }
            public string Label { get; set; }
        }

        private class GroupData
        {
            public string Id { get; set; }
            public string Label { get; set; }
            public string Condition { get; set; }
            public string Page { get; set; }
        }

        private class ParamCodeData
        {
            public enum ParamType
            {
                RenderInfo,
                Option,
                Sampler,
                UniformBlockMaterial,
                Group,
            }

            public string Group { get; set; }
            public ParamType Type { get; set; }
            public string Variable { get; set; }
            public string Id { get; set; }
            public string Label { get; set; }
            public string Default { get; set; }
            public string OtherAnnotation { get; set; }
            public string Preprocess { get; set; }
            public int Order { get; set; }
        }
    }
}
