﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// TextureEncoder

//=============================================================================
// include
//=============================================================================
#include "Common.h"
#include "DccLib.h"

// テクスチャコンバータのヘッダを利用します。
#include <Encoder/Encoder.h>

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//=============================================================================
//! @brief テクスチャエンコーダのクラスです。
//=============================================================================
class TextureEncoder
{
protected:
    std::string m_DllPath; //!< エンコーダの DLL ファイルのパスです。
    HMODULE m_hDll; //!< エンコーダの DLL のインスタンスハンドルです。

public:
    gfx::tool::texenc::ConvertFormatFunction ConvertFormat; //!< 画像のフォーマットを変換する関数です。

public:
    //! @biref コンストラクタです。
    TextureEncoder()
    : m_hDll(nullptr),
      ConvertFormat(nullptr)
    {
    }

    //! @biref デストラクタです。
    ~TextureEncoder()
    {
        Finalize();
    }

    //! @biref エンコーダの DLL ファイルのパスを設定します。
    //!
    //! @param[in] dllPath エンコーダの DLL ファイルのパスです。
    //!
    void SetDllPath(const char* dllPath)
    {
        m_DllPath = dllPath;
    }

    //! @biref エンコーダの DLL ファイルのパスを返します。
    std::string GetDllPath() const
    {
        return m_DllPath;
    }

    //! @biref 初期化します。
    //!
    //! @param[in] dllPath エンコーダの DLL ファイルのパスです。
    //!                    nullptr なら SetDllPath で設定したパスを使用します。
    //!
    //! @return 処理結果を返します。
    //!
    gfx::tool::dcc::RStatus Initialize(const char* dllPath);

    //! @biref 終了します。
    void Finalize();

    //! @biref 初期化済みなら true を返します。
    bool IsInitialized() const
    {
        return (m_hDll != nullptr);
    }
};

//=============================================================================
// テクスチャ関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのテクスチャ要素から画像ファイルを作成します。
//!
//! @param[out] pCtexFormat ctex フォーマットを格納します。
//! @param[out] pMipLevel ミップマップのレベル数を格納します。
//! @param[in,out] pEncoder テクスチャエンコーダへのポインタです。
//! @param[in] dstPath 画像ファイル（現在は DDS のみ対応）のパスです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] texName テクスチャ名です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
gfx::tool::dcc::RStatus CreateImageFileFromCtex(
    std::string* pCtexFormat,
    int* pMipLevel,
    TextureEncoder* pEncoder,
    const std::string& dstPath,
    const CIntermediateFile& inputFile,
    const std::string& texName
);

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

