﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// グローバル関数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief RVec2 型の属性を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 属性のインデントに必要なタブの数です。
//! @param[in] name 属性名です。
//! @param[in] vec 属性の値です。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec2& vec
);

//-----------------------------------------------------------------------------
//! @brief RVec3 型の属性を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 属性のインデントに必要なタブの数です。
//! @param[in] name 属性名です。
//! @param[in] vec 属性の値です。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec3& vec
);

//-----------------------------------------------------------------------------
//! @brief RVec4 型の属性を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 属性のインデントに必要なタブの数です。
//! @param[in] name 属性名です。
//! @param[in] vec 属性の値です。
//-----------------------------------------------------------------------------
void ROutVecAttr(
    std::ostream& os,
    const int tabCount,
    const char* name,
    const RVec4& vec
);

//-----------------------------------------------------------------------------
//! @brief 配列形式の要素を出力します。テンプレート関数です。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 要素のインデントに必要なタブの数です。
//! @param[in] array 配列です。
//! @param[in] name 配列の要素名です。
//-----------------------------------------------------------------------------
template <typename T>
void ROutArrayElement(
    std::ostream& os,
    const int tabCount,
    const std::vector<T>& array,
    const char* name
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount > 0)
    {
        const int& tc = tabCount;
        os << RTab(tc) << "<" << name << " length=\"" << valCount << "\">" << R_ENDL;
        for (int ival = 0; ival < valCount; ++ival)
        {
            array[ival].Out(os, tc + 1, ival);
        }
        os << RTab(tc) << "</" << name << ">" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief 変換（スケール・回転・移動）の属性を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 属性のインデントに必要なタブの数です。
//! @param[in] s スケールです。
//! @param[in] r 回転です。
//! @param[in] t 移動です。
//-----------------------------------------------------------------------------
void ROutTransform(
    std::ostream& os,
    const int tabCount,
    const RVec3& s,
    const RVec3& r,
    const RVec3& t
);

//-----------------------------------------------------------------------------
//! @brief 整数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RIntArray& array,
    const int column
);

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RFloatArray& array,
    const int column
);

//-----------------------------------------------------------------------------
//! @brief バイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
);

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

