﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// RRenderState RDepthTest RAlphaTest RColorBlend RLogicalBlend
// RSampler ROriginalTexsrt
// RStencilTest
// RMaterial ROriginalColor

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
//! @brief デプステストのクラスです。
//=============================================================================
class RDepthTest
{
public:
    //! @brief デプステストの関数を表す列挙型です。
    enum Func
    {
        NEVER, LESS, EQUAL, LEQUAL, GREATER, NEQUAL, GEQUAL, ALWAYS
    };

    bool m_Enable; //!< デプステストが有効なら true です。
    bool m_Write; //!< デプスバッファに書き込むなら true です。
    Func m_Func; //!< デプステストの関数です。

public:
    //! コンストラクタです。
    RDepthTest()
    : m_Enable(true),
      m_Write(true),
      m_Func(LEQUAL)
    {
    }

    //! 出力します。
    void Out(std::ostream& os, const int tabCount) const;

    //! 同値であれば true を返します。
    bool operator==(const RDepthTest& rhs) const
    {
        return (
            m_Enable == rhs.m_Enable &&
            m_Write  == rhs.m_Write  &&
            m_Func   == rhs.m_Func);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RDepthTest& rhs) const
    {
        return !(*this == rhs);
    }
};

//=============================================================================
//! @brief アルファテストのクラスです。
//=============================================================================
class RAlphaTest
{
public:
    //! @brief アルファテストの関数を表す列挙型です。
    enum Func
    {
        NEVER, LESS, EQUAL, LEQUAL, GREATER, NEQUAL, GEQUAL, ALWAYS
    };

    bool m_Enable; //!< アルファテストが有効なら true です。
    Func m_Func; //!< アルファテストの関数です。
    float m_Value; //!< アルファテストの比較値です。

public:
    //! コンストラクタです。
    RAlphaTest()
    : m_Enable(false),
      m_Func(GEQUAL),
      m_Value(0.5f)
    {
    }

    //! 出力します。
    void Out(std::ostream& os, const int tabCount) const;

    //! 同値であれば true を返します。
    bool operator==(const RAlphaTest& rhs) const
    {
        return (
            m_Enable == rhs.m_Enable &&
            m_Func   == rhs.m_Func   &&
            m_Value  == rhs.m_Value);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RAlphaTest& rhs) const
    {
        return !(*this == rhs);
    }
};

//=============================================================================
//! @brief カラーブレンドのクラスです。
//=============================================================================
class RColorBlend
{
public:
    //! @brief ブレンド制御のカラーを表す列挙型です。
    enum Func
    {
        ZERO, ONE,
        SRC_COLOR, ONE_MINUS_SRC_COLOR,
        SRC_ALPHA, ONE_MINUS_SRC_ALPHA,
        DST_COLOR, ONE_MINUS_DST_COLOR,
        DST_ALPHA, ONE_MINUS_DST_ALPHA,
        CONST_COLOR, ONE_MINUS_CONST_COLOR,
        CONST_ALPHA, ONE_MINUS_CONST_ALPHA,
        SRC1_COLOR, ONE_MINUS_SRC1_COLOR,
        SRC1_ALPHA, ONE_MINUS_SRC1_ALPHA,
        SRC_ALPHA_SATURATE
    };

    //! @brief RGBA ブレンドのオペレーションを表す列挙型です。
    enum Op { ADD, SRC_MINUS_DST, MIN, MAX, DST_MINUS_SRC };

    Func m_RgbSrcFunc; //!< RGB ブレンドのソースカラーです。
    Func m_RgbDstFunc; //!< RGB ブレンドのデスティネーションカラーです。
    Op m_RgbOp; //!< RGB ブレンドのオペレーションです。
    Func m_AlphaSrcFunc; //!< アルファブレンドのソースカラーです。
    Func m_AlphaDstFunc; //!< アルファブレンドのデスティネーションカラーです。
    Op m_AlphaOp; //!< アルファブレンドのオペレーションです。
    RVec4 m_ConstColor; //!< カラーブレンドに利用する定数カラーです。

public:
    //! コンストラクタです。
    RColorBlend()
    : m_RgbSrcFunc(SRC_ALPHA), m_RgbDstFunc(ONE_MINUS_SRC_ALPHA),
      m_RgbOp(ADD),
      m_AlphaSrcFunc(ONE), m_AlphaDstFunc(ZERO),
      m_AlphaOp(ADD),
      m_ConstColor(RVec4::kZero)
    {
    }

    //! 出力します。
    void Out(std::ostream& os, const int tabCount) const;

    //! 同値であれば true を返します。
    bool operator==(const RColorBlend& rhs) const
    {
        return (
            m_RgbSrcFunc   == rhs.m_RgbSrcFunc   &&
            m_RgbDstFunc   == rhs.m_RgbDstFunc   &&
            m_RgbOp        == rhs.m_RgbOp        &&
            m_AlphaSrcFunc == rhs.m_AlphaSrcFunc &&
            m_AlphaDstFunc == rhs.m_AlphaDstFunc &&
            m_AlphaOp      == rhs.m_AlphaOp      &&
            m_ConstColor   == rhs.m_ConstColor);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RColorBlend& rhs) const
    {
        return !(*this == rhs);
    }
};

//=============================================================================
//! @brief 論理ブレンドのクラスです。
//=============================================================================
class RLogicalBlend
{
public:
    //! @brief 論理ブレンドのオペレーションを表す列挙型です。
    enum Op
    {
        CLEAR, SET, COPY, INV_COPY, NO_OP, INV,
        AND, NAND, OR, NOR, XOR, EQUIV,
        REV_AND, INV_AND, REV_OR, INV_OR
    };

    Op m_Op; //!< 論理ブレンドのオペレーションです。

public:
    //! コンストラクタです。
    RLogicalBlend()
    : m_Op(COPY)
    {
    }

    //! 出力します。
    void Out(std::ostream& os, const int tabCount) const;

    //! 同値であれば true を返します。
    bool operator==(const RLogicalBlend& rhs) const
    {
        return (
            m_Op == rhs.m_Op);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RLogicalBlend& rhs) const
    {
        return !(*this == rhs);
    }
};

//=============================================================================
//! @brief レンダーステートのクラスです。
//=============================================================================
class RRenderState
{
public:
    //! @brief レンダーステートのモードを表す列挙型です。
    //         OPAQUE は WinGDI.h で定義されているので OPA にしています。
    enum Mode
    {
        CUSTOM, //!< 既定のモードでない、カスタムモードです。
        OPA,    //!< 不透明モードです。
        MASK,   //!< 抜きモードです。
        XLU     //!< 半透明モードです。
    };

    //! @brief 表示面を表す列挙型です。
    enum DisplayFace { BOTH, FRONT, BACK, FACE_NONE };

    //! @brief ブレンド制御のモードを表す列挙型です。
    enum BlendMode { BLEND_NONE, COLOR, LOGIC };

    Mode m_Mode; //!< レンダーステートのモードです。
    DisplayFace m_DisplayFace; //!< 表示面です。
    BlendMode m_BlendMode; //!< ブレンド制御のモードです。

    RDepthTest m_DepthTest; //!< デプステストです。
    RAlphaTest m_AlphaTest; //!< アルファテストです。
    RColorBlend m_ColorBlend; //!< カラーブレンドです。
    RLogicalBlend m_LogicalBlend; //!< 論理ブレンドです。

public:
    //! コンストラクタです。
    RRenderState()
    : m_Mode(OPA),
      m_DisplayFace(FRONT),
      m_BlendMode(BLEND_NONE)
    {
    }

    //! 出力します。
    void Out(std::ostream& os, const int tabCount) const;

    //! @brief レンダーステートのモードを設定します。
    //!
    //! @param[in] mode レンダーステートのモードです。
    //!
    void SetMode(const Mode mode);

    //! 同値であれば true を返します。
    bool operator==(const RRenderState& rhs) const
    {
        return (
            m_Mode         == rhs.m_Mode         &&
            m_DisplayFace  == rhs.m_DisplayFace  &&
            m_BlendMode    == rhs.m_BlendMode    &&
            m_DepthTest    == rhs.m_DepthTest    &&
            m_AlphaTest    == rhs.m_AlphaTest    &&
            m_ColorBlend   == rhs.m_ColorBlend   &&
            m_LogicalBlend == rhs.m_LogicalBlend);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RRenderState& rhs) const
    {
        return !(*this == rhs);
    }
};

//=============================================================================
//【現在は使用されていません】
//! @brief ステンシルテストのクラスです。
//=============================================================================
//class RStencilTest
//{
//public:
//  bool m_Enable; //!< ステンシルテストが有効なら true です。
//
//public:
//  //! コンストラクタです。
//  RStencilTest()
//  : m_Enable(false)
//  {
//  }
//  //! 出力します。
//  void Out(std::ostream& os, const int tabCount) const;
//  //! 同値であれば true を返します。
//  bool operator==(const RStencilTest& rhs) const
//  {
//      return (
//          m_Enable == rhs.m_Enable);
//  }
//  //! 同値でなければ true を返します。
//  bool operator!=(const RStencilTest& rhs) const
//  {
//      return !(*this == rhs);
//  }
//};

//=============================================================================
//! @brief オリジナルテクスチャ SRT のクラスです。
//=============================================================================
class ROriginalTexsrt
{
public:
    //! @brief テクスチャ SRT の計算方式を表す列挙型です。
    enum Mode { MAYA, MAX, SOFTIMAGE };

    //! @brief テクスチャ SRT アニメーションするパラメータを表す列挙型です。
    enum Param
    {
        SCALE_X,        //!< スケールの X 成分です。
        SCALE_Y,        //!< スケールの Y 成分です。
        ROTATE,         //!< 回転です。
        TRANSLATE_X,    //!< 移動の X 成分です。
        TRANSLATE_Y,    //!< 移動の Y 成分です。
        PARAM_COUNT     //!< パラメータの総数です。
    };

    Mode m_Mode; //!< テクスチャ SRT の計算方式です。
    RVec2 m_Scale; //!< UV のスケールです。
    float m_Rotate; //!< UV の回転です。
    RVec2 m_Translate; //!< UV の移動です。
    int m_UvHintIdx; //!< UV セットのヒント情報インデックスです。

public:
    //! コンストラクタです。
    ROriginalTexsrt()
    : m_Mode(MAYA),
      m_Scale(RVec2::kOne),
      m_Rotate(0.0f),
      m_Translate(RVec2::kZero),
      m_UvHintIdx(0)
    {
    }

    //! @brief 計算方式の文字列を返します。
    std::string GetModeString() const;

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <original_texsrt> 要素のインデントに必要なタブの数です。
    //! @param[in] index <original_texsrt_array> 内でのインデックスです。
    //! @param[in] hint hintStr 属性の値です。
    //!
    void Out(
        std::ostream& os,
        const int tabCount,
        const int index,
        const std::string& hintStr
    ) const;

    //! 同値であれば true を返します。
    bool operator==(const ROriginalTexsrt& rhs) const
    {
        return (
            m_Mode      == rhs.m_Mode      &&
            m_Scale     == rhs.m_Scale     &&
            m_Rotate    == rhs.m_Rotate    &&
            m_Translate == rhs.m_Translate);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const ROriginalTexsrt& rhs) const
    {
        return !(*this == rhs);
    }

    //! スケールのパラメータなら true を返します。
    static bool IsScale(const int paramIdx)
    {
        return (paramIdx == SCALE_X || paramIdx == SCALE_Y);
    }

    //! 回転のパラメータなら true を返します。
    static bool IsRotate(const int paramIdx)
    {
        return (paramIdx == ROTATE);
    }

    //! 移動のパラメータなら true を返します。
    static bool IsTranslate(const int paramIdx)
    {
        return (paramIdx == TRANSLATE_X || paramIdx == TRANSLATE_Y);
    }

    //! テクスチャ SRT アニメーションするパラメータの名前を返します。
    static const char* GetParamName(const int paramIdx);
};

//=============================================================================
//! @brief サンプラのクラスです。
//=============================================================================
class RSampler
{
public:
    //! @brief テクスチャの関連付けに利用するヒント情報を表す列挙型です。
    enum Hint
    {
        ALBEDO,         //!< アルベドマップです。
        OPACITY,        //!< 不透明度マップです。
        EMISSION,       //!< エミッションマップです。
        NORMAL,         //!< 法線マップです。
        TANGENT,        //!< 接線マップです。
        SPECULAR,       //!< スペキュラマップです。
        REFLECTION,     //!< 反射マップです。
        EXTRA,          //!< エクストラマップです。
        HINT_COUNT      //!< ヒント情報の総数です。
    };

    //! @brief ラップを表す列挙型です。
    enum Wrap
    {
        REPEAT,
        MIRROR,
        CLAMP,
        MIRROR_ONCE
        //CLAMP_HALF_BORDER,
        //MIRROR_ONCE_HALF_BORDER,
        //CLAMP_BORDER,
        //MIRROR_ONCE_BORDER
    };

    //! @brief フィルタを表す列挙型です。
    enum Filter { NONE, POINT, LINEAR };

    //! @brief アニソトロピックフィルタの最大値を表す列挙型です。
    enum MaxAniso { ANISO1, ANISO2, ANISO4, ANISO8, ANISO16 };

    // attr
    Hint m_Hint; //!< テクスチャの関連付けに利用するヒント情報です。
    int m_HintIndex; //!< テクスチャの関連付けに利用するヒント情報のインデックスです。
    std::string m_NameOverride; //!< サンプラの名前のオーバーライドです。
    std::string m_HintOverride; //!< ヒント情報の文字列のオーバーライドです。
    std::string m_TexName; //!< テクスチャ名です。

    Wrap m_WrapU; //!< U 方向のラップです。
    Wrap m_WrapV; //!< V 方向のラップです。
    Wrap m_WrapW; //!< W 方向のラップです。

    Filter m_FilterMag; //!< 拡大時のフィルタです。
    Filter m_FilterMin; //!< 縮小時のフィルタです。
    Filter m_FilterMip; //!< ミップマップ間のフィルタです。
    MaxAniso m_MaxAniso; //!< アニソトロピックフィルタの最大値です。

    float m_LodMin; //!< LOD の最小値です。
    float m_LodMax; //!< LOD の最大値です。
    float m_LodBias; //!< LOD のバイアス値です。

    // original texsrt
    ROriginalTexsrt m_OriginalTexsrt; //!< オリジナルテクスチャ SRT です。

    // internal
    std::string m_TexPath; //!< ftx ファイルのパスです。
    int m_ImageIndex; //!< テクスチャイメージのインデックスです。
    int m_TexNodeIndex; //!< DCC ツールのテクスチャノードのインデックスです。

    //! @brief テクスチャ SRT アニメーションのインデックスです。
    //!        テクスチャ SRT アニメーションが設定されていなければ -1 です。
    int m_TexSrtAnimIndex;

    //! @brief テクスチャパターンアニメーションのインデックスです。
    //!        テクスチャパターンアニメーションが設定されていなければ -1 です。
    int m_TexPatAnimIndex;

public:
    //! コンストラクタです。
    RSampler(const Hint hint = ALBEDO, const int hintIndex = 0)
    : m_Hint(hint),
      m_HintIndex(hintIndex),

      m_WrapU(CLAMP),
      m_WrapV(CLAMP),
      m_WrapW(CLAMP),

      m_FilterMag(LINEAR),
      m_FilterMin(LINEAR),
      m_FilterMip(NONE),
      m_MaxAniso(ANISO1),

      m_LodMin(0.0f),
      m_LodMax(13.0f),
      m_LodBias(0.0f),

      m_ImageIndex(0),
      m_TexNodeIndex(0),
      m_TexSrtAnimIndex(-1),
      m_TexPatAnimIndex(-1)
    {
    }

    //! サンプラの名前を返します。
    std::string GetName() const;

    //! ヒント情報の文字列を返します。
    std::string GetHintString() const;

    //! @brief テクスチャイメージ用のヒント情報の文字列を返します。
    std::string GetImageHintString() const;

    //! 出力します。
    void Out(std::ostream& os, const int tabCount, const int samplerIdx) const;

    //! 同値であれば true を返します。
    bool operator==(const RSampler& rhs) const
    {
        return (
            m_Hint       == rhs.m_Hint       &&
            m_HintIndex  == rhs.m_HintIndex  &&
            m_TexName    == rhs.m_TexName    &&
            m_WrapU      == rhs.m_WrapU      &&
            m_WrapV      == rhs.m_WrapV      &&
            m_WrapW      == rhs.m_WrapW      &&
            m_FilterMag  == rhs.m_FilterMag  &&
            m_FilterMin  == rhs.m_FilterMin  &&
            m_FilterMip  == rhs.m_FilterMip  &&
            m_MaxAniso   == rhs.m_MaxAniso   &&
            m_LodMin     == rhs.m_LodMin     &&
            m_LodMax     == rhs.m_LodMax     &&
            m_LodBias    == rhs.m_LodBias    &&

            m_OriginalTexsrt  == rhs.m_OriginalTexsrt &&

            m_TexPath         == rhs.m_TexPath);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RSampler& rhs) const
    {
        return !(*this == rhs);
    }

    //! アルベドに関連付けするなら true を返します。
    bool IsAlbedo() const { return m_Hint == ALBEDO; }

    //! 不透明度に関連付けするなら true を返します。
    bool IsOpacity() const { return m_Hint == OPACITY; }

    //! エミッションに関連付けするなら true を返します。
    bool IsEmission() const { return m_Hint == EMISSION; }

    //! 法線マップに関連付けするなら true を返します。
    bool IsNormal() const { return m_Hint == NORMAL; }

    //! 接線マップに関連付けするなら true を返します。
    bool IsTangent() const { return m_Hint == TANGENT; }

    //! スペキュラに関連付けするなら true を返します。
    bool IsSpecular() const { return m_Hint == SPECULAR; }

    //! 反射マップに関連付けするなら true を返します。
    bool IsReflection() const { return m_Hint == REFLECTION; }
};

//! @brief サンプラ配列の定義です。
typedef std::vector<RSampler> RSamplerArray;

//=============================================================================
//! @brief オリジナルカラーのクラスです。
//=============================================================================
class ROriginalColor
{
public:
    //! @brief カラーの関連付けに利用するヒント情報を表す列挙型です。
    enum Hint
    {
        DIFFUSE,    //!< ディフューズカラーです。
        OPACITY,    //!< 不透明カラーです。
        AMBIENT,    //!< アンビエントカラーです。
        EMISSION,   //!< エミッションカラーです。
        SPECULAR,   //!< スペキュラカラーです。
    };

    Hint m_Hint; //!< カラーの関連付けに利用するヒント情報です。
    RVec3 m_Color; //!< カラーの値です。

public:
    //! コンストラクタです。
    ROriginalColor(const Hint hint, const RVec3& color)
    : m_Hint(hint),
      m_Color(color)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <original_color> 要素のインデントに必要なタブの数です。
    //! @param[in] index <original_color_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;
};

//! @brief オリジナルカラー配列の定義です。
typedef std::vector<ROriginalColor> ROriginalColorArray;

//=============================================================================
//! @brief マテリアルのクラスです。
//=============================================================================
class RMaterial
{
public:
    //! @brief カラーアニメーションするパラメータを表す列挙型です。
    enum ColorParam
    {
        DIFFUSE_R ,         //!< ディフューズカラーの R 成分です。
        DIFFUSE_G ,         //!< ディフューズカラーの G 成分です。
        DIFFUSE_B ,         //!< ディフューズカラーの B 成分です。
        OPACITY_R ,         //!< 不透明カラーの R 成分です。
        OPACITY_G ,         //!< 不透明カラーの G 成分です。
        OPACITY_B ,         //!< 不透明カラーの B 成分です。
        AMBIENT_R ,         //!< アンビエントカラーの R 成分です。
        AMBIENT_G ,         //!< アンビエントカラーの G 成分です。
        AMBIENT_B ,         //!< アンビエントカラーの B 成分です。
        EMISSION_R,         //!< エミッションカラーの R 成分です。
        EMISSION_G,         //!< エミッションカラーの G 成分です。
        EMISSION_B,         //!< エミッションカラーの B 成分です。
        SPECULAR_R,         //!< スペキュラカラーの R 成分です。
        SPECULAR_G,         //!< スペキュラカラーの G 成分です。
        SPECULAR_B,         //!< スペキュラカラーの B 成分です。
        COLOR_PARAM_COUNT   //!< パラメータの総数です。
    };

    // attr
    std::string m_Name; //!< 出力用のマテリアル名です。
    bool m_Visibility; //!< マテリアルを割り当てたシェイプの可視性です。
    bool m_CompressEnable; //!< 最適化で圧縮可能なら true です。
    RRenderState m_RenderState; //!< レンダーステートです。
    RSamplerArray m_Samplers; //!< サンプラ配列です。

    // original color
    RVec3 m_Diffuse;  //!< ディフューズカラーです。
    RVec3 m_Opacity; //!< 不透明カラーです。
    RVec3 m_Ambient;  //!< アンビエントカラーです。
    RVec3 m_Emission; //!< エミッションカラーです。
    RVec3 m_Specular; //!< スペキュラカラーです。

    // internal attr
    bool m_IsSpecularEnable; //!< スペキュラを使用するかどうかです。
    float m_Shininess; //!< スペキュラの光沢です。小さいほどハイライトが大きくなります。
    bool m_IsAnisotropic; //!< 異方性反射かどうかです。

    // user data
    RUserDataArray m_UserDatas; //!< ユーザーデータ配列です。

public:
    //! コンストラクタです。
    RMaterial()
    :
      // attr
      m_Visibility(true),
      m_CompressEnable(true),

      m_Diffuse(RVec3::kOne),
      m_Opacity(RVec3::kOne),
      m_Ambient(RVec3::kZero),
      m_Emission(RVec3::kZero),
      m_Specular(RVec3::kZero),

      // internal attr
      m_IsSpecularEnable(false),
      m_Shininess(30.0f),
      m_IsAnisotropic(false)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <material> 要素のインデントに必要なタブの数です。
    //! @param[in] matIdx <material_array> 内でのインデックスです。
    //!
    void Out(
        std::ostream& os,
        const int tabCount,
        const int matIdx
    ) const;

    //! @brief オリジナルマテリアルを出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <original_material> 要素のインデントに必要なタブの数です。
    //! @param[in] matIdx <original_material_array> 内でのインデックスです。
    //!
    void OutOriginal(
        std::ostream& os,
        const int tabCount,
        const int matIdx
    ) const;

    //! @brief 他のマテリアルと要素の値が同じなら true を返します。
    //!
    //! @param[in] other 比較する他のマテリアルです。
    //!
    //! @return 要素の値が同じなら true を返します。
    //!
    bool IsSameElement(const RMaterial& other) const;

    //! @brief レンダーステートのモードを取得します。
    //!
    //! @return レンダーステートのモードです。
    //!
    RRenderState::Mode GetRenderStateMode() const
    {
        return m_RenderState.m_Mode;
    }

    //! @brief レンダーステートのモードを設定します。
    //!
    //! @param[in] mode レンダーステートのモードです。
    //!
    void SetRenderStateMode(const RRenderState::Mode mode)
    {
        m_RenderState.SetMode(mode);
    }

    //! ディフューズカラーのパラメータなら true を返します。
    static bool IsDiffuse(const int paramIdx)
    {
        return (DIFFUSE_R <= paramIdx && paramIdx <= DIFFUSE_B);
    }

    //! 不透明カラーのパラメータなら true を返します。
    static bool IsOpacity(const int paramIdx)
    {
        return (OPACITY_R <= paramIdx && paramIdx <= OPACITY_B);
    }

    //! アンビエントカラーのパラメータなら true を返します。
    static bool IsAmbient(const int paramIdx)
    {
        return (AMBIENT_R <= paramIdx && paramIdx <= AMBIENT_B);
    }

    //! エミッションカラーのパラメータなら true を返します。
    static bool IsEmission(const int paramIdx)
    {
        return (EMISSION_R <= paramIdx && paramIdx <= EMISSION_B);
    }

    //! スペキュラカラーのパラメータなら true を返します。
    static bool IsSpecular(const int paramIdx)
    {
        return (SPECULAR_R <= paramIdx && paramIdx <= SPECULAR_B);
    }

    //! カラーアニメーションするパラメータの名前を返します。
    static const char* GetColorParamName(const int paramIdx);
};

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

