﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// RUnicodeCArray
// OptionArgInfo
// CIntermediateFile

//=============================================================================
// include
//=============================================================================

// DCC Library のヘッダを利用します。
#include <DccCommon.h>

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//=============================================================================
//! @brief ユニコード C 配列のクラスです。
//=============================================================================
class RUnicodeCArray
{
protected:
    gfx::tool::dcc::RWStringArray m_WStrings; //!< ユニコード文字列配列です。
    const wchar_t** m_CArray; //!< ユニコード C 配列です。

public:
    //! コンストラクタです（RStringArray から生成）。
    explicit RUnicodeCArray(const gfx::tool::dcc::RStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int strIdx = 0; strIdx < strCount; ++strIdx)
        {
            m_WStrings.push_back(gfx::tool::dcc::RGetUnicodeFromShiftJis(srcs[strIdx]));
        }
        CreateCArray();
    }

    //! コンストラクタです（RWStringArray から生成）。
    explicit RUnicodeCArray(const gfx::tool::dcc::RWStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int strIdx = 0; strIdx < strCount; ++strIdx)
        {
            m_WStrings.push_back(srcs[strIdx]);
        }
        CreateCArray();
    }

    //! ユニコード C 配列を作成します。
    void CreateCArray()
    {
        const int strCount = static_cast<int>(m_WStrings.size());
        m_CArray = new const wchar_t*[strCount + 1];
        for (int strIdx = 0; strIdx < strCount; ++strIdx)
        {
            m_CArray[strIdx] = m_WStrings[strIdx].c_str();
        }
        m_CArray[strCount] = nullptr; // 最終要素を nullptr にします。
    }

    //! デストラクタです。
    virtual ~RUnicodeCArray()
    {
        //std::cerr << "~RUnicodeCArray(): " << m_WStrings.size() << std::endl;
        delete[] m_CArray;
    }

    //! ユニコード C 配列を返します。
    const wchar_t** GetCArray() const { return m_CArray; }
};

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 文字列の末尾の整数値を取得します。
//!
//! @param[out] pNumber 整数値を格納します。文字列の末尾が数値でない場合は -1 を格納します。
//! @param[in] str 文字列です。
//!
//! @return 文字列の末尾の数値を削除した文字列を返します。
//!
std::string GetSuffixNumberFromString(int* pNumber, const std::string& str);

//=============================================================================
// 一般的な Windows 関連の関数です。
//=============================================================================

//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//!
//! @param[in] hModule モジュールのハンドルです
//!                    nullptr なら現在のプロセスを作成するために使われたファイルのパスを取得します。
//!
//! @return 実行ファイルのフルパスを返します。
//!
std::string GetModuleFilePath(HMODULE hModule);

//=============================================================================
//! @brief オプション引数情報の構造体です。
//=============================================================================
struct OptionArgInfo
{
    char shortName; //!< ショートネームです。ショートネームなしなら空白です。
    const char* longName; //!< ロングネームです。nullptr なら配列の最後を表します。末尾の # は整数値を表します。
    bool requiresValue; //!< 値を必要とするなら true です。
};

//=============================================================================
// 引数関連の関数です。
//=============================================================================

//! @brief テキストファイルの 1 行から引数のトークンを取得します。
//!
//! @param[in,out] pTokens 取得した引数のトークンを追加する配列へのポインタです。
//! @param[in] lineStr 1 行の文字列です。
//!
void GetArgTokensFromLine(gfx::tool::dcc::RStringArray* pTokens, const std::string& lineStr);

//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[in,out] pTokens 取得した引数のトークンを追加する配列へのポインタです。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理結果を返します。
//!
gfx::tool::dcc::RStatus GetArgTokensFromFile(gfx::tool::dcc::RStringArray* pTokens, const std::string& filePath);

//! @brief 引数配列からオプション値を取得します。
//!
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//!
//! @return 処理結果を返します。
//!
gfx::tool::dcc::RStatus GetOptionArgValue(
    std::string* pValue,
    int* pArgIdx,
    const gfx::tool::dcc::RStringArray& args
);

//! @brief 引数からオプション引数情報へのポインタを取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @return オプション引数情報へのポインタを返します。
//!         該当するオプション引数情報がなければ nullptr を返します。
//!
const OptionArgInfo* GetOptionArgInfo(
    bool* pIsLongName,
    int* pSuffixNum,
    const OptionArgInfo* optionArgInfos,
    const std::string& arg
);

//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
bool IsNextArgOptionValue(const OptionArgInfo* optionArgInfos, const std::string& arg);

//! @brief オプション引数を解析します。
//!
//! @param[out] ppInfo オプション引数情報へのポインタを格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//! @param[in] optionArgInfos オプション引数情報配列です。
//!
//! @return 処理結果を返します。
//!
gfx::tool::dcc::RStatus ParseOptionArg(
    const OptionArgInfo** ppInfo,
    int* pSuffixNum,
    std::string* pValue,
    int* pArgIdx,
    const gfx::tool::dcc::RStringArray& args,
    const OptionArgInfo* optionArgInfos
);

//=============================================================================
// ファイル関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief フォルダが存在しなければ作成します。
//!
//! @param[in] folderPath フォルダのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
gfx::tool::dcc::RStatus CreateFolderIfNotExist(const std::string& folderPath);

//-----------------------------------------------------------------------------
//! @brief メモリを確保してファイルをリードします。
//!
//! @param[out] ppBuf 確保したメモリへのポインタを格納します。
//! @param[out] pSize ファイルサイズ（バイト数）を格納します。
//! @param[in] filePath ファイルのパスを指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
gfx::tool::dcc::RStatus ReadFileToMemory(
    uint8_t** ppBuf,
    size_t* pSize,
    const std::string& filePath
);

//=============================================================================
//! @brief NW4C 中間ファイルのクラスです。
//=============================================================================
class CIntermediateFile
{
public:
    std::string m_Path; //!< 中間ファイルのパスです。
    gfx::tool::dcc::RXMLElement m_Xml; //!< ルートの XML 要素です。
    const gfx::tool::dcc::RXMLElement* m_GraphicsElem; //!< グラフィックス要素へのポインタです。

public:
    //! @biref コンストラクタです。
    //!        ファイルをメモリに読み込んで XML を解析します。
    //!
    //! @param[out] pStatus 処理結果を格納します。
    //! @param[in] path 中間ファイルのパスです。
    //!
    CIntermediateFile(gfx::tool::dcc::RStatus* pStatus, const std::string& path);

    //! デストラクタです。
    virtual ~CIntermediateFile()
    {
    }
};

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

