﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// C2NnConverter C2NnOption

//=============================================================================
// include
//=============================================================================
#include "DccLib.h"
#include "Texture.h"

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//=============================================================================
//! @brief NW4C から NintendoSDK への変換オプションのクラスです。
//=============================================================================
class C2NnOption
{
public:
    //! @brief リニア変換フラグを表す列挙型です。
    enum LinearFlag
    {
        LinearFlag_None    = 0,        //!< sRGB からリニアに変換しません。
        LinearFlag_R       = (1 << 3), //!< R 成分を sRGB からリニアに変換します。
        LinearFlag_G       = (1 << 2), //!< G 成分を sRGB からリニアに変換します。
        LinearFlag_B       = (1 << 1), //!< B 成分を sRGB からリニアに変換します。
        LinearFlag_A       = (1 << 0), //!< A 成分を sRGB からリニアに変換します。
        LinearFlag_Rgb     = LinearFlag_R | LinearFlag_G | LinearFlag_B, //!< RGB 成分を sRGB からリニアに変換します。
        LinearFlag_Rgba    = LinearFlag_Rgb | LinearFlag_A,          //!< RGBA 成分を sRGB からリニアに変換します。
        LinearFlag_Invalid = -2,       //!< 無効なリニア変換です。
        LinearFlag_Default = -1        //!< デフォルトのリニア変換（指定なし）です。
    };

public:
    bool m_IsSilent; //!< 処理内容の表示を抑制するなら true です。
    std::string m_ProjectRootPath; //!< プロジェクトのルートフォルダーのパスです。
    bool m_DisablesFileInfo; //!< <file_info> を出力しないなら true です。
    std::string m_GraphicsToolsPath; //!< グラフィックスツールの実行ファイルが存在するフォルダのパスです。
    std::string m_3dToolsPath; //!< 3D ツールの実行ファイルが存在するフォルダのパスです。
    TextureEncoder* m_pTextureEncoder; //!< テクスチャエンコーダへのポインタです。

    float m_ConversionMagnify; //!< 移動値や頂点座標に掛ける倍率です。
    bool m_ConvertsTex; //!< ctex ファイルを ftx ファイルに変換するなら true です。
    bool m_ConvertsAllTex; //!< Textures フォルダに存在する全 ctex ファイルを変換するなら true です。
    bool m_ForcesMaxMip; //!< 強制的に NintendoSDK の最大レベルまでミップマップを作成するなら true です。
    std::string m_MipGenFilter; //!< ミップマップ生成フィルタです（空文字なら指定なし）。
    int m_TexLinearFlag; //!< テクスチャのリニア変換フラグです。
    int m_VtxColLinearFlag; //!< 頂点カラーのリニア変換フラグです。

    std::string m_InputPath; //!< 入力ファイルのパスです。
    std::string m_ElementName; //!< 変換するモデル名またはアニメーション名です。

    std::string m_OutputPath; //!< 出力ファイルのパスです。
    std::string m_NoExtOutputPath; //!< 拡張子を含まない出力ファイルのパスです。
    bool m_IsBinaryFormat; //!< 出力ファイルがバイナリ形式なら true です。
    bool m_UsesFclFtsFtp; //!< fcl / fts / ftp ファイルを使用するなら true、fma ファイルを使用するなら false です。
    bool m_SeparatesFma; //!< 各種類のマテリアルアニメーションを個別の fma ファイルに出力するなら true です。
    std::string m_CommentText; //!< 編集用コメント文です。
    bool m_ArrangesFormat; //!< 中間ファイルの書式を調整するなら true です（非公開）。

public:
    //! @brief コンストラクタです。
    C2NnOption()
    : m_IsSilent(false),
      m_DisablesFileInfo(false),
      m_pTextureEncoder(nullptr),
      m_ConversionMagnify(1.0f),
      m_ConvertsTex(true),
      m_ConvertsAllTex(false),
      m_ForcesMaxMip(false),
      m_TexLinearFlag(LinearFlag_None),
      m_VtxColLinearFlag(LinearFlag_None),
      m_IsBinaryFormat(true),
      m_UsesFclFtsFtp(false),
      m_SeparatesFma(false),
      m_ArrangesFormat(false)
    {
    }

    //! @brief デストラクタです。
    virtual ~C2NnOption()
    {
    }

    //! @brief 初期化します。
    void Initialize()
    {
        *this = C2NnOption();
    }

    //! @brief ジョブオプションを設定します。
    //!
    //! @param[in] options ジョブオプションのユニコード文字列配列です。
    //!
    //! @return 処理結果を返します。
    //!
    gfx::tool::dcc::RStatus SetJobOption(const wchar_t* options[]);

    //! @brief 入力ファイルパスを設定します。
    //!
    //! @param[in] path 入力ファイルパスのユニコード文字列です。
    //!
    //! @return 処理結果を返します。
    //!
    gfx::tool::dcc::RStatus SetInputPath(const wchar_t* path);

    //! @brief 入力ファイルの情報を表示します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //!
    void DisplayFileInfo(std::ostream& os) const;
};

//=============================================================================
//! @brief NW4C から NintendoSDK へのコンバータのクラスです。
//=============================================================================
class C2NnConverter
{
protected:
    //-----------------------------------------------------------------------------
    // DLL 関連
    HINSTANCE m_hDLL; //!< コンバータの DLL のインスタンスハンドルです。

    //-----------------------------------------------------------------------------
    // オプション関連
    bool m_IsSilent; //!< 処理内容の表示を抑制するなら true です。
    std::string m_GraphicsToolsPath; //!< グラフィックスツールの実行ファイルが存在するフォルダのパスです。
    std::string m_3dToolsPath; //!< 3D ツールの実行ファイルが存在するフォルダのパスです。
    TextureEncoder m_TextureEncoder; //!< テクスチャエンコーダです。

    C2NnOption m_Option; //!< NW4C から NintendoSDK への変換オプションです。

    //-----------------------------------------------------------------------------
    // テクスチャ関連
    gfx::tool::dcc::RStringArray m_OutputFtxPaths; //!< 出力済みの ftx ファイルのパス配列です。

    //-----------------------------------------------------------------------------
    // エラー関連
    std::string m_ErrorString; //!< エラーの Shift JIS 文字列です。
    std::wstring m_ErrorWString; //!< エラーのユニコード文字列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] DLL のインスタンスハンドルです。
    //!
    explicit C2NnConverter(HINSTANCE hinstDLL);

    //! @brief デストラクタです。
    virtual ~C2NnConverter();

    //! @brief コンバータのバージョンを返します。
    int GetConverterVersion() const;

    //! コンバータの使用方法を表示します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] converterName コンバータ名です。
    //! @param[in] cmdName コンバータのコマンド名です。
    //! @param[in] isVersionOnly バージョンのみ表示するなら true を指定します。
    //!
    void ShowUsage(
        std::ostream& os,
        const wchar_t* converterName,
        const wchar_t* cmdName,
        const bool isVersionOnly
    ) const;

    //! @brief 引数がグローバルオプションなら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 引数がグローバルオプションなら true を返します。
    //!
    bool IsGlobalOptionArg(const wchar_t* arg) const;

    //! @brief 指定した引数の次の引数がグローバルオプションの値なら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 次の引数がグローバルオプションの値なら true を返します。
    //!
    bool IsNextArgGlobalOptionValue(const wchar_t* arg) const;

    //! @brief 指定した引数の次の引数がジョブオプションの値なら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 次の引数がジョブオプションの値なら true を返します。
    //!
    bool IsNextArgJobOptionValue(const wchar_t* arg) const;

    //! @brief グローバルオプションを設定します。
    //!
    //! @param[in] グローバルオプションのユニコード文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetOption(const wchar_t* options[]);

    //! @brief メモリをクリアし、ジョブオプションを初期化します。
    void Clear()
    {
        InitializeJobOption();
    }

    //! @brief 出力したテクスチャファイルの情報を初期化します。
    void ClearTextureInfo()
    {
        m_OutputFtxPaths.clear();
    }

    //! @brief エラー文字列をクリアします。
    void ClearError()
    {
        m_ErrorString.clear();
        m_ErrorWString.clear();
    }

    //! @brief 入力ファイルを変換して NintendoSDK 中間ファイルを出力します。
    //!
    //! @param[in] path 入力ファイルのパスです。
    //! @param[in] options ジョブオプション文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool Convert(const wchar_t* path, const wchar_t* options[]);

    //! @brief 出力したテクスチャファイルの情報を表示します。
    void DisplayTextureInfo(std::ostream& os) const;

    //! @brief エラーのユニコード文字列を返します。
    const wchar_t* GetErrorString() const
    {
        return m_ErrorWString.c_str();
    }

    //! @brief エラーの Shift JIS 文字列を返します。
    const char* GetErrorShiftJisString() const
    {
        return m_ErrorString.c_str();
    }

protected:
    //! @brief グローバルオプションを初期化します。
    void InitializeGlobalOption()
    {
        m_IsSilent = false;
    };

    //! @brief グローバルオプションを NW4C から NintendoSDK への変換オプションにコピーします。
    void CopyGlobalOption()
    {
        m_Option.m_IsSilent          = m_IsSilent;
        m_Option.m_GraphicsToolsPath = m_GraphicsToolsPath;
        m_Option.m_3dToolsPath       = m_3dToolsPath;
    };

    //! @brief ジョブオプションを初期化します。
    void InitializeJobOption()
    {
        m_Option.Initialize();
        CopyGlobalOption();
        m_Option.m_pTextureEncoder = &m_TextureEncoder;
    };

    //! @brief 処理結果ステータスを解析します。
    bool ParseStatus(const gfx::tool::dcc::RStatus& status)
    {
        if (!status)
        {
            m_ErrorString = status.GetMessage();
            m_ErrorWString = gfx::tool::dcc::RGetUnicodeFromShiftJis(m_ErrorString);
        }
        return static_cast<bool>(status);
    }

    //! @brief 入力ファイルを変換して NintendoSDK 中間ファイルを出力します。
    //!
    //! @param[in] path 入力ファイルのパスです。
    //! @param[in] options ジョブオプション文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    gfx::tool::dcc::RStatus ConvertSub(const wchar_t* path, const wchar_t* options[]);
};

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

