﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <clocale>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <eh.h>
#include <io.h>
#include <Fcntl.h>

#include <util/UtilError.h>
#include <nn/gfxTool/gfxTool_Util.h>

#include <bindefs.h>
#include <BinConverter.h>
#define DLLEXPORT extern "C" __declspec(dllexport)

bool g_IsDllInitialized = false;

static nn::g3dTool::Converter g_Converter;
FILE* g_pOutLogFile = NULL;
FILE* g_pWarningLogFile = NULL;
FILE* g_pErrLogFile = NULL;
std::wstring s_DllPath;
const char* g_StringResourceName = "3dConverterStringResource.csv";

void se_translator_function(unsigned int /*code*/, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; //標準C++の例外を発生させる。
}

DLLEXPORT
bool nng3dToolBinaryConverterSetLogStream(const nngfxToolSetLogStreamArg* pArg)
{
    if (!g_IsDllInitialized)
    {
        PRINT_ERROR_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
        return false;
    }
    if (pArg->hLogStream)
    {
        int nHandle = _open_osfhandle(reinterpret_cast<intptr_t>(pArg->hLogStream), _O_TEXT);
        switch (static_cast<nn::gfxTool::Logger::LogType>(pArg->logType))
        {
        case nn::gfxTool::Logger::LogType::Default:
            {
                g_pOutLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetOutStream(g_pOutLogFile);
            }
            break;
        case nn::gfxTool::Logger::LogType::Warning:
            {
                g_pWarningLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetWarningStream(g_pWarningLogFile);
            }
            break;
        case nn::gfxTool::Logger::LogType::Error:
            {
                g_pErrLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetErrorStream(g_pErrLogFile);
            }
            break;
        default:
            PRINT_ERROR_LOG("Invalid LogType.");
            return false;
        }
    }
    return true;
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterOpenLogFile(const wchar_t* out[2], const wchar_t* err[2])
{
    if (_wfopen_s(&g_pOutLogFile, out[0], out[1]) == 0)
    {
        nw::g3d::tool::LogConfig::SetOutStream(g_pOutLogFile);
    }

    if (_wfopen_s(&g_pErrLogFile, err[0], err[1]) == 0)
    {
        nw::g3d::tool::LogConfig::SetErrorStream(g_pErrLogFile);
    }
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterCloseLogFile()
{
    if (g_pOutLogFile != NULL)
    {
        fclose(g_pOutLogFile);
        g_pOutLogFile = NULL;
    }

    if (g_pErrLogFile != NULL)
    {
        fclose(g_pErrLogFile);
        g_pErrLogFile = NULL;
    }

    nw::g3d::tool::LogConfig::SetOutStream(stdout);
    nw::g3d::tool::LogConfig::SetErrorStream(stderr);
}

DLLEXPORT
uint32_t nng3dToolBinaryConverterGetConverterVersion()
{
    return g_Converter.GetCvtrVersion();
}

DLLEXPORT
bool nng3dToolBinaryConverterInitialize()
{
    if (g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is already initialized.");
    }

    // エラー用の英語もしくは日本語のメッセージリストを初期化
    std::string stringResourcePath;
#if defined(_M_IX86)
    stringResourcePath = nw::g3d::tool::util::ToUTF8(s_DllPath.c_str()) + "../" + g_StringResourceName;
#else
    stringResourcePath = nw::g3d::tool::util::ToUTF8(s_DllPath.c_str()) + g_StringResourceName;
#endif
    if (!nw::g3d::tool::LogConfig::InitializeStringResource(stringResourcePath.c_str()))
    {
        // メッセージリスト構築に失敗してもコンバート自体は出来るので、エラーにはしない。
        PRINT_SYSTEM_LOG("Faild to setup StringResource");
    }

    bool result = g_Converter.Initialize();
    g_IsDllInitialized = true;
    return result;
}

DLLEXPORT
bool nng3dToolBinaryConverterClear()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.Clear();
}

DLLEXPORT
bool nng3dToolBinaryConverterShutdown()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    nw::g3d::tool::LogConfig::FinalizeStringResource();
    bool result = g_Converter.Shutdown();
    g_IsDllInitialized = false;
    return result;
}

DLLEXPORT
bool nng3dToolBinaryConverterSetOptions(const wchar_t* options[])
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.SetOptions(options);
}

DLLEXPORT
bool nng3dToolBinaryConverterAddFile(void* pData, size_t dataSize, const wchar_t* path[3], const wchar_t* options[])
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.AddFile(pData, dataSize, path, options);
}

//! @brief 内部で使用しているヒープサイズを返します。リロケーションテーブルは含みません。
DLLEXPORT
size_t nng3dToolBinaryConverterCalculateSize()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    void* pDummy = nullptr;
    size_t size = 0;
    if( g_Converter.CalculateSize() != 0 && g_Converter.Convert( pDummy, size ) != false )
    {
        size_t fileSize = g_Converter.GetBaseSize() + g_Converter.GetRelocationTableSize();
        return fileSize;
    }

    return 0;	// エラーの時は旧コンバータの仕様では 0 を返します。
}

//! @brief ファイルのアライメントを返します。
DLLEXPORT
size_t nng3dToolBinaryConverterGetAlignmentSize()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.GetAlignmentSize();
}

DLLEXPORT
bool nng3dToolBinaryConverterConvert(void* pBuffer, size_t size)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    size_t serializedDataSize = g_Converter.GetBaseSize();
    size_t relocationTblSize = g_Converter.GetRelocationTableSize();
    size_t fileSize = serializedDataSize + relocationTblSize;	// relocation table を含めたサイズ
    if( fileSize > size )
    {
        return false;
    }

    // データ部分をコピー
    std::memcpy( pBuffer, g_Converter.GetBasePtr(), serializedDataSize );

    // リロケーションテーブルをコピー
    char* pDst = static_cast< char* >( pBuffer ) + serializedDataSize;
    std::memcpy( pDst, g_Converter.GetRelocationTablePtr(), relocationTblSize );

    return true;
}

DLLEXPORT
bool nng3dToolBinaryConverterBind(void* pBuffer, size_t size)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.Bind( pBuffer, size );
}

DLLEXPORT
bool nng3dToolBinaryConverterSwapEndian(void* pBuffer, size_t size)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dBinaryConverter.dll is not initialized. Please call nng3dToolBinaryConverterInitialize().");
    }
    return g_Converter.SwapEndian(pBuffer, size);
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterForceNwArchive(void)
{
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterForceNnArchive(void)
{
}

DLLEXPORT
uint32_t nng3dToolBinaryConverterGetBinaryVersion( void )
{
    return g_Converter.GetCvtrBinaryVersion();
}

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID /*lpvReserved*/)
{
    wchar_t dllfile[_MAX_PATH];
    wchar_t drive[_MAX_DRIVE];
    wchar_t dllpath[_MAX_DIR];
    std::wstringstream wss;
    int size = 0;
    switch (fdwReason)
    {
    case DLL_PROCESS_ATTACH:
        setlocale(LC_CTYPE, "");
        _set_se_translator(se_translator_function); // このスレッドでのみ有効。

        size = GetModuleFileName(hinstDLL, dllfile, _MAX_PATH);
        if (size != 0)
        {
            _wsplitpath_s(dllfile, drive, _MAX_DRIVE, dllpath, _MAX_DIR, NULL, 0, NULL, 0);

            wss << drive << dllpath;
            s_DllPath = wss.str();
            std::replace(s_DllPath.begin(), s_DllPath.end(), L'\\', L'/');
        }

        break;
    case DLL_PROCESS_DETACH:
        nng3dToolBinaryConverterCloseLogFile();
        break;
    case DLL_THREAD_ATTACH:
        break;
    case DLL_THREAD_DETACH:
        break;
    }

    // 32 ビット版の精度を 64 ビット版相当にします。
    // 32 ビット版でも SIMD の精度にする。
#ifndef _WIN64
    unsigned int control_word;
    int err = _controlfp_s(&control_word, _PC_24, MCW_PC);
    if (err != 0)
    {
        return FALSE;
    }
#endif

    return TRUE;
}
