﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <ShapeUtil.h>

namespace nn {
namespace g3dTool {

// 座標データからバウンディングを構築する。
BoundingVolume ComputeBoundingVolume(const std::vector<nw::g3d::tool::util::Vec3_t>& positions,
                                const int* indices, int startIndex, int count, int skinningCount, int lodOffset )
{
    BoundingVolume bounding;
    int blendBoneCount = std::max(1, skinningCount);
    int endIndex = startIndex + count;
    // mesh(lodOffset) + index[submesh.offset] をオフセットした頂点を取得
    const nw::g3d::tool::util::Vec3_t& startPos = positions[indices[startIndex] + lodOffset];
    std::memcpy(&bounding.volumeMin, &startPos, sizeof(bounding.volumeMin));
    std::memcpy(&bounding.volumeMax, &startPos, sizeof(bounding.volumeMax));
    for (int indicesIndex = startIndex; indicesIndex < endIndex; ++indicesIndex)
    {
        for (int blendBoneIndex = 0; blendBoneIndex < blendBoneCount; ++blendBoneIndex)
        {
            int positionIndex = (indices[indicesIndex] + lodOffset) * blendBoneCount + blendBoneIndex;
            const nw::g3d::tool::util::Vec3_t& pos = positions[positionIndex];
            // NOTE: 一度使った頂点を set で保存して max, min の比較を行わないようにしたら速度が低下した。
            bounding.volumeMax.x = std::max(bounding.volumeMax.x, pos.x);
            bounding.volumeMax.y = std::max(bounding.volumeMax.y, pos.y);
            bounding.volumeMax.z = std::max(bounding.volumeMax.z, pos.z);

            bounding.volumeMin.x = std::min(bounding.volumeMin.x, pos.x);
            bounding.volumeMin.y = std::min(bounding.volumeMin.y, pos.y);
            bounding.volumeMin.z = std::min(bounding.volumeMin.z, pos.z);
        }
    }
    return bounding;
}

// バウンディングを拡張する。
BoundingVolume MergeBoundingVolume(const BoundingVolume& lhs, const BoundingVolume& rhs)
{
    BoundingVolume bounding;
    bounding.volumeMin.x = std::min(lhs.volumeMin.x, rhs.volumeMin.x);
    bounding.volumeMin.y = std::min(lhs.volumeMin.y, rhs.volumeMin.y);
    bounding.volumeMin.z = std::min(lhs.volumeMin.z, rhs.volumeMin.z);

    bounding.volumeMax.x = std::max(lhs.volumeMax.x, rhs.volumeMax.x);
    bounding.volumeMax.y = std::max(lhs.volumeMax.y, rhs.volumeMax.y);
    bounding.volumeMax.z = std::max(lhs.volumeMax.z, rhs.volumeMax.z);

    return bounding;
}

// 中心点からバウンディングスフィアを計算する。
float ComputeBoundingSphere(const std::vector<nw::g3d::tool::util::Vec3_t>& positions, const nn::util::Float3& center,
                            const int* indices, int startIndex, int count, int skinningCount, int lodOffset )
{
    int blendBoneCount = std::max(1, skinningCount);
    float distance = 0.0f;
    int endIndex = startIndex + count;
    for (int indicesIdx = startIndex; indicesIdx < endIndex; ++indicesIdx)
    {
        for (int blendBoneIndex = 0; blendBoneIndex < blendBoneCount; ++blendBoneIndex)
        {
            int positionIdx = (indices[indicesIdx] + lodOffset) * blendBoneCount + blendBoneIndex;
            float dx = (center.x - positions[positionIdx].x) * (center.x - positions[positionIdx].x);
            float dy = (center.y - positions[positionIdx].y) * (center.y - positions[positionIdx].y);
            float dz = (center.z - positions[positionIdx].z) * (center.z - positions[positionIdx].z);
            distance = std::max(distance, dx + dy + dz);
        }
    }
    return static_cast<float>(std::sqrt(distance));
}

nn::g3d::Bounding BoundingVolumeToBounding(const BoundingVolume& BoundingVolume)
{
    nn::util::Float3 volumeMax = BoundingVolume.volumeMax;
    nn::util::Float3 volumeMin = BoundingVolume.volumeMin;
    nn::g3d::Bounding bounding;
    bounding.center = nn::util::MakeFloat3((volumeMax.x + volumeMin.x) / 2.f, (volumeMax.y + volumeMin.y) / 2.f, (volumeMax.z + volumeMin.z) / 2.f);
    bounding.extent = nn::util::MakeFloat3((volumeMax.x - volumeMin.x) / 2.f, (volumeMax.y - volumeMin.y) / 2.f, (volumeMax.z - volumeMin.z) / 2.f);

    return bounding;
}

// min, max から center, extent に変換する。
BoundingVolume BoundingToBoundingVolume(const nn::g3d::Bounding& bounding)
{
    nn::util::Float3 center = bounding.center;
    nn::util::Float3 extent = bounding.extent;
    BoundingVolume BoundingVolume;
    BoundingVolume.volumeMax = nn::util::MakeFloat3(center.x + extent.x, center.y + extent.y, center.z + extent.z);
    BoundingVolume.volumeMin = nn::util::MakeFloat3(center.x - extent.x, center.y - extent.y, center.z - extent.z);

    return BoundingVolume;
}

void CalculateBounding(nn::g3d::Bounding* pBoundingArray,
    float* pRadiusArray,
    const nw::g3d::tool::g3dif::elem_shape& elemShape,
    const std::vector<nw::g3d::tool::util::Vec3_t>* pLocalPositions,
    const std::vector<int>* pLodOffsets)
{
    nn::g3d::Bounding* pSubmeshBoundingArray = pBoundingArray;
    auto& meshArray = elemShape.mesh_array;
    int meshCount = static_cast<int>(meshArray.size());

    for (int idxMesh = 0; idxMesh < meshCount; ++idxMesh)
    {
        auto& elemMesh = meshArray[idxMesh];
        int lodOffset = (*pLodOffsets)[idxMesh];

        // バウンディングの初期値を与える。
        // 各サブメッシュの BoundingVolume をマージしてシェイプの AABB を求めるので、先にサブメッシュ 0 を計算する必要がある。
        const nw::g3d::tool::g3dif::elem_submesh& elemSubmesh0 = elemMesh.submesh_array[0];
        const int* indices = static_cast<const int*>(elemMesh.stream.rawdata.get());
        BoundingVolume shapeBoundingVolume = ComputeBoundingVolume(*pLocalPositions, indices, elemSubmesh0.offset.value, elemSubmesh0.count.value,
            elemShape.shape_info.vertex_skinning_count.value, lodOffset);
        pSubmeshBoundingArray[0] = BoundingVolumeToBounding(shapeBoundingVolume);

        // サブメッシュバウンディングとシェイプのバウンディングを求める。
        int submeshCount = static_cast<int>(elemMesh.submesh_array.size());
        for (int idxSubmesh = 1; idxSubmesh < submeshCount; ++idxSubmesh)
        {
            const nw::g3d::tool::g3dif::elem_submesh el_submesh = elemMesh.submesh_array[idxSubmesh];
            BoundingVolume submeshBoundingVolume = ComputeBoundingVolume(*pLocalPositions, indices, el_submesh.offset.value, el_submesh.count.value,
                elemShape.shape_info.vertex_skinning_count.value, lodOffset);

            pSubmeshBoundingArray[idxSubmesh] = BoundingVolumeToBounding(submeshBoundingVolume);
            shapeBoundingVolume = MergeBoundingVolume(shapeBoundingVolume, submeshBoundingVolume);
        }

        // シェイプのバウンディング球を計算
        nn::g3d::Bounding* pShapeBounding = pSubmeshBoundingArray + submeshCount;
        *pShapeBounding = BoundingVolumeToBounding(shapeBoundingVolume);
        pRadiusArray[idxMesh] = 0.f;
        for (int idxSubmesh = 0; idxSubmesh < submeshCount; ++idxSubmesh)
        {
            const auto& el_submesh = elemMesh.submesh_array[idxSubmesh];
            float radius = ComputeBoundingSphere(*pLocalPositions, pShapeBounding->center, indices, el_submesh.offset.value,
                el_submesh.count.value, elemShape.shape_info.vertex_skinning_count.value, lodOffset);
            pRadiusArray[idxMesh] = std::max(pRadiusArray[idxMesh], radius);
        }
        pSubmeshBoundingArray += submeshCount + 1;	// サブメッシュ数 + シェイプ分インクリメント
    }
}

float GetAllowableError(float value, int significantDigits)
{
    float absValue = abs(value);
    float maxDigitLog;
    if (absValue == 0.0f)
    {
        maxDigitLog = static_cast<float>(-(significantDigits - 1));
    }
    else
    {
        maxDigitLog = floor(log10(absValue)) - (significantDigits - 1);
    }
    return powf(10, maxDigitLog);
}

bool IsFloatEqual(float value1, float value2, int significantDigits)
{
    // 誤差を許さない場合
    if (significantDigits == -1)
    {
        if (value1 == value2)
        {
            return true;
        }
        return false;
    }
    else
    {
        float allowableError = GetAllowableError(value1, significantDigits);
        if (abs(value1 - value2) <= allowableError)
        {
            return true;
        }
        return false;
    }
}

// 各サブメッシュを LOD 毎に計算して誤差を比較する。
bool HasBoundingPerLod(const nn::g3d::Bounding* pBoundingArray, const nw::g3d::tool::g3dif::elem_shape& elemShape)
{
    const int lodNum = static_cast<int>(elemShape.mesh_array.size());
    if (lodNum == 1)
    {
        return false;	// LOD を持たない場合はフラグを落とす
    }

    const int subMeshNum = static_cast<int>(elemShape.mesh_array[0].submesh_array.size());
    for (auto& mesh : elemShape.mesh_array)
    {
        if (static_cast<int>(mesh.submesh_array.size()) != subMeshNum)
        {
            return false;	// サブメッシュの個数が LOD 間で違う場合は境界不一致が確定するのでフラグを落とす
        }
    }

    const float	allowableErrorRatio = 0.3f;	// 全サブメッシュのうち significantDigits を満たさないサブメッシュの割合
    const int	significantDigits = 2;	// 許容する有効数字
    int inconsistentSubmeshNum = 0;
    for (int idxSubmesh = 0; idxSubmesh < subMeshNum; ++idxSubmesh)
    {
        nn::g3d::Bounding prevBounding = pBoundingArray[idxSubmesh];
        for (int lod = 1; lod < lodNum; ++lod)
        {
            const int boundingNumPerLod = 1 + subMeshNum;	// サブメッシュに続いてシェイプのバウンディングが格納されている
            const int idxOffset = boundingNumPerLod * lod;
            const auto& bounding = pBoundingArray[idxSubmesh + idxOffset];

            // center 誤差比較
            bool isEqual = true;
            isEqual &= IsFloatEqual(prevBounding.center.x, bounding.center.x, significantDigits);
            isEqual &= IsFloatEqual(prevBounding.center.y, bounding.center.y, significantDigits);
            isEqual &= IsFloatEqual(prevBounding.center.z, bounding.center.z, significantDigits);

            // extent 誤差比較
            isEqual &= IsFloatEqual(prevBounding.extent.x, bounding.extent.x, significantDigits);
            isEqual &= IsFloatEqual(prevBounding.extent.y, bounding.extent.y, significantDigits);
            isEqual &= IsFloatEqual(prevBounding.extent.z, bounding.extent.z, significantDigits);

            if (!isEqual)
            {
                ++inconsistentSubmeshNum;
                const float errorRatio = static_cast<float>(inconsistentSubmeshNum) / static_cast<float>(subMeshNum);
                if (errorRatio > allowableErrorRatio)	// 誤差が大きいサブメッシュの比率が allowableErrorRatio を超えるとアーリーアウトさせます
                {
                    return false;
                }
            }
        }
    }

    return true;
}

}
}
