﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinVertex.h>
#include <g3dif/Model.h>
#include <nn/util/util_FloatFormat.h>
#include <algorithm>
#include <util/UtilMath.h>
#include <util/UtilError.h>

#include <GfxHelper.h>

#include <iostream>

namespace nn {
namespace g3dTool {
namespace {
NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    vtx_attrib, quantize_type, elem_size,
    0,						// quantize_type_none の時は 0 を返すものとする。
    1, 1, 1, 1, 1, 1,       // 8
    1,                      // 4_4
    2, 2, 2, 2, 2, 2, 2,    // 16
    1, 1, 1, 1, 1, 1,       // 8_8
    4, 4, 4,                // 32
    2, 2, 2, 2, 2, 2, 2,    // 16_16
    4,                      // 10_11_11
    1, 1, 1, 1, 1, 1,       // 8_8_8_8
    4, 4, 4, 4,             // 10_10_10_2
    4, 4, 4,                // 32_32
    2, 2, 2, 2, 2, 2, 2,    // 16_16_16_16
    4, 4, 4,                // 32_32_32
    4, 4, 4                 // 32_32_32_32
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    vtx_attrib, quantize_type, elem_alignment_size,
    4,						// quantize_type_none の attribute アライメントは nvn では 4byte
    1, 1, 1, 1, 1, 1,       // 8
    1,                      // 4_4
    2, 2, 2, 2, 2, 2, 2,    // 16
    1, 1, 1, 1, 1, 1,       // 8_8
    4, 4, 4,                // 32
    2, 2, 2, 2, 2, 2, 2,    // 16_16
    4,                      // 10_11_11
    1, 1, 1, 1, 1, 1,       // 8_8_8_8
    4, 4, 4, 4,             // 10_10_10_2
    4, 4, 4,                // 32_32
    2, 2, 2, 2, 2, 2, 2,    // 16_16_16_16
    4, 4, 4,             // 32_32_32
    4, 4, 4              // 32_32_32_32
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    vtx_attrib, quantize_type, cmp_count,
    0,						// quantize_type_none の時は 0 を返すものとする。
    1, 1, 1, 1, 1, 1,       // 8
    1,                      // 4_4
    1, 1, 1, 1, 1, 1, 1,    // 16
    2, 2, 2, 2, 2, 2,       // 8_8
    1, 1, 1,                // 32
    2, 2, 2, 2, 2, 2, 2,    // 16_16
    1,                      // 10_11_11
    4, 4, 4, 4, 4, 4,       // 8_8_8_8
    1, 1, 1, 1,             // 10_10_10_2
    2, 2, 2,                // 32_32
    4, 4, 4, 4, 4, 4, 4,    // 16_16_16_16
    3, 3, 3,                // 32_32_32
    4, 4, 4                 // 32_32_32_32
    );

typedef nn::util::FloatFormat<0, 5, 5> Float10;
typedef nn::util::FloatFormat<0, 5, 6> Float11;
typedef nn::util::FloatFormat<1, 5, 10> Float16;
}


//--------------------------------------------------------------------------------------------------

void BinVtxAttrib::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vtx_attrib& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    pCtx->SetStr(m_pElem->name.value.c_str());
}

void BinVtxAttrib::CalculateSize()
{
    // ヒープは親から割り当てられている。
    // インターリーヴの関係上、頂点バッファの実データサイズは ResVertex で計算される。
}

void BinVtxAttrib::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResVertexAttrData& vtxAttrib = *GetPtr<nn::g3d::ResVertexAttrData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    pCtx->LinkStr( &vtxAttrib.pName, nn::util::string_view( m_pElem->name.value.c_str() ) );
    vtxAttrib.bufferIndex = static_cast<uint8_t>( m_BufferIndex );
    vtxAttrib.offset = static_cast<uint16_t>(m_VtxOffset);

    m_pDst = nw::g3d::tool::util::AddOffset( pCtx->GetMemBlockPtr( Context::MemBlockType_VtxStream ), m_DataOffset + m_VtxOffset );
    void* pDst = m_pDst;

    const void* pSrc = m_pElem->stream.rawdata.get();
    int vtxCount = m_pElem->count.value;
    int attribCount = m_pElem->type.value % 4 + 1;
    int size = attribCount * sizeof(float);

    const int uint_32_table[] =
    {
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32_32_32
    };

    const int int_32_table[] =
    {
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32_32_32
    };

    const int float_32_table[] =
    {
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32_32,
        nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32_32_32
    };

    if (m_pElem->quantize_type.value == nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_none ||
        m_pElem->quantize_type.value == uint_32_table[attribCount - 1] ||
        m_pElem->quantize_type.value == int_32_table[attribCount - 1] ||
        m_pElem->quantize_type.value == float_32_table[attribCount - 1])
    {
        vtxAttrib.format = GetGfxAttributeFormat( m_pElem->type.value );

        for (int idxVertex = 0; idxVertex < vtxCount; ++idxVertex)
        {
            memcpy(pDst, pSrc, size);
            memset(nw::g3d::tool::util::AddOffset(pDst, size), 0, m_Padding);
            pDst = nw::g3d::tool::util::AddOffset(pDst, m_VtxStride);
            pSrc = nw::g3d::tool::util::AddOffset(pSrc, size);
        }
        // ゼロクリア
        for (int idxVertex = vtxCount; idxVertex < static_cast<int>(m_VtxMax); ++idxVertex)
        {
            memset(pDst, 0, size);
            pDst = nw::g3d::tool::util::AddOffset(pDst, m_VtxStride);
            pSrc = nw::g3d::tool::util::AddOffset(pSrc, size);
        }
    }
    else
    {
        vtxAttrib.format = GetGfxAttributeFormatQuantized( m_pElem->quantize_type.value );

        if (m_pElem->type.value >= nw::g3d::tool::g3dif::elem_vtx_attrib::type_float)
        {
            Quantize<float>(pDst, static_cast<const float*>(pSrc), vtxCount, size);
        }
        else
        {
            Quantize<int>(pDst, static_cast<const int*>(pSrc), vtxCount, size);
        }
    }
}

template<typename T>
void BinVtxAttrib::Quantize(void* pDst, const T* pSrc, int numVertex, int size)
{
    int cmpCount = ToEnum_cmp_count(m_pElem->quantize_type.value);
    for (int idxVertex = 0; idxVertex < numVertex; ++idxVertex)
    {
        void* postDst = pDst;
        bool isQuantized = true;
        switch (m_pElem->quantize_type.value)
        {
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8_8_8_8:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToUnorm8(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint8_t));
                }
                else
                {
                    uint8_t* dst = static_cast<uint8_t*>(postDst);
                    *dst = (0x1 << 8) - 1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint8_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_8_8_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_8_8_8_8:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToUInt8(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint8_t));
                }
                else
                {
                    uint8_t* dst = static_cast<uint8_t*>(postDst);
                    *dst = 0x1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint8_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_8_8_8_8:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToSnorm8(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint8_t));
                }
                else
                {
                    uint8_t* dst = static_cast<uint8_t*>(postDst);
                    *dst = (0x1 << 7) - 1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint8_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_8_8_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_8_8_8_8:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToSInt8(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint8_t));
                }
                else
                {
                    uint8_t* dst = static_cast<uint8_t*>(postDst);
                    *dst = 0x1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint8_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16_16_16_16:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToUnorm16(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint16_t));
                }
                else
                {
                    uint16_t* dst = static_cast<uint16_t*>(postDst);
                    *dst = (0x1 << 16) - 1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint16_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_16_16_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_to_float_16_16_16_16:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToUInt16(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint16_t));
                }
                else
                {
                    uint16_t* dst = static_cast<uint16_t*>(postDst);
                    *dst = 0x1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint16_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_16_16_16_16:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToSnorm16(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint16_t));
                }
                else
                {
                    uint16_t* dst = static_cast<uint16_t*>(postDst);
                    *dst = (0x1 << 15) - 1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint16_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_16_16_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_to_float_16_16_16_16:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToSInt16(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint16_t));
                }
                else
                {
                    uint16_t* dst = static_cast<uint16_t*>(postDst);
                    *dst = 0x1;
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint16_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_16_16_16_16:
            for (int i = 0; i < cmpCount; ++i)
            {
                if (static_cast<int>(sizeof(T) * i) < size)
                {
                    isQuantized = QuantizeToFloat16(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                    if (!isQuantized)
                    {
                        break;
                    }
                    postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint16_t));
                }
                else
                {
                    uint16_t* dst = static_cast<uint16_t*>(postDst);
                    *dst = static_cast<uint16_t>( Float16::Encode(1.0f) );
                    postDst = nw::g3d::tool::util::AddOffset(dst, sizeof(uint16_t));
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_32_32_32_32:
            for (int i = 0; i < cmpCount; ++i)
            {
                isQuantized = QuantizeToUInt32(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                if (!isQuantized)
                {
                    break;
                }
                postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint32_t));
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_32_32_32_32:
            for (int i = 0; i < cmpCount; ++i)
            {
                isQuantized = QuantizeToSInt32(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                if (!isQuantized)
                {
                    break;
                }
                postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint32_t));
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32_32:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_32_32_32_32:
            for (int i = 0; i < cmpCount; ++i)
            {
                isQuantized = QuantizeToFloat32(postDst, *nw::g3d::tool::util::AddOffset<T>(pSrc, sizeof(T) * i));
                if (!isQuantized)
                {
                    break;
                }
                postDst = nw::g3d::tool::util::AddOffset(pDst, (i + 1) * sizeof(uint32_t));
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_4_4:
            isQuantized = QuantizeToUnorm4_4(postDst, pSrc);
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint8_t));
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_float_10_11_11:
            isQuantized = QuantizeToFloat10_11_11(pDst, pSrc);
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint32_t));
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_10_10_10_2:
            if (size <= sizeof(float) * 3)
            {
                isQuantized = QuantizeToUnorm10_10_10_2_Type3(postDst, pSrc);
            }
            else
            {
                isQuantized = QuantizeToUnorm10_10_10_2_Type4(postDst, pSrc);
            }
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint32_t));
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_uint_10_10_10_2:

            if (size <= sizeof(float) * 3)
            {
                isQuantized = QuantizeToUInt10_10_10_2_Type3(postDst, pSrc);
            }
            else
            {
                isQuantized = QuantizeToUInt10_10_10_2_Type4(postDst, pSrc);
            }
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint32_t));
            break;

        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_snorm_10_10_10_2:
            if (size <= sizeof(float) * 3)
            {
                isQuantized = QuantizeToSnorm10_10_10_2_Type3(postDst, pSrc);
            }
            else
            {
                isQuantized = QuantizeToSnorm10_10_10_2_Type4(postDst, pSrc);
            }
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint32_t));
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_sint_10_10_10_2:

            if (size <= sizeof(float) * 3)
            {
                isQuantized = QuantizeToSInt10_10_10_2_Type3(postDst, pSrc);
            }
            else
            {
                isQuantized = QuantizeToSInt10_10_10_2_Type4(postDst, pSrc);
            }
            postDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(uint32_t));
            break;
        }

        // 指定された量子化型で量子化できない場合はエラーとする
        if (!isQuantized)
        {
            THROW_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_QUANTIZE_VALUE, "Invalid quantization format bound at %hs, vertexIndex=%d.\n", m_pElem->name.value.c_str(), idxVertex);
        }

        // パディングをゼロクリア
        memset(postDst, 0, m_Padding);

        pSrc = nw::g3d::tool::util::AddOffset<T>(pSrc, size);
        pDst = nw::g3d::tool::util::AddOffset(pDst, m_VtxStride);
    }

    // ゼロクリア
    for (int idxVertex = numVertex; idxVertex < static_cast<int>(m_VtxMax); ++idxVertex)
    {
        memset(pDst, 0,  ToEnum_elem_size(m_pElem->quantize_type.value) * ToEnum_cmp_count(m_pElem->quantize_type.value));

        pSrc = nw::g3d::tool::util::AddOffset<T>(pSrc, size);
        pDst = nw::g3d::tool::util::AddOffset(pDst, m_VtxStride);
    }
}

template<>
bool BinVtxAttrib::QuantizeToUnorm4_4(void* pDst, const float* src)
{
    float src0 = *src;
    float src1 = *(src+1);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]", src0, src1);
        return false;
    }

    // 0 - 127
    uint8_t* dst = static_cast<uint8_t*>(pDst);
    *dst = ((nw::g3d::tool::util::Round<uint8_t>(src0 * ((0x1 << 4) - 1)) & 0xF) << 4) | (nw::g3d::tool::util::Round<uint8_t>(src1 * ((0x1 << 4) - 1)) & 0xF);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUnorm8(void* pDst, float src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 1.0]", src);
        return false;
    }

    // 0 - 255
    uint8_t* dst = static_cast<uint8_t*>(pDst);
    *dst = nw::g3d::tool::util::Round<uint8_t>(src * static_cast<float>((0x1 << (sizeof(uint8_t) * 8)) - 1));
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUnorm10_10_10_2_Type3(void* pDst, const float* src)
{
    float src0 = *src;
    float src1 = *(src+1);
    float src2 = *(src+2);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]", src0, src1, src2);
        return false;
    }

    // 0 - 1023
    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        ((nw::g3d::tool::util::Round<uint32_t>(src0 * ((0x1 << 10) - 1)) & 0x3FF))       |
        ((nw::g3d::tool::util::Round<uint32_t>(src1 * ((0x1 << 10) - 1)) & 0x3FF) << 10) |
        ((nw::g3d::tool::util::Round<uint32_t>(src2 * ((0x1 << 10) - 1)) & 0x3FF) << 20) |
        ((nw::g3d::tool::util::Round<uint32_t>(1.0f * ((0x1 <<  2) - 1)) & 0x3)   << 30);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUnorm10_10_10_2_Type4(void* pDst, const float* src)
{
    float src0 = *src;
    float src1 = *(src+1);
    float src2 = *(src+2);
    float src3 = *(src+3);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, 0.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src3, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]", src0, src1, src2, src3);
        return false;
    }

    // 0 - 1023
    // 0 - 3
    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        ((nw::g3d::tool::util::Round<uint32_t>(src0 * ((0x1 << 10) - 1)) & 0x3FF))       |
        ((nw::g3d::tool::util::Round<uint32_t>(src1 * ((0x1 << 10) - 1)) & 0x3FF) << 10) |
        ((nw::g3d::tool::util::Round<uint32_t>(src2 * ((0x1 << 10) - 1)) & 0x3FF) << 20) |
        ((nw::g3d::tool::util::Round<uint32_t>(src3 * ((0x1 <<  2) - 1)) & 0x3)   << 30);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSnorm10_10_10_2_Type3(void* pDst, const float* src)
{
    float src0 = *src;
    float src1 = *(src+1);
    float src2 = *(src+2);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, -1.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, -1.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, -1.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [-1.0, 1.0]"
            L"value: %f, bound [-1.0, 1.0]"
            L"value: %f, bound [-1.0, 1.0]", src0, src1, src2);
        return false;
    }

    // -511 - 511
    int16_t src0_temp = nw::g3d::tool::util::Round<int16_t>(src0 * ((0x1 << (10 - 1)) - 1));
    int16_t src1_temp = nw::g3d::tool::util::Round<int16_t>(src1 * ((0x1 << (10 - 1)) - 1));
    int16_t src2_temp = nw::g3d::tool::util::Round<int16_t>(src2 * ((0x1 << (10 - 1)) - 1));
    int16_t src3_temp = 0x1;

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        (src0_temp & 0x3FF)       |
        (src1_temp & 0x3FF) << 10 |
        (src2_temp & 0x3FF) << 20 |
        (src3_temp & 0x3)   << 30;
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSnorm10_10_10_2_Type4(void* pDst, const float* src)
{
    float src0 = *src;
    float src1 = *(src+1);
    float src2 = *(src+2);
    float src3 = *(src+3);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, -1.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, -1.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, -1.0f, 1.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src3, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [-1.0, 1.0]"
            L"value: %f, bound [-1.0, 1.0]"
            L"value: %f, bound [-1.0, 1.0]"
            L"value: %f, bound [0.0, 1.0]", src0, src1, src2, src3);
        return false;
    }

    // -511 - 511
    // 0 - 3
    int16_t src0_temp = nw::g3d::tool::util::Round<int16_t>(src0 * ((0x1 << (10 - 1)) - 1));
    int16_t src1_temp = nw::g3d::tool::util::Round<int16_t>(src1 * ((0x1 << (10 - 1)) - 1));
    int16_t src2_temp = nw::g3d::tool::util::Round<int16_t>(src2 * ((0x1 << (10 - 1)) - 1));
    uint16_t src3_temp = nw::g3d::tool::util::Round<uint16_t>(src3 * ((0x1 <<  2) - 1));

    uint32_t* dst = static_cast<uint32_t*>(pDst);

    *dst =
        (src0_temp & 0x3FF)       |
        (src1_temp & 0x3FF) << 10 |
        (src2_temp & 0x3FF) << 20 |
        (src3_temp & 0x3)   << 30;

    //*dst &= 0x3FFFFFFF; // 00

    //*dst |= 0x3FFFFFFF; // 00
    //*dst |= 0x4FFFFFFF; // 01
    //*dst |= 0x8FFFFFFF; // 10
    //*dst |= 0xC0000000; // 11

    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUnorm16(void* pDst, float src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, 0.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 1.0]", src);
        return false;
    }

    // 0 - 65535
    uint16_t* dst = static_cast<uint16_t*>(pDst);
    float temp = src * ((0x1 << sizeof(uint16_t) * 8) - 1);
    *dst = nw::g3d::tool::util::Round<uint16_t>(temp);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUInt8(void* pDst, int src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, 0, (0x1 << 8) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 255]", src);
        return false;
    }

    uint8_t* dst = static_cast<uint8_t*>(pDst);
    *dst = static_cast<uint8_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUInt10_10_10_2_Type3(void* pDst, const int* src)
{
    int src0 = *src;
    int src1 = *(src+1);
    int src2 = *(src+2);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0, (0x1 << 10) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0, (0x1 << 10) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, 0, (0x1 << 10) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 1023]"
            L"value: %d, bound [0, 1023]"
            L"value: %d, bound [0, 1023]", src0, src1, src2);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        (static_cast<uint32_t>(src0 & 0x3FF))       |
        (static_cast<uint32_t>(src1 & 0x3FF) << 10) |
        (static_cast<uint32_t>(src2 & 0x3FF) << 20) |
        (static_cast<uint32_t>(0x1  & 0x3)   << 30);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUInt10_10_10_2_Type4(void* pDst, const int* src)
{
    int src0 = *src;
    int src1 = *(src+1);
    int src2 = *(src+2);
    int src3 = *(src+3);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0, (0x1 << 10) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0, (0x1 << 10) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, 0, (0x1 << 10) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src3, 0, (0x1 << 2) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 1023]"
            L"value: %d, bound [0, 1023]"
            L"value: %d, bound [0, 1023]"
            L"value: %d, bound [0, 3]", src0, src1, src2, src3);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        (static_cast<uint32_t>(src0 & 0x3FF))       |
        (static_cast<uint32_t>(src1 & 0x3FF) << 10) |
        (static_cast<uint32_t>(src2 & 0x3FF) << 20) |
        (static_cast<uint32_t>(src3 & 0x3)   << 30);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSInt10_10_10_2_Type3(void* pDst, const int* src)
{
    int src0 = *src;
    int src1 = *(src+1);
    int src2 = *(src+2);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, -(0x1 << 9), (0x1 << 9) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, -(0x1 << 9), (0x1 << 9) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, -(0x1 << 9), (0x1 << 9) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-512, 511]"
            L"value: %d, bound [-512, 511]"
            L"value: %d, bound [-512, 511]", src0, src1, src2);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        (src0 & 0x3FF)       |
        (src1 & 0x3FF) << 10 |
        (src2 & 0x3FF) << 20 |
         0x1           << 30;
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSInt10_10_10_2_Type4(void* pDst, const int* src)
{
    int src0 = *src;
    int src1 = *(src+1);
    int src2 = *(src+2);
    int src3 = *(src+3);

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, -(0x1 << 9), (0x1 << 9) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, -(0x1 << 9), (0x1 << 9) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, -(0x1 << 9), (0x1 << 9) - 1) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src3, -(0x1 << 1), (0x1 << 1) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-512, 511]"
            L"value: %d, bound [-512, 511]"
            L"value: %d, bound [-512, 511]"
            L"value: %d, bound [-2, 1]", src0, src1, src2, src3);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        (src0 & 0x3FF)       |
        (src1 & 0x3FF) << 10 |
        (src2 & 0x3FF) << 20 |
        (src3 & 0x3)   << 30;
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUInt16(void* pDst, int src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, 0, (0x1 << 16) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 65535]", src);
        return false;
    }

    uint16_t* dst = static_cast<uint16_t*>(pDst);
    *dst = static_cast<uint16_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToUInt32(void* pDst, int src)
{
    // NOTE: 現状では uint32_t の範囲は対応していないので、int の範囲でチェックする。
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, 0, (0x1 << 16) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 2147483647]", src);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst = static_cast<uint32_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSInt8(void* pDst, int src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -(0x1 << 7), (0x1 << 7) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-128, 127]", src);
        return false;
    }

    int8_t* dst = static_cast<int8_t*>(pDst);
    *dst = static_cast<int8_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSInt16(void* pDst, int src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -(0x1 << 15), (0x1 << 15) - 1))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-32768, 32767]", src);
        return false;
    }

    int16_t* dst = static_cast<int16_t*>(pDst);
    *dst = static_cast<int16_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSInt32(void* pDst, int src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -2147483647, 2147483647))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-2147483647, 2147483647]", src);
        return false;
    }

    int32_t* dst = static_cast<int32_t*>(pDst);
    *dst = static_cast<int32_t>(src);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSnorm8(void* pDst, float src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -1.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [-1.0, 1.0]", src);
        return false;
    }

    // -127 - 127
    int8_t* dst = static_cast<int8_t*>(pDst);
    float temp = src * ((0x1 << (sizeof(int8_t) * 8 - 1)) - 1);
    *dst = nw::g3d::tool::util::Round<int8_t>(temp);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToSnorm16(void* pDst, float src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -1.0f, 1.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [-1.0, 1.0]", src);
        return false;
    }

    // -32767 - 32767
    int16_t* dst = static_cast<int16_t*>(pDst);
    float temp = src * ((0x1 << (sizeof(int16_t) * 8 -1)) - 1);
    *dst = nw::g3d::tool::util::Round<int16_t>(temp);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToFloat10_11_11(void* pDst, const float* src)
{
    // zyx のパックなので反転して代入する。
    float src0 = *(src+2);
    float src1 = *(src+1);
    float src2 = *src;

    if (!nw::g3d::tool::util::CheckGEMinLEMax(src0, 0.0f, 64512.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src1, 0.0f, 65024.0f) ||
        !nw::g3d::tool::util::CheckGEMinLEMax(src2, 0.0f, 65024.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [0.0, 64512.0]"
            L"value: %f, bound [0.0, 65024.0]"
            L"value: %f, bound [0.0, 65024.0]", src0, src1, src2);
        return false;
    }

    uint32_t* dst = static_cast<uint32_t*>(pDst);
    *dst =
        Float10::Encode(src0) << 22 |
        Float11::Encode(src1) << 11 |
        Float11::Encode(src2);
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToFloat16<float>(void* pDst, float src)
{
    if (!nw::g3d::tool::util::CheckGEMinLEMax(src, -65504.0f, 65504.0f))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %f, bound [-65504.0, 65504.0]", src);
        return false;
    }

    uint16_t* dst = static_cast<uint16_t*>(pDst);
    *dst = static_cast<uint16_t>( Float16::Encode(src) );
    return true;
}

template<>
bool BinVtxAttrib::QuantizeToFloat32<float>(void* pDst, float src)
{
    float* dst = static_cast<float*>(pDst);
    *dst = src;
    return true;
}

template<>
bool BinVtxAttrib::DequantizeFromUnorm8(float* pDst, uint8_t src)
{
    uint16_t srcTemp = static_cast<uint16_t>(src);
    if (!nw::g3d::tool::util::CheckGEMinLEMax(srcTemp, static_cast<uint16_t>(0), static_cast<uint16_t>((0x1 << 8) - 1)))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 255]", src);
        return false;
    }

    float* dst = static_cast<float*>(pDst);
    *dst = static_cast<float>(src) / static_cast<float>((0x1 << (sizeof(uint8_t) * 8)) - 1);
    return true;
}

template<>
bool BinVtxAttrib::DequantizeFromUnorm16(float* pDst, uint16_t src)
{
    uint32_t srcTemp = static_cast<uint32_t>(src);
    if (!nw::g3d::tool::util::CheckGEMinLEMax(srcTemp, static_cast<uint32_t>(0), static_cast<uint32_t>((0x1 << 16) - 1)))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [0, 65535]", src);
        return false;
    }

    float* dst = static_cast<float*>(pDst);
    *dst = static_cast<float>(src) / static_cast<float>((0x1 << (sizeof(uint16_t) * 8)) - 1);
    return true;
}

template<>
bool BinVtxAttrib::DequantizeFromSnorm10_10_10_2(float* dst0, float* dst1, float* dst2, float* dst3, uint32_t src)
{
    // 一度最上位ビットに左シフトすることで
    // 右シフトするときに符号ビットを立てていく
    int16_t src0 = static_cast<int16_t>(src <<  6) >> 6;
    int16_t src1 = static_cast<int16_t>(src >>  4) >> 6;
    int16_t src2 = static_cast<int16_t>(src >> 14) >> 6;
    int8_t  src3 = static_cast<int8_t>(src >> 22) >> 6;

    if (!nw::g3d::tool::util::CheckGEMinLEMax<int16_t>(src0, -511, 511) ||
        !nw::g3d::tool::util::CheckGEMinLEMax<int16_t>(src1, -511, 511) ||
        !nw::g3d::tool::util::CheckGEMinLEMax<int16_t>(src2, -511, 511) ||
        !nw::g3d::tool::util::CheckGEMinLEMax<int8_t>(src3, 0, 3))
    {
        PRINT_SYSTEM_LOG(
            "Invalid quantization format bound.\n"
            L"value: %d, bound [-511, 511]"
            L"value: %d, bound [-511, 511]"
            L"value: %d, bound [-511, 511]"
            L"value: %d, bound [0, 3]", src0, src1, src2, src3);
        return false;
    }

    *dst0 = std::max(static_cast<float>(src0) / static_cast<float>((0x1 << (10 - 1)) - 1), -1.0f);
    *dst1 = std::max(static_cast<float>(src1) / static_cast<float>((0x1 << (10 - 1)) - 1), -1.0f);
    *dst2 = std::max(static_cast<float>(src2) / static_cast<float>((0x1 << (10 - 1)) - 1), -1.0f);
    *dst3 = static_cast<float>(src3) / static_cast<float>(0x1 << 2);
    return true;
}

//--------------------------------------------------------------------------------------------------

void BinVertex::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vertex& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    // 頂点バッファーから参照される頂点属性をビットフラグで管理
    int flagEnable = 0;
    for (auto iter = elem.vtx_buffer_array.cbegin();
        iter != elem.vtx_buffer_array.cend(); ++iter)
    {
        for (auto iterInput = iter->input_array.cbegin();
            iterInput != iter->input_array.cend(); ++iterInput)
        {
            flagEnable |= (0x1 << iterInput->attrib_index.value);
        }
    }

    // 頂点バッファーから参照される頂点属性のみをバイナリーに含める
    int attribIndex = 0, useAttribCount = 0;
    m_AttribTable.resize(elem.vtx_attrib_array.size(), -1);
    for (auto iter = elem.vtx_attrib_array.cbegin();
        iter != elem.vtx_attrib_array.cend();
        ++iter, ++attribIndex)
    {
        if (((flagEnable >> attribIndex) & 0x1) != 0)
        {
            m_AttribTable[attribIndex] = useAttribCount;
            ++useAttribCount;
        }
    }

    std::vector<const nw::g3d::tool::g3dif::elem_vtx_attrib*> vtx_attrib_array;
    m_DicVtxAttrib.Build(pCtx, useAttribCount);
    attribIndex = 0;
    for (auto iter = elem.vtx_attrib_array.cbegin();
        iter != elem.vtx_attrib_array.cend();
        ++iter, ++attribIndex)
    {
        int useIndex = m_AttribTable[attribIndex];
        if (useIndex != -1)
        {
            m_DicVtxAttrib.SetName(useIndex, iter->name.value);
            vtx_attrib_array.push_back(&(*iter));
        }
    }

    m_VtxAttribArray.resize(vtx_attrib_array.size());
    SetParentBlockArray(m_VtxAttribArray, this);
    BuildArray(pCtx, m_VtxAttribArray, vtx_attrib_array);

    // VtxBuffer を VtxAttrib に分解します。
    SplitInput( pCtx, elem );

    // VtxBuffer のインデクスを設定します。
    int vtxBufferIndex = 0;
    for (auto iter = elem.vtx_buffer_array.cbegin();
        iter != elem.vtx_buffer_array.cend(); ++iter, ++vtxBufferIndex)
    {
        for (auto iterInput = iter->input_array.cbegin();
            iterInput != iter->input_array.cend(); ++iterInput)
        {
            BinVtxAttrib& attrib = m_VtxAttribArray[m_AttribTable[iterInput->attrib_index.value]];
            attrib.SetBufferIndex(vtxBufferIndex);
        }
    }
}

struct PairLess
{
public:
    bool operator()(const std::pair<int, int> lhs, const std::pair<int, int> rhs) const {
        return lhs.second > rhs.second;
    }
};

void BinVertex::SplitInput(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vertex& elem)
{
    std::vector<int> attribArray;
    attribArray.resize(elem.vtx_attrib_array.size(), -1);

    m_VtxBufferArray.reserve(elem.vtx_buffer_array.size());
    int vtxbufferIndex = 0;
    for (auto iter = elem.vtx_buffer_array.cbegin(); iter != elem.vtx_buffer_array.cend(); ++iter, ++vtxbufferIndex)
    {
        // VBO 内部で必要なアライメントが多い順にソートします。
        int inputIndex = 0;
        std::vector<std::pair<int, int>> sortedInputArray;
        sortedInputArray.reserve(iter->input_array.size());
        for (auto iterInput = iter->input_array.cbegin(); iterInput != iter->input_array.cend(); ++iterInput, ++inputIndex)
        {
            const nw::g3d::tool::g3dif::elem_vtx_attrib& attrib = *iterInput->attrib;

            if (attribArray[iterInput->attrib_index.value] == -1)
            {
                attribArray[iterInput->attrib_index.value] = inputIndex;
            }
            else
            {
                // 複数のバッファから同一アトリビュートが参照されるのには未対応です。
                THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_ATTRIB_INDEX, "Identifier_DuplicatedAttributeIndex",
                    iterInput->attrib_index.value, vtxbufferIndex);
            }

            size_t elementSize = 0;
            size_t cmpCount = 0;
            if (attrib.quantize_type.value == nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_none)
            {
                elementSize = sizeof(float);
                cmpCount = (attrib.type.value % 4) + 1;
            }
            else
            {
                elementSize = ToEnum_elem_size(attrib.quantize_type.value);
                cmpCount = ToEnum_cmp_count(attrib.quantize_type.value);
            }

            sortedInputArray.push_back(std::make_pair(inputIndex, static_cast<int>( elementSize * cmpCount ) ) );
        }

        if( !pCtx->IsDisableAttributeAlignmentSort() )
        {
            std::sort(sortedInputArray.begin(), sortedInputArray.end(), PairLess());
        }

        // アライメントが多い順に並べ、必要な箇所にアライメントを挿入します。
        // 全体のサイズ (stride) に関しては最大のアライメントに合わせます。
        int numVtx = 0;
        size_t offsetVtx = 0;
        size_t maxElementSize = 0;
        for (auto iterInput = sortedInputArray.cbegin(); iterInput != sortedInputArray.cend(); ++iterInput)
        {
            const nw::g3d::tool::g3dif::elem_input& input = iter->input_array[iterInput->first];

            const nw::g3d::tool::g3dif::elem_vtx_attrib& attrib = *input.attrib;
            BinVtxAttrib& vtxAttrib = m_VtxAttribArray[m_AttribTable[input.attrib_index.value]];

            size_t alignmentSize = 0;
            alignmentSize = ToEnum_elem_alignment_size(attrib.quantize_type.value);

            // アライメントが適切でない場合はアライメントを調整して、パディングを挿入する。
            if (offsetVtx % alignmentSize != 0)
            {
                size_t padding = nw::g3d::tool::util::Align(offsetVtx, alignmentSize) - offsetVtx;

                const nw::g3d::tool::g3dif::elem_input& preInput = iter->input_array[(iterInput - 1)->first];
                BinVtxAttrib& preVtxAttrib = m_VtxAttribArray[m_AttribTable[preInput.attrib_index.value]];

                preVtxAttrib.SetPaddingRight(padding);
                offsetVtx += padding;
            }

            vtxAttrib.SetVtxOffset(offsetVtx);

            offsetVtx += iterInput->second;

            // 最も数が多い頂点を調べる
            numVtx = std::max(numVtx, attrib.count.value);
            // 最もアライメントが大きい頂点を調べる
            maxElementSize = std::max(alignmentSize, maxElementSize);
        }

        size_t padding = nw::g3d::tool::util::Align(offsetVtx, maxElementSize) - offsetVtx;
        const nw::g3d::tool::g3dif::elem_input& backInput = iter->input_array[sortedInputArray.back().first];
        BinVtxAttrib& backVtxAttrib = m_VtxAttribArray[m_AttribTable[backInput.attrib_index.value]];
        backVtxAttrib.SetPaddingRight(padding);

        m_VtxBufferArray.push_back(std::make_pair(offsetVtx + padding, numVtx));
    }
}

void BinVertex::CalculateSize()
{
    // ResVertex のサイズは親の ResModelData から与えられます。

    m_Chunk[ChunkType_VtxAttr].size = sizeof(nn::g3d::ResVertexAttrData) * m_VtxAttribArray.size();
    m_Chunk[ChunkType_GfxBuffArray].size = GetGfxMaxImplDataSize<nn::gfx::BufferImplData>() * m_VtxBufferArray.size();
    m_Chunk[ChunkType_GfxBuffInfoDataArray].size = sizeof( nn::gfx::BufferInfoData ) * m_VtxBufferArray.size();
    m_Chunk[ChunkType_GfxVtxBuffStateInfoDataArray].size = sizeof( nn::gfx::VertexBufferStateInfoData ) * m_VtxBufferArray.size();
    m_Chunk[ChunkType_GfxBuffPtrArray].size = sizeof( uint64_t ) * m_VtxBufferArray.size();	// 32/64bit でサイズを変えないようにするために 8byte で確保する

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));


    // インターリーヴされた頂点ストリームのデータサイズを計算。
    size_t size = 0;
    for (auto iter = m_VtxBufferArray.begin(); iter != m_VtxBufferArray.end(); ++iter)
    {
        size += nw::g3d::tool::util::Align(iter->first * iter->second, ALIGNMENT_VERTEX_STREAM);
    }
    SetBlockSize(Context::MemBlockType_VtxStream, size);
}

void BinVertex::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    // VtxAttrib にオフセットをセットする。
    ptrdiff_t ofsVtxAttrib = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_VtxAttr].offset;
    for (auto iter = m_VtxAttribArray.begin(); iter != m_VtxAttribArray.end(); ++iter)
    {
        iter->SetStructOffset(ofsVtxAttrib);
        ofsVtxAttrib += sizeof(nn::g3d::ResVertexAttrData);
    }

    // BufferIndex と BufferOffset をセットする。
    ptrdiff_t ofsData = GetBlockOffset(Context::MemBlockType_VtxStream);
    int index = 0;
    for (auto iter = m_pElem->vtx_buffer_array.cbegin();
        iter != m_pElem->vtx_buffer_array.cend(); ++iter, ++index)
    {
        auto vertexSize = m_VtxBufferArray[index];
        for (auto iterInput = iter->input_array.cbegin();
            iterInput != iter->input_array.cend(); ++iterInput)
        {
            BinVtxAttrib& attrib = m_VtxAttribArray[m_AttribTable[iterInput->attrib_index.value]];
            attrib.SetDataOffset(ofsData);
            attrib.SetVtxStride(vertexSize.first);
            attrib.SetVtxMax(vertexSize.second);
        }
        ofsData += nw::g3d::tool::util::Align(vertexSize.first * vertexSize.second, ALIGNMENT_VERTEX_STREAM);
    }
}

void BinVertex::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResVertexData& vertex = *GetPtr<nn::g3d::ResVertexData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    vertex.count = 0;
    if (m_VtxAttribArray.size() != 0)
    {
        vertex.count = static_cast<uint32_t>(m_VtxAttribArray[0].m_pElem->count.value);
    }
    vertex.influenceCount = static_cast<uint8_t>(m_SkinningCount);
    vertex.blockHeader.signature.SetPacked( nn::g3d::ResVertex::Signature );
    vertex.pUserPtr.Set(nullptr);

    vertex.vertexAttrCount = static_cast<uint8_t>(m_VtxAttribArray.size());

    // VertexAttrib
    pCtx->LinkPtr( &vertex.pVertexAttrArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_VtxAttr ].offset));
    m_DicVtxAttrib.ConvertData(pCtx, vertex.pVertexAttrDic, m_VtxAttribArray);

    // VtxBufferInfodata, state infor data
    ConvertVtxBuffer(vertex, pCtx);

    // VtxBuffer Heap
    pCtx->LinkPtr( &vertex.pVertexBufferArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxBuffArray ].offset));

    // VtxBufferPtr Heap
    pCtx->LinkPtr( &vertex.pVertexBufferPtrArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_GfxBuffPtrArray].offset));

    // Memory Pool Heap
    pCtx->LinkPtr( &vertex.pMemoryPool, pCtx->GetMemBlockPtr( Context::MemBlockType_GfxMemPool ) );
}

void BinVertex::ConvertVtxBuffer(nn::g3d::ResVertexData &vertex, std::shared_ptr<Context> pCtx)
{
    vertex.vertexBufferCount = static_cast<uint8_t>(m_VtxBufferArray.size());

    nn::gfx::BufferInfoData* pVtxBuffer =
        GetPtr< nn::gfx::BufferInfoData >(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxBuffInfoDataArray ].offset);
    pCtx->LinkPtr( &vertex.pVertexBufferInfoArray, pVtxBuffer );


    nn::gfx::VertexBufferStateInfoData* pVertexBufStateInfoData =
        GetPtr< nn::gfx::VertexBufferStateInfoData >(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxVtxBuffStateInfoDataArray ].offset);
    pCtx->LinkPtr( &vertex.pVertexBufferStateInfoArray, pVertexBufStateInfoData );


    int index = 0;
    int memoryPoolSize = 0;
    void* pData = GetPtr(pCtx, Context::MemBlockType_VtxStream, 0);
    for (auto iter = m_VtxBufferArray.cbegin(); iter != m_VtxBufferArray.cend();
        ++iter, ++pVtxBuffer, ++pVertexBufStateInfoData, ++index)
    {
        auto vertexSize = m_VtxBufferArray[index];

        nn::gfx::BufferInfo* pBufferInfo = nn::gfx::DataToAccessor( pVtxBuffer );
        pBufferInfo->SetDefault();

        pVtxBuffer->size = static_cast<uint16_t>(vertexSize.first) * vertexSize.second;
        pVertexBufStateInfoData->divisor	= 0;
        pVertexBufStateInfoData->stride		= static_cast<uint16_t>(vertexSize.first);

        // アラインメントのためのパディング分を埋める。
        pData = nw::g3d::tool::util::AddOffset(pData, pVtxBuffer->size);
        size_t paddingSize = nw::g3d::tool::util::Align(pVtxBuffer->size, ALIGNMENT_VERTEX_STREAM) - pVtxBuffer->size;
        std::memset(pData, 0, paddingSize);
        pData = nw::g3d::tool::util::AddOffset(pData, paddingSize);
        memoryPoolSize += static_cast<int>( (pVtxBuffer->size + paddingSize) );
    }

    // 頂点バッファの先頭
    util::BytePtr memPoolBasePtr = util::BytePtr( pCtx->GetMemBlockPtr( Context::MemBlockType_IdxStream ) ); // メモリプールの先頭アドレス
    vertex.memoryPoolOffset = static_cast<uint32_t>( memPoolBasePtr.Distance( GetPtr(pCtx, Context::MemBlockType_VtxStream, 0) ) );
}

// TODO: WeightIndex が 5 以上の場合を考慮できていない。
//       最適化コンバータと合わせて修正する必要がある。
void BinVertex::Adjust( std::shared_ptr<Context> pCtx )
{
    NN_UNUSED( pCtx );

    std::vector<BinVtxAttrib*> weightAttribArray;

    for (int i = 0; i < static_cast<int>(m_VtxAttribArray.size()); ++i)
    {
        BinVtxAttrib& attrib = m_VtxAttribArray[i];

        if (attrib.m_pElem->hint.value.find("blendweight") != std::string::npos)
        {
            weightAttribArray.push_back(&attrib);
        }
    }

    // blendweight の場合は量子化後の値が合計で 1.0 になるように補正を行います。
    if (weightAttribArray.size() <= 0)
    {
        return;
    }

    int vtxCount = weightAttribArray[0]->m_pElem->count.value;
    for (int vtxIndex = 0; vtxIndex < vtxCount; ++vtxIndex)
    {
        BinVtxAttrib* maxAttrib = weightAttribArray[0];
        float maxValue = 0.0f;
        int maxIndex = 0;

        for (int i = 0; i < static_cast<int>(weightAttribArray.size()); ++i)
        {
            BinVtxAttrib* attrib = weightAttribArray[i];

            int attribCount = attrib->m_pElem->type.value % 4 + 1;
            int size = attribCount * sizeof(float);
            const void* pSrc = nw::g3d::tool::util::AddOffset(attrib->m_pElem->stream.rawdata.get(), vtxIndex * size);

            // 有効な値がいくつかを調べます。
            // 最も大きな値を補正します。
            for (int j = attribCount - 1; j >= 0; --j)
            {
                float value = *nw::g3d::tool::util::AddOffset<float>(pSrc, j * sizeof(float));
                if (value >= maxValue)
                {
                    // 補正を行う箇所を決定します。
                    maxAttrib = attrib;
                    maxValue = value;
                    maxIndex = j;
                }
            }
        }

        float totalWeight = 1.0f;
        switch (maxAttrib->m_pElem->quantize_type.value)
        {
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8_8:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_8_8_8_8:
            {
                for (int i = 0; i < static_cast<int>(weightAttribArray.size()); ++i)
                {
                    BinVtxAttrib* attrib = weightAttribArray[i];
                    void* pDst = nw::g3d::tool::util::AddOffset(attrib->m_pDst, attrib->m_VtxStride * vtxIndex);

                    int attribCount = attrib->m_pElem->type.value % 4 + 1;

                    for (int j = 0; j < attribCount; ++j)
                    {
                        float value = 0.0f;
                        if (BinVtxAttrib::DequantizeFromUnorm8(&value, *nw::g3d::tool::util::AddOffset<uint8_t>(pDst, j * sizeof(uint8_t))) == false)
                        {
                            THROW_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_QUANTIZE_VALUE, "Invalid quantization format bound at %hs, vertexIndex=%d.\n", attrib->m_pElem->name.value.c_str(), vtxIndex);
                        }
                        totalWeight -= value;
                    }
                }

                // 最も大きな weight 値に不足分を加算する
                uint8_t* maxQuantizeValue = nw::g3d::tool::util::AddOffset<uint8_t>(maxAttrib->m_pDst, maxAttrib->m_VtxStride * vtxIndex + maxIndex * sizeof(uint8_t));
                if (totalWeight != 0.0f)
                {
                    if (BinVtxAttrib::QuantizeToUnorm8(maxQuantizeValue, maxValue + totalWeight) == false)
                    {
                        THROW_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_QUANTIZE_VALUE, "Invalid quantization format bound at vertexIndex=%d.\n", vtxIndex);
                    }
                }

                // 丸め誤差で 255 にならない場合があるので強制的に補正します。
                unsigned int total = 0;
                for (int i = 0; i < static_cast<int>(weightAttribArray.size()); ++i)
                {
                    BinVtxAttrib* attrib = weightAttribArray[i];
                    void* pDst = nw::g3d::tool::util::AddOffset(attrib->m_pDst, attrib->m_VtxStride * vtxIndex);

                    int attribCount = attrib->m_pElem->type.value % 4 + 1;

                    for (int j = 0; j < attribCount; ++j)
                    {
                        total += *nw::g3d::tool::util::AddOffset<uint8_t>(pDst, j * sizeof(uint8_t));
                    }
                }

                if (total == (0x1 << sizeof(uint8_t) * 8) - 2)
                {
                    // 254 の場合
                    *maxQuantizeValue = *maxQuantizeValue + 1;
                }
                else if (total == (0x1 << sizeof(uint8_t) * 8))
                {
                    // 256 の場合
                    *maxQuantizeValue = *maxQuantizeValue - 1;
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16_16:
        case nw::g3d::tool::g3dif::elem_vtx_attrib::qtype_unorm_16_16_16_16:
            {
                for (int i = 0; i < static_cast<int>(weightAttribArray.size()); ++i)
                {
                    BinVtxAttrib* attrib = weightAttribArray[i];
                    void* pDst = nw::g3d::tool::util::AddOffset(attrib->m_pDst, attrib->m_VtxStride * vtxIndex);

                    int attribCount = attrib->m_pElem->type.value % 4 + 1;

                    for (int j = 0; j < attribCount; ++j)
                    {
                        float value = 0.0f;
                        if (BinVtxAttrib::DequantizeFromUnorm16(&value, *nw::g3d::tool::util::AddOffset<uint16_t>(pDst, j * sizeof(uint16_t))) == false)
                        {
                            THROW_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_QUANTIZE_VALUE, "Invalid quantization format bound at %hs, vertexIndex=%d.\n", attrib->m_pElem->name.value.c_str(), vtxIndex);
                        }
                        totalWeight -= value;
                    }
                }

                uint16_t* maxQuantizeValue = nw::g3d::tool::util::AddOffset<uint16_t>(maxAttrib->m_pDst, maxIndex * sizeof(uint16_t));
                if (totalWeight != 0.0f)
                {
                    if (BinVtxAttrib::QuantizeToUnorm16(maxQuantizeValue, maxValue + totalWeight) == false)
                    {
                        THROW_BINARY_BLOCK_ERROR(ERRCODE_VERTEX_INVALID_QUANTIZE_VALUE, "Invalid quantization format bound at vertexIndex=%d.\n", vtxIndex);
                    }
                }

                // 丸め誤差で 65535 にならない場合があるので強制的に補正します。
                unsigned int total = 0;
                for (int i = 0; i < static_cast<int>(weightAttribArray.size()); ++i)
                {
                    BinVtxAttrib* attrib = weightAttribArray[i];
                    void* pDst = nw::g3d::tool::util::AddOffset(attrib->m_pDst, maxAttrib->m_VtxStride * vtxIndex + attrib->m_VtxStride * vtxIndex);

                    int attribCount = attrib->m_pElem->type.value % 4 + 1;

                    for (int j = 0; j < attribCount; ++j)
                    {
                        total += *nw::g3d::tool::util::AddOffset<uint16_t>(pDst, j * sizeof(uint16_t));
                    }
                }

                if (total == (0x1 << sizeof(uint16_t) * 8) - 2)
                {
                    // 65534 の場合
                    *maxQuantizeValue = *maxQuantizeValue + 1;
                }
                else if (total == (0x1 << sizeof(uint16_t) * 8))
                {
                    // 65536 の場合
                    *maxQuantizeValue = *maxQuantizeValue - 1;
                }
            }
            break;
        }
    }
}

}
}
