﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinShapeAnim.h>
#include <algorithm>

namespace nn {
namespace g3dTool {

//--------------------------------------------------------------------------------------------------
void BinVertexShapeAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vertex_shape_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_KeyShapeAnimCount = static_cast<int>(elem.shape_anim_target_array.size() + 1);

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iter = elem.shape_anim_target_array.cbegin();
        iter != elem.shape_anim_target_array.cend(); ++iter)
    {
        if (iter->curve)
        {
            curveArray.push_back(&iter->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetTargetOffset(0);
        iter->SetType(BinAnimCurve::FLOAT);
    }

    // 文字列の登録。
    pCtx->SetStr( elem.shape_name.value.c_str() );
    pCtx->SetStr( elem.base_name.value.c_str() );

    for (auto iter = elem.shape_anim_target_array.cbegin();
        iter != elem.shape_anim_target_array.cend(); ++iter)
    {
        pCtx->SetStr( iter->key_shape_name.value.c_str() );
    }
}

void BinVertexShapeAnim::CalculateSize()
{
    m_Chunk[ANIM_INFO].size = sizeof(nn::g3d::ResKeyShapeAnimInfo) * m_KeyShapeAnimCount;
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = nw::g3d::tool::util::Align(sizeof(float) * m_pElem->shape_anim_target_array.size(), ALIGNMENT_DEFAULT);
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinVertexShapeAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }
}

void BinVertexShapeAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResVertexShapeAnimData& vertexShapeAnim = *GetPtr<nn::g3d::ResVertexShapeAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    vertexShapeAnim.curveCount = nw::g3d::tool::NumericCast<uint16_t>(m_CurveArray.size());
    vertexShapeAnim.keyShapeAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_KeyShapeAnimCount);
    vertexShapeAnim.beginCurve = m_BeginCurve;
    vertexShapeAnim.beginKeyShapeAnim = m_BeginKeyShapeAnim;

    pCtx->LinkStr( &vertexShapeAnim.pName, m_pElem->shape_name.value.c_str() );

    // 0 番目に baseShape の情報を入れる。
    nn::g3d::ResKeyShapeAnimInfo* pKeyShapeAnimInfo = GetPtr<nn::g3d::ResKeyShapeAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[ANIM_INFO].offset);
    int shapeInfoIndex = 0;
    int curveIndex = 0;
    nn::g3d::ResKeyShapeAnimInfo& baseAnimInfo = pKeyShapeAnimInfo[0];
    baseAnimInfo.curveIndex = -1;
    baseAnimInfo.subbindIndex = -1;
    pCtx->LinkStr( &baseAnimInfo.pName, m_pElem->base_name.value.c_str() );
    ++shapeInfoIndex;

    for (auto iter = m_pElem->shape_anim_target_array.cbegin(); iter != m_pElem->shape_anim_target_array.cend(); ++iter, ++shapeInfoIndex)
    {
        nn::g3d::ResKeyShapeAnimInfo& keyShapeAnimInfo = pKeyShapeAnimInfo[shapeInfoIndex];
        pCtx->LinkStr( &keyShapeAnimInfo.pName, iter->key_shape_name.value.c_str() );
        keyShapeAnimInfo.subbindIndex = -1;

        if (iter->curve)
        {
            keyShapeAnimInfo.curveIndex = static_cast<int8_t>(curveIndex);
            ++curveIndex;
        }
        else
        {
            keyShapeAnimInfo.curveIndex = -1;
        }
    }
    pCtx->LinkPtr( &vertexShapeAnim.pKeyShapeAnimInfoArray, pKeyShapeAnimInfo );

    pCtx->LinkPtr( &vertexShapeAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    float* pBaseValue = GetPtr<float>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    pCtx->LinkPtr( &vertexShapeAnim.pBaseValueArray, pBaseValue );

    int targetIndex = 0;
    for (auto iter = m_pElem->shape_anim_target_array.cbegin();
        iter != m_pElem->shape_anim_target_array.cend(); ++iter, ++targetIndex)
    {
        pBaseValue[targetIndex] = iter->base_value.value;
    }
}

//--------------------------------------------------------------------------------------------------
void BinShapeAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shape_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_VertexShapeAnimArray.resize(elem.vertex_shape_anim_array.size());
    SetParentBlockArray(m_VertexShapeAnimArray, this);
    BuildArray(pCtx, m_VertexShapeAnimArray, elem.vertex_shape_anim_array);

    int32_t numKeyShapeAnim = 0;
    for (auto iter = m_VertexShapeAnimArray.begin(); iter != m_VertexShapeAnimArray.end(); ++iter)
    {
        iter->SetBeginKeyShapeAnim(numKeyShapeAnim);
        numKeyShapeAnim += iter->GetKeyShapeAnimCount();
    }
    m_KeyShapeAnimCount = numKeyShapeAnim;

    int32_t numCurve = 0;
    for (auto iter = m_VertexShapeAnimArray.begin(); iter != m_VertexShapeAnimArray.end(); ++iter)
    {
        iter->SetBeginCurve(numCurve);
        numCurve += iter->GetCurveCount();
    }
    m_CurveCount = numCurve;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr( elem.path.c_str() );
    pCtx->SetStr( elem.name.c_str() );

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinShapeAnim::CalculateSize()
{
    auto numAnim =  m_pElem->vertex_shape_anim_array.size();
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_VertexShapeAnim].size = sizeof(nn::g3d::ResVertexShapeAnimData) * numAnim;
    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinShapeAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_VertexShapeAnim].offset;
    for (auto iter = m_VertexShapeAnimArray.begin(); iter != m_VertexShapeAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResVertexShapeAnimData);
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_UserDataData ].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinShapeAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResShapeAnimData& shapeAnim = *GetPtr<nn::g3d::ResShapeAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    shapeAnim.blockHeader.signature.SetPacked( nn::g3d::ResShapeAnim::Signature );
    pCtx->AddBinBlockHeader( &shapeAnim.blockHeader );
    pCtx->LinkStr( &shapeAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &shapeAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_shape_anim_info& info = m_pElem->shape_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::ResShapeAnim::Flag_PlayPolicyLoop;
    }

    shapeAnim.flag = flag;
    shapeAnim.frameCount = m_pElem->shape_anim_info.frame_count.value;
    shapeAnim.vertexShapeAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_pElem->vertex_shape_anim_array.size());

    shapeAnim.keyShapeAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_KeyShapeAnimCount);
    shapeAnim.curveCount = nw::g3d::tool::NumericCast<uint16_t>(m_CurveCount);
    shapeAnim.bakedSize = 0;

    shapeAnim.pBindModel.Set( nullptr );
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    pCtx->LinkPtr( &shapeAnim.pBindIndexArray, pBindIndexArray );
    int numAnim = static_cast<int>(m_VertexShapeAnimArray.size());
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }

    pCtx->LinkPtr( &shapeAnim.pVertexShapeAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_VertexShapeAnim].offset) );

    // UserData
    shapeAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    pCtx->LinkPtr( &shapeAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
    m_DicUserData.ConvertData(pCtx, shapeAnim.pUserDataDic, m_UserDataArray);
}

void BinShapeAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResShapeAnim* shapeAnim = GetPtr<nn::g3d::ResShapeAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    nn::g3d::ResShapeAnimData& shapeAnimData = shapeAnim->ToData();
    shapeAnimData.pVertexShapeAnimArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxAnim = 0, numAnim = shapeAnim->GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        nn::g3d::ResVertexShapeAnim* pVertexShapeAnim = shapeAnim->GetVertexShapeAnim(idxAnim);
        pVertexShapeAnim->ToData().pCurveArray.Relocate( pCtx->GetBasePtr() );

        for (int idxCurve = 0, numCurve = pVertexShapeAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pVertexShapeAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedFloatSize();
        }
        pVertexShapeAnim->ToData().pCurveArray.Unrelocate( pCtx->GetBasePtr() );
    }
    shapeAnimData.pVertexShapeAnimArray.Unrelocate( pCtx->GetBasePtr() );
    shapeAnimData.bakedSize = static_cast<uint32_t>(size);
}

}
}
