﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Arithmetic.h>
#include <BinSceneAnim.h>
#include <util/UtilMath.h>

namespace nn {
namespace g3dTool {

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    camera_anim, rotate_mode, binary,
    nn::g3d::ResCameraAnim::Flag_RotAim,
    nn::g3d::ResCameraAnim::Flag_RotEulerZxy
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    camera_anim, projection_mode, binary,
    nn::g3d::ResCameraAnim::Flag_ProjOrtho,
    nn::g3d::ResCameraAnim::Flag_ProjPersp
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    camera_anim_target, target, binary,
    4, 5, 6,    // pos[3]
    7, 8, 9,    // aim[3]
    10,         // twist
    7, 8, 9,    // rotate[3]
    2,          // aspect
    0,          // nearZ
    1,          // farZ
    3,          // height
    3           // fovy
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    light_anim_target, target, binary,
    0,          // enable
    1, 2, 3,    // pos[3]
    4, 5, 6,    // dir[3]
    4, 5, 6,    // aim[3]
    7, 8,       // distAttn[2]
    9, 10,      // angleAttn[2]
    11, 12, 13, // color[0][3]
    14, 15, 16  // color[1][3]
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    fog_anim_target, target, binary,
    0, 1,       // distAttn[2]
    2, 3, 4     // color[3]
    );

void BinCameraAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_camera_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_TargetAnimArray.reserve(elem.camera_anim_target_array.size());
    for(auto& target : elem.camera_anim_target_array)
    {
        if ((target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_x ||
            target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_y ||
            target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_z) &&
            elem.rotate_mode.value != nw::g3d::tool::g3dif::elem_camera_anim::enum_rotate_mode::euler_zxy)
        {
            continue;
        }
        if ((target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_aim_x ||
            target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_aim_y ||
            target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_aim_z) &&
            elem.rotate_mode.value != nw::g3d::tool::g3dif::elem_camera_anim::enum_rotate_mode::aim)
        {
            continue;
        }
        if (target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_ortho_height &&
            elem.projection_mode.value != nw::g3d::tool::g3dif::elem_camera_anim::enum_projection_mode::ortho)
        {
            continue;
        }
        if (target.target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_persp_fovy &&
            elem.projection_mode.value != nw::g3d::tool::g3dif::elem_camera_anim::enum_projection_mode::persp)
        {
            continue;
        }
        m_TargetAnimArray.push_back(&target);
    }

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iterTarget = m_TargetAnimArray.cbegin();
        iterTarget != m_TargetAnimArray.cend(); ++iterTarget)
    {
        if ((*iterTarget)->curve)
        {
            curveArray.push_back(&(*iterTarget)->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int curveIndex = 0;
    for (auto iterTarget = m_TargetAnimArray.cbegin();
        iterTarget != m_TargetAnimArray.cend(); ++iterTarget)
    {
        if ((*iterTarget)->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(sizeof(nn::Bit32) * ToEnum_binary((*iterTarget)->target.value));
            curve.SetType(BinAnimCurve::FLOAT);

            if ((*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_twist    ||
                (*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_x ||
                (*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_y ||
                (*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_z ||
                (*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_persp_fovy)
            {
                curve.SetDegreeValue(true);
            }
            ++curveIndex;
        }
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());

    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.camera_name.value.c_str());

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinCameraAnim::CalculateSize()
{
    // カメラデータは nn::g3d::ResSceneAnimData に含まれる
    //m_Chunk[CAMERA].size = sizeof(nn::g3d::ResCameraAnimData)
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = sizeof(nn::g3d::CameraAnimResult);
    m_Chunk[ChunkType_UserDataData].size = sizeof(nn::gfx::ResUserDataData) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinCameraAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }

    // ResUserDataData のヒープを BinSerData に与えます。
    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_UserDataData].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::gfx::ResUserDataData );
    }
}

void BinCameraAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResCameraAnimData& cameraAnim = *GetPtr<nn::g3d::ResCameraAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));

    cameraAnim.blockHeader.signature.SetPacked( nn::g3d::ResCameraAnim::Signature );
    uint16_t flag = 0;
    if (m_pElem->loop.value)
    {
        flag |= nn::g3d::ResCameraAnim::Flag_PlayPolicyLoop;
    }
    flag |= ToEnum_binary(m_pElem->rotate_mode.value);
    flag |= ToEnum_binary(m_pElem->projection_mode.value);

    cameraAnim.flag = flag;
    cameraAnim.frameCount = m_pElem->frame_count.value;
    cameraAnim.bakedSize = 0;

    float* pBaseValue = GetPtr<float>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    pCtx->LinkPtr( &cameraAnim.pBaseValueArray, pBaseValue );
    memset(pBaseValue, 0, sizeof(nn::g3d::CameraAnimResult));
    for (auto iter = m_TargetAnimArray.cbegin(); iter != m_TargetAnimArray.cend(); ++iter)
    {
        int offset = ToEnum_binary((*iter)->target.value);

        float baseValue = (*iter)->base_value.value;

        if ((*iter)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_twist    ||
            (*iter)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_x ||
            (*iter)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_y ||
            (*iter)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_rotate_z ||
            (*iter)->target.value == nw::g3d::tool::g3dif::elem_camera_anim_target::target_persp_fovy)
        {
            baseValue = nn::util::DegreeToRadian(baseValue);
        }

        pBaseValue[offset] = baseValue;
    }
    pCtx->LinkStr( &cameraAnim.pName, m_pElem->camera_name.value.c_str() );

    cameraAnim.curveCount = static_cast<uint8_t>(m_CurveArray.size());
    pCtx->LinkPtr( &cameraAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    // UserData
    cameraAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, cameraAnim.pUserDataDic, m_UserDataArray);
    pCtx->LinkPtr( &cameraAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );

}

void BinCameraAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResCameraAnim* pCameraAnim = GetPtr<nn::g3d::ResCameraAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResCameraAnimData& cameraAnimData = pCameraAnim->ToData();
    cameraAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxCurve = 0, numCurve = pCameraAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        nn::g3d::ResAnimCurve* curve = pCameraAnim->GetCurve(idxCurve);
        size += curve->CalculateBakedFloatSize();
    }
    cameraAnimData.bakedSize = static_cast<uint32_t>(size);
    cameraAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
}

void BinLightAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_light_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iterTarget = elem.light_anim_target_array.cbegin();
        iterTarget != elem.light_anim_target_array.cend(); ++iterTarget)
    {
        if (iterTarget->curve)
        {
            curveArray.push_back(&iterTarget->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int curveIndex = 0;
    for (auto iterTarget = elem.light_anim_target_array.cbegin();
        iterTarget != elem.light_anim_target_array.cend(); ++iterTarget)
    {
        if (iterTarget->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(sizeof(nn::Bit32) * ToEnum_binary(iterTarget->target.value));

            if (iterTarget->target.value == nw::g3d::tool::g3dif::elem_light_anim_target::target_enable)
            {
                curve.SetType(BinAnimCurve::BOOL);
            }
            else
            {
                curve.SetType(BinAnimCurve::FLOAT);
            }

            if (iterTarget->target.value == nw::g3d::tool::g3dif::elem_light_anim_target::target_angle_attn_start ||
                iterTarget->target.value == nw::g3d::tool::g3dif::elem_light_anim_target::target_angle_attn_end)
            {
                curve.SetDegreeValue(true);
            }
            ++curveIndex;
        }
    }

    m_LightAnimTarget.resize(nw::g3d::tool::g3dif::elem_light_anim_target::num_target, nullptr);

    // target をリソースの順に並び替えて格納する。
    for (int i = 0; i < nw::g3d::tool::g3dif::elem_light_anim_target::num_target; ++i)
    {
        for (auto iterTarget = m_pElem->light_anim_target_array.cbegin();
            iterTarget != m_pElem->light_anim_target_array.cend(); ++iterTarget)
        {
            if (static_cast<nw::g3d::tool::g3dif::elem_light_anim_target::enum_target>(i) == iterTarget->target.value)
            {
                m_LightAnimTarget[i] = &(*iterTarget);
                break;
            }
        }
    }

    static const int tblDimension[] = { 1, 3, 3, 3, 2, 2, 3, 3 };
    static const int tblBaseValue[] = { 0, 1, 2, 2, 3, 4, 5, 6 };
    static const nw::g3d::tool::g3dif::elem_light_anim_target::enum_target tblTarget[] = {
        nw::g3d::tool::g3dif::elem_light_anim_target::target_enable,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_position_x,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_direction_x,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_aim_x,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_dist_attn_start,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_angle_attn_start,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_color0_r,
        nw::g3d::tool::g3dif::elem_light_anim_target::target_color1_r
    };
    static const int NUM_PARAM = sizeof(tblDimension) / sizeof(int);
    m_TargetOffset.resize(m_LightAnimTarget.size());

    for (int idxParam = 0; idxParam < NUM_PARAM; ++idxParam)
    {
        for (auto iterTarget = m_LightAnimTarget.cbegin();
            iterTarget != m_LightAnimTarget.cend(); ++iterTarget)
        {
            if ((*iterTarget) != nullptr &&
                tblTarget[idxParam] == (*iterTarget)->target.value)
            {
                m_Flag |= (nn::g3d::ResLightAnim::Flag_BaseEnable << tblBaseValue[idxParam]);
                int targetIndex = static_cast<int>((*iterTarget)->target.value);
                for (int i = 0; i < tblDimension[idxParam]; ++i)
                {
                    m_TargetOffset[targetIndex + i] = m_BaseValue + i;
                }

                m_BaseValue += tblDimension[idxParam];

                // Enable のアニメーションについて有効／無効を判定
                if (((*iterTarget)->target.value == nw::g3d::tool::g3dif::elem_light_anim_target::target_enable) &&
                    (*iterTarget)->curve)
                {
                    m_Flag |= nn::g3d::ResLightAnim::Flag_CurveEnable;
                }
                break;
            }
        }
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.light_name.value.c_str());
    pCtx->SetStr(elem.type.value.c_str());
    pCtx->SetStr(elem.dist_attn_func.value.c_str());
    pCtx->SetStr(elem.angle_attn_func.value.c_str());

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinLightAnim::CalculateSize()
{
    // ライトデータは nn::g3d::ResSceneAnimData に含まれる
    //m_Chunk[LIGHT].size = sizeof(ResLightData)
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = sizeof(float) * m_BaseValue;
    m_Chunk[ChunkType_UserDataData].size = sizeof(nn::gfx::ResUserDataData) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinLightAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }

    // ResUserDataData のヒープを BinSerData に与えます。
    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_UserDataData].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::gfx::ResUserDataData );
    }
}

void BinLightAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResLightAnimData& lightAnim = *GetPtr<nn::g3d::ResLightAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));

    lightAnim.blockHeader.signature.SetPacked( nn::g3d::ResLightAnim::Signature );

    if (m_pElem->loop.value)
    {
        m_Flag |= nn::g3d::ResLightAnim::Flag_PlayPolicyLoop;
    }

    lightAnim.flag = m_Flag;
    lightAnim.frameCount = m_pElem->frame_count.value;
    lightAnim.lightTypeIndex = -1;
    lightAnim.distAttnFuncIndex = -1;
    lightAnim.bakedSize = 0;

    float* pBaseValue = GetPtr<float>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    pCtx->LinkPtr( &lightAnim.pBaseValueArray, pBaseValue );
    for (auto iter = m_LightAnimTarget.cbegin();
        iter != m_LightAnimTarget.cend(); ++iter)
    {
        if ((*iter) == nullptr)
        {
            continue;
        }

        int targetIndex = static_cast<int>((*iter)->target.value);
        int offset = m_TargetOffset[targetIndex];
        if ((*iter)->target.value == nw::g3d::tool::g3dif::elem_light_anim_target::target_enable)
        {
            // enable の場合は int で値を代入
            *reinterpret_cast<int*>(&pBaseValue[offset]) = static_cast<int>((*iter)->base_value.value);
        }
        else
        {
            pBaseValue[offset] = (*iter)->base_value.value;
        }
    }
    pCtx->LinkStr( &lightAnim.pName, m_pElem->light_name.value.c_str() );
    pCtx->LinkStr( &lightAnim.pLightType, m_pElem->type.value.c_str() );
    pCtx->LinkStr( &lightAnim.pDistAttnFunc, m_pElem->dist_attn_func.value.c_str() );
    pCtx->LinkStr( &lightAnim.pAngleAttnFunc, m_pElem->angle_attn_func.value.c_str() );

    lightAnim.curveCount = static_cast<uint8_t>(m_CurveArray.size());
    pCtx->LinkPtr( &lightAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    // UserData
    lightAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, lightAnim.pUserDataDic, m_UserDataArray);
    pCtx->LinkPtr( &lightAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
}

void BinLightAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResLightAnim* lightAnim = GetPtr<nn::g3d::ResLightAnim>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    nn::g3d::ResLightAnimData& lightAnimData = lightAnim->ToData();
    lightAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxCurve = 0, numCurve = lightAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        nn::g3d::ResAnimCurve* curve = lightAnim->GetCurve(idxCurve);
        size += curve->CalculateBakedFloatSize();
    }
    lightAnimData.bakedSize = static_cast<uint32_t>(size);
    lightAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
}

void BinFogAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_fog_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iterTarget = elem.fog_anim_target_array.cbegin();
        iterTarget != elem.fog_anim_target_array.cend(); ++iterTarget)
    {
        if (iterTarget->curve)
        {
            curveArray.push_back(&iterTarget->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int curveIndex = 0;
    for (auto iterTarget = elem.fog_anim_target_array.cbegin();
        iterTarget != elem.fog_anim_target_array.cend(); ++iterTarget)
    {
        if (iterTarget->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(sizeof(nn::Bit32) * ToEnum_binary(iterTarget->target.value));
            curve.SetType(BinAnimCurve::FLOAT);

            ++curveIndex;
        }
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.fog_name.value.c_str());
    pCtx->SetStr(elem.dist_attn_func.value.c_str());

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinFogAnim::CalculateSize()
{
    // フォグデータは nn::g3d::ResSceneAnimData に含まれる
    //m_Chunk[FOG].size = sizeof(nn::g3d::ResFogAnimData)
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = sizeof(nn::g3d::FogAnimResult);
    m_Chunk[ChunkType_UserDataData].size = sizeof(nn::gfx::ResUserDataData) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinFogAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }

    // ResUserDataData のヒープを BinUerData に与えます。
    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_UserDataData].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::gfx::ResUserDataData );
    }
}

void BinFogAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResFogAnimData& fogAnim = *GetPtr<nn::g3d::ResFogAnimData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    fogAnim.blockHeader.signature.SetPacked( nn::g3d::ResFogAnim::Signature );
    uint16_t flag = 0;
    if (m_pElem->loop.value)
    {
        flag |= nn::g3d::ResFogAnim::Flag_PlayPolicyLoop;
    }

    fogAnim.flag = flag;
    fogAnim.frameCount = m_pElem->frame_count.value;
    fogAnim.distAttnFuncIndex = -1;
    fogAnim.bakedSize = 0;

    float* pBaseValue = GetPtr<float>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    pCtx->LinkPtr( &fogAnim.pBaseValueArray, pBaseValue );

    memset(pBaseValue, 0, sizeof(nn::g3d::FogAnimResult));
    for (auto iter = m_pElem->fog_anim_target_array.cbegin();
        iter != m_pElem->fog_anim_target_array.cend(); ++iter)
    {
        int offset = ToEnum_binary(iter->target.value);
        pBaseValue[offset] = iter->base_value.value;
    }
    pCtx->LinkStr( &fogAnim.pName, nn::util::string_view( (m_pElem->fog_name.value.c_str() ) ) );
    pCtx->LinkStr( &fogAnim.pDistAttnFunc, nn::util::string_view( (m_pElem->dist_attn_func.value.c_str() ) ) );

    fogAnim.curveCount = static_cast<uint8_t>(m_CurveArray.size());
    pCtx->LinkPtr( &fogAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    // UserData
    fogAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, fogAnim.pUserDataDic, m_UserDataArray);
    pCtx->LinkPtr( &fogAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
}

void BinFogAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResFogAnim* fogAnim = GetPtr<nn::g3d::ResFogAnim>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    nn::g3d::ResFogAnimData& fogAnimData = fogAnim->ToData();
    fogAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxCurve = 0, numCurve = fogAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        nn::g3d::ResAnimCurve* curve = fogAnim->GetCurve(idxCurve);
        size += curve->CalculateBakedFloatSize();
    }
    fogAnimData.bakedSize = static_cast<uint32_t>(size);
    fogAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
}

void BinSceneAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_scene_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_camera_anim*> cameraAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_light_anim*> lightAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_fog_anim*> fogAnimArray;

    auto numCamera = m_pElem->camera_anim_array.size();
    cameraAnimArray.reserve(numCamera);
    auto numLight = m_pElem->light_anim_array.size();
    lightAnimArray.reserve(numLight);
    auto numFog = m_pElem->fog_anim_array.size();
    fogAnimArray.reserve(numFog);

    for (auto iter = m_pElem->camera_anim_array.cbegin(); iter != m_pElem->camera_anim_array.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_camera_anim& camera = *iter;
        cameraAnimArray.push_back(&camera);
    }

    for (auto iter = m_pElem->light_anim_array.cbegin(); iter != m_pElem->light_anim_array.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_light_anim& light = *iter;
        lightAnimArray.push_back(&light);
    }

    for (auto iter = m_pElem->fog_anim_array.cbegin(); iter != m_pElem->fog_anim_array.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_fog_anim& fog = *iter;
        fogAnimArray.push_back(&fog);
    }

    m_DicCameraAnim.Build(pCtx, cameraAnimArray.size());
    m_DicLightAnim.Build(pCtx, lightAnimArray.size());
    m_DicFogAnim.Build(pCtx, fogAnimArray.size());

    m_CameraAnimArray.resize(cameraAnimArray.size());
    m_LightAnimArray.resize(lightAnimArray.size());
    m_FogAnimArray.resize(fogAnimArray.size());
    SetParentBlockArray(m_CameraAnimArray, this);
    SetParentBlockArray(m_LightAnimArray, this);
    SetParentBlockArray(m_FogAnimArray, this);
    BuildArray(pCtx, m_CameraAnimArray, cameraAnimArray);
    BuildArray(pCtx, m_LightAnimArray, lightAnimArray);
    BuildArray(pCtx, m_FogAnimArray, fogAnimArray);

    int cameraAnimIndex = 0;
    for (auto iter = cameraAnimArray.cbegin(); iter != cameraAnimArray.cend(); ++iter, ++cameraAnimIndex)
    {
        m_DicCameraAnim.SetName(cameraAnimIndex, (*iter)->camera_name.value);
    }

    int lightAnimIndex = 0;
    for (auto iter = lightAnimArray.cbegin(); iter != lightAnimArray.cend(); ++iter, ++lightAnimIndex)
    {
        m_DicLightAnim.SetName(lightAnimIndex, (*iter)->light_name.value);
    }

    int fogAnimIndex = 0;
    for (auto iter = fogAnimArray.cbegin(); iter != fogAnimArray.cend(); ++iter, ++fogAnimIndex)
    {
        m_DicFogAnim.SetName(fogAnimIndex, (*iter)->fog_name.value);
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。

    pCtx->SetStr(elem.path.c_str());
    pCtx->SetStr(elem.name.c_str());

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinSceneAnim::CalculateSize()
{
    m_Chunk[SCENE_ANIM].size = sizeof(nn::g3d::ResSceneAnimData);
    m_Chunk[CAMERA_ANIM].size = sizeof(nn::g3d::ResCameraAnimData) * m_pElem->camera_anim_array.size();
    m_Chunk[LIGHT_ANIM].size = sizeof(nn::g3d::ResLightAnimData) * m_pElem->light_anim_array.size();
    m_Chunk[FOG_ANIM].size = sizeof(nn::g3d::ResFogAnimData) * m_pElem->fog_anim_array.size();
    m_Chunk[ChunkType_UserDataData].size = sizeof(nn::gfx::ResUserDataData) * m_pElem->user_data_array.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinSceneAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t cameraOffset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[CAMERA_ANIM].offset;
    for (auto iter = m_CameraAnimArray.begin(); iter != m_CameraAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(cameraOffset);
        cameraOffset += sizeof(nn::g3d::ResCameraAnimData);
    }

    ptrdiff_t lightOffset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[LIGHT_ANIM].offset;
    for (auto iter = m_LightAnimArray.begin(); iter != m_LightAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(lightOffset);
        lightOffset += sizeof(nn::g3d::ResLightAnimData);
    }

    ptrdiff_t fogOffset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[FOG_ANIM].offset;
    for (auto iter = m_FogAnimArray.begin(); iter != m_FogAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(fogOffset);
        fogOffset += sizeof(nn::g3d::ResFogAnimData);
    }

    ptrdiff_t userDataOffset = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataData ].offset;
    for (auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter)
    {
        iter->SetStructOffset(userDataOffset);
        userDataOffset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinSceneAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResSceneAnimData& sceneAnim = *GetPtr<nn::g3d::ResSceneAnimData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    sceneAnim.blockHeader.signature.SetPacked( nn::g3d::ResSceneAnim::Signature );
    pCtx->AddBinBlockHeader( &sceneAnim.blockHeader );

    pCtx->LinkStr( &sceneAnim.pName, nn::util::string_view( m_pElem->name.c_str() ) );
    pCtx->LinkStr( &sceneAnim.pPath, nn::util::string_view( m_pElem->path.c_str() ) );

    sceneAnim.cameraAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_pElem->camera_anim_array.size());
    m_DicCameraAnim.ConvertData(pCtx, sceneAnim.pCameraAnimDic, m_CameraAnimArray);
    pCtx->LinkPtr( &sceneAnim.pCameraAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[CAMERA_ANIM].offset) );

    sceneAnim.lightAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_pElem->light_anim_array.size());
    m_DicLightAnim.ConvertData(pCtx, sceneAnim.pLightAnimDic, m_LightAnimArray);
    pCtx->LinkPtr( &sceneAnim.pLightAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[LIGHT_ANIM].offset) );

    sceneAnim.fogAnimCount = nw::g3d::tool::NumericCast<uint16_t>(m_pElem->fog_anim_array.size());
    m_DicFogAnim.ConvertData(pCtx, sceneAnim.pFogAnimDic, m_FogAnimArray);
    pCtx->LinkPtr( &sceneAnim.pFogAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[FOG_ANIM].offset) );

    // UserData
    sceneAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, sceneAnim.pUserDataDic, m_UserDataArray);
    pCtx->LinkPtr( &sceneAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
}

}
}
