﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/g3d_ResMaterialAnim.h>
#include <g3dif/TexPatternAnim.h>
#include <bindefs.h>
#include <BinaryBlock.h>
#include <BinDictionary.h>
#include <BinAnimCurve.h>
#include <BinUserData.h>


namespace nn {
namespace g3dTool {

//! @brief マテリアルごとのアニメーションデータのバイナリ化を行うクラスです。
class BinTexPatternMatAnim : public BinaryBlock
{
public:
    BinTexPatternMatAnim()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_CurveArray()
        , m_BeginCurve(0)
        , m_BeginTexture(0)
        , m_ConstantAnimCount(0)
        , m_Chunk()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_tex_pattern_mat_anim& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("TexPatternMatAnim : %hs", m_pElem->mat_name.value.c_str());
    }

    int GetCurveCount() const { return static_cast<int>(m_CurveArray.size()); }
    int GetTextureCount() const { return static_cast<int>(m_pElem->pattern_anim_target_array.size()); }

    void SetBeginCurve(int beginCurve) { this->m_BeginCurve = beginCurve; }
    void SetBeginTexture(int beginTexture) { this->m_BeginTexture = beginTexture; }

private:
    const nw::g3d::tool::g3dif::elem_tex_pattern_mat_anim* m_pElem;

    std::vector<BinAnimCurve> m_CurveArray;
    int m_BeginCurve;
    int m_BeginTexture;
    uint16_t m_ConstantAnimCount;

    enum
    {
        // BinTexPatternAnim に含まれる
        // MAT_ANIM
        TEX_INFO,
        ChunkType_Curve,
        ChunkType_Constant,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

//! @brief テクスチャパターンアニメーションデータのバイナリ化を行うクラスです。
class BinTexPatternAnim : public BinaryBlock
{
public:
    BinTexPatternAnim()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_TexPatternMatAnimArray()
        , m_TextureCount(0)
        , m_CurveCount(0)
        , m_DicUserData()
        , m_UserDataArray()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_tex_pattern_anim& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );
    virtual void Adjust( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("TexPatternAnim : %hs", m_pElem->name.c_str());
    }

private:
    const nw::g3d::tool::g3dif::elem_tex_pattern_anim* m_pElem;

    std::vector<BinTexPatternMatAnim> m_TexPatternMatAnimArray;
    int m_TextureCount;
    int m_CurveCount;

    BinDictionary m_DicUserData;
    std::vector<BinUserData> m_UserDataArray;

    enum ChunkType
    {
        ChunkType_TexturePatternAnim,
        ChunkType_BindIndex,
        ChunkType_TexturePatternMaterialAnim,
        ChunkType_TextureDescriptorSlotArray,
        ChunkType_GfxTextureViewPtrArray,
        ChunkType_TextureNamePtrArray,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

}
}
