﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <CommandLineParserWrapper.h>
#include <Windows.h>
#include <util/UtilErrorCode.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {
namespace {

template< typename TType >
TType GetProcessAddr( const HMODULE& hModule, const char* pStr )
{
    TType returnValue = reinterpret_cast< TType >( GetProcAddress( hModule, pStr ) );
    if( returnValue == NULL )
    {
        THROW_ERROR(ERRCODE_INTERNAL, "Failed to get the DLL function pointer.");
    }

    return returnValue;
}

inline void FormatToNwOption( std::string* pOutStr, const char* pKey, const char* pReplaceTo )
{
    std::string::size_type pos = 0;
    pos = pOutStr->find( pKey, pos, 1 );
    while( pos != std::string::npos )
    {
        if( pos == std::string::npos )
        {
            break;	// 一文字も見つからない場合はアーリーアウト
        }
        pOutStr->replace( pos, 1, pReplaceTo );
        ++pos;
        pos = pOutStr->find( pKey, pos, 1 );
    }

    pOutStr->insert( 0, "--" );	// gfxTool のパーサではオプション名には "--" は含まれないため付加する。
}

inline void FormatToNwOption( std::string* pOutStr, const std::string& suffics, const char* pTarget, const char* pReplaceTo )
{
    FormatToNwOption( pOutStr, pTarget, pReplaceTo );
    *pOutStr += "=";
    *pOutStr += suffics;
}

//!< @brief 旧 nw が内部で扱っているフォーマットに成形し、pOutStrArray に保存します。"フルオプション名=値" に整形します。
template< typename TType >
inline void ConvertToNwOptionNameFmt( std::vector< std::wstring >* pOutStrArray, const TType& option )
{
    if( !option.IsExisting() )
    {
        return;
    }

    if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::None )
    {
        std::string oldFormat( option.GetDefinition().pLongName );
        FormatToNwOption( &oldFormat, "-", "-" );
        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
    else if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::One )
    {
        std::string oldFormat( option.GetDefinition().pLongName );

        // option の値が文字列の時に'\"'が入る場合がある（ args-file に output="hoge.bfres" と記述した場合等）
        // なので、文字列の先頭と最後の'\"'を除去しておく。
        std::string optionValue( option.GetValue() );
        if( *optionValue.begin() == '\"' && *( optionValue.end() - 1 ) == '\"' )
        {
            optionValue.erase( optionValue.begin() );
            optionValue.erase( optionValue.end() - 1 );
        }

        FormatToNwOption( &oldFormat, optionValue, "-", "-" );

        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
}

template<>
inline void ConvertToNwOptionNameFmt< nn::gfxTool::CommandLineOption<> >(
    std::vector< std::wstring >* pOutStrArray, const nn::gfxTool::CommandLineOption<>& option )
{
    if( !option.IsExisting() )
    {
        return;
    }

    if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::None )
    {
        std::string oldFormat( option.GetDefinition().pLongName );
        FormatToNwOption( &oldFormat, "-", "-" );
        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
}
}

CommandLineParserWrapper::CommandLineParserWrapper( int argc, const wchar_t* argv[] )
{
    // wchar から char に変換します。
    for( int idx = 0; idx < argc; ++idx )
    {
        std::wstring str( argv[ idx ] );
        m_CharBuff.push_back( std::string( str.begin(), str.end() ) );
    }

    m_GlobalOptionParser.Parse( m_CharBuff );

    // Siglo コマンドラインオプションを旧オプションに変換します。
    // value を持つ option は = で接続します。（旧オプションの仕様）
    {
        // 旧コンバータの仕様上、以下の順番で文字列ポインタを並べないといけない。
        // 1. 実行ファイル名　2. option　3. input_file_name

        // 1. 実行ファイル名を追加します。
        auto inputArgs	= m_GlobalOptionParser.GetInputArgs();
        auto addStr = [ this ]( std::string str )
        {
            this->m_WideCharBuff.push_back( std::wstring( str.begin(), str.end() ) );
        };
        std::for_each( inputArgs.begin(), inputArgs.begin() + 1, addStr );

        // 2. option 名を追加します。
        const nn::g3dTool::GlobalOptionParser::CommandLineOptionDefinitions& options = m_GlobalOptionParser.GetCommandLineOptionDefinitions();
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.editor );
        //ConvertToNwOptionNameFmt( &m_WideCharBuff, options.externalFile );			// externalFile は後でパースします。
        //ConvertToNwOptionNameFmt( &m_WideCharBuff, options.externalFileAlignment );	// externalFileAlignment はファイルオプションとして独立させる。
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.help );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.ignoreAssign );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.ignoreVersion );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.output );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.rootPath );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.silent );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.verify );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.version );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.viewer );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.disableAttributeAlignmentSort );

        // 3. input_file_name を追加します。
        if( inputArgs.begin() + 1 != inputArgs.end() )
        {
            std::for_each( inputArgs.begin() + 1, inputArgs.end(), addStr );
        }

        // 4. external_file を追加します。
        if( options.externalFile.IsExisting() )
        {
            auto value = options.externalFile.GetValue();
            for( auto it = value.begin(); it != value.end(); ++it )
            {
                // 1. ファイルオプションをパースします。
                // 必ず第一引数が外部ファイルへのフルパスという約束
                std::vector< std::string > tempFileOptions;
                if( (*it).c_str()[0] == '\"' || (*it).c_str()[0] == '\'' )
                {
                    const char divider = (*it).c_str()[0];
                    auto dividerPos = it->find_last_of( divider );		// 最後に出現する divider を取得
                    if( dividerPos == std::string::npos )
                    {
                        THROW_ERROR( ERRCODE_INVALID_FILENAME, "external-file value is invalid. Cannot find end of quotation.\n" );
                    }
                    std::string path( *it, 1, dividerPos - 1 );			// divider は読み飛ばす
                    tempFileOptions.emplace_back( path );

                    // divider の次に ' ' がある場合は --external-file-alignment オプションがあるので追加する
                    if( (*it).c_str()[dividerPos + 1] == ' ' )
                    {
                        std::string alignOpt( *it, dividerPos + 2, ( *it ).length() );	// ' ' は読み飛ばす
                        tempFileOptions.emplace_back( alignOpt );
                    }
                }
                else // --external-file の値に空白を含まない場合は空白区切りで配列化する。
                {
                    tempFileOptions = nn::gfxTool::TextToArray< std::vector< std::string >, char >( (*it).c_str() );
                }

                m_FileOptionParser.Parse( tempFileOptions );

                // 2. パースしたオプションからオプションを取得します。 --external-file-path=, --external-file-align= の二つがあります。
                // 2.1. input_file 名を追加します。
                auto fileOptionInputArg	= m_FileOptionParser.GetInputArgs();
                std::for_each( fileOptionInputArg.begin(), fileOptionInputArg.end(), addStr );


                // 2.2. オプション名（--external-file)を追加します。
                {
                    std::string oldFormat( m_GlobalOptionParser.GetCommandLineOptionDefinitions().externalFile.GetDefinition().pLongName );
                    FormatToNwOption( &oldFormat, "-", "-" );
                    m_WideCharBuff.push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
                }

                // 2.3. ファイルオプションを持っていたら追加する。
                auto fileOptions	= m_FileOptionParser.GetCommandLineOptionDefinitions();
                if( fileOptions.externalFileAlign.IsExisting() )
                {
                    ConvertToNwOptionNameFmt( &m_WideCharBuff, fileOptions.externalFileAlign );
                }

                // 3. 最後にパーサのリセット
                m_FileOptionParser.ResetOptions();

            }
        }
    }

    {
        const wchar_t**	wideCharStrPtrArray = new( const wchar_t* [ m_WideCharBuff.size() ] );
        for( int idx = 0; idx < static_cast< int >( m_WideCharBuff.size() ); ++idx )
        {
            wideCharStrPtrArray[ idx ] = m_WideCharBuff[ idx ].c_str();
        }

        m_pCommandArgs.reset( new nw::g3d::tool::CmdArgs( static_cast< int >( m_WideCharBuff.size() ), wideCharStrPtrArray ) );
        delete[] wideCharStrPtrArray;
    }

}

void CommandLineParserWrapper::LoadDllFuncs( const HMODULE& hModule )
{
    m_DllFuncs.pFuncAddFile	= GetProcessAddr< DllFuncs::AddFile >( hModule, "nng3dToolBinaryConverterAddFile" );
    m_DllFuncs.pFuncBind	= GetProcessAddr< DllFuncs::Bind >( hModule, "nng3dToolBinaryConverterBind" );
    m_DllFuncs.pFuncCalculateSize	= GetProcessAddr< DllFuncs::CalculateSize>( hModule, "nng3dToolBinaryConverterCalculateSize" );
    m_DllFuncs.pFuncConvert	= GetProcessAddr< DllFuncs::Convert >( hModule, "nng3dToolBinaryConverterConvert" );
    m_DllFuncs.pFuncGetAlignmentSize	= GetProcessAddr< DllFuncs::GetAlignmentSize >( hModule, "nng3dToolBinaryConverterGetAlignmentSize" );
    m_DllFuncs.pFuncGetCvtrVersion	= GetProcessAddr< DllFuncs::GetCvtrVersion >( hModule, "nng3dToolBinaryConverterGetConverterVersion" );
    m_DllFuncs.pFuncGetBinVersion	= GetProcessAddr< DllFuncs::GetBinaryVersion >( hModule, "nng3dToolBinaryConverterGetBinaryVersion" );
    m_DllFuncs.pFuncInitialize  = GetProcessAddr< DllFuncs::Initialize >( hModule, "nng3dToolBinaryConverterInitialize" );
    m_DllFuncs.pFuncSetOptions	= GetProcessAddr< DllFuncs::SetOptions >( hModule, "nng3dToolBinaryConverterSetOptions" );
    m_DllFuncs.pFuncShutdown	= GetProcessAddr< DllFuncs::Shutdown >( hModule, "nng3dToolBinaryConverterShutdown" );
    m_DllFuncs.pFuncSwapEndian	= GetProcessAddr< DllFuncs::SwapEndian >( hModule, "nng3dToolBinaryConverterSwapEndian" );
}

}
}
