﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <Windows.h>
#include <vector>
#include <memory>
#include <util/UtilCmdArgs.h>
#include <GlobalOptionParser.h>
#include <FileOptionParser.h>

namespace nn {
namespace g3dTool {

class CommandLineParserWrapper
{
public:
    //!< @brief dll 関数ポインタの定義と宣言です。
    struct DllFuncs
    {
        typedef uint32_t ( *GetCvtrVersion )();
        typedef uint32_t ( *GetBinaryVersion )();
        typedef bool ( *Initialize ) ();
        typedef bool ( *Shutdown ) ();
        typedef bool ( *SetOptions )(const wchar_t* options[]);
        typedef bool ( *AddFile )(void* pData, size_t dataSize, const wchar_t* path[3], const wchar_t* options[] );
        typedef size_t ( *CalculateSize )();
        typedef size_t ( *GetAlignmentSize )();
        typedef bool ( *Convert )(void* pBuffer, size_t size);
        typedef bool ( *Bind )(void* pBuffer, size_t size);
        typedef bool ( *SwapEndian )(void* pBuffer, size_t size);

        GetCvtrVersion			pFuncGetCvtrVersion;
        GetBinaryVersion		pFuncGetBinVersion;
        Initialize				pFuncInitialize;
        Shutdown  				pFuncShutdown;
        SetOptions  			pFuncSetOptions;
        AddFile  				pFuncAddFile;
        CalculateSize  			pFuncCalculateSize;
        GetAlignmentSize  		pFuncGetAlignmentSize;
        Convert  				pFuncConvert;
        Bind  					pFuncBind;
        SwapEndian  			pFuncSwapEndian;
    };

    explicit
    CommandLineParserWrapper( int argc, const wchar_t* argv[] );

    virtual ~CommandLineParserWrapper()
    {
        m_pCommandArgs.reset();
    }

    //!< @brief HMODULE から関数ポンタを取得します。
    void LoadDllFuncs( const HMODULE& hModule );

    //!< @brief 初期化済みの nw コマンドラインパーサを取得します。
    std::shared_ptr< nw::g3d::tool::CmdArgs > GetCmdArgs()
    {
        return m_pCommandArgs;
    }

    //!< @brief dll 関数ポインタを返します。
    const DllFuncs& GetDllFuncs() const
    {
        return m_DllFuncs;
    }

    //!< @brief ヘルプを表示します。
    void PrintHelp( const char* pToolName, const char* pToolDescription )
    {
        // グローバルオプションのヘルプ表示
        m_GlobalOptionParser.PrintHelp( pToolName, pToolDescription );

        // ファイルオプションのヘルプ表示
        {
            FileOptionParser::CommandLineOptionDefinitions fileOptionDefinitions =
                m_FileOptionParser.GetCommandLineOptionDefinitions();

            NN_GFXTOOL_PRINT( "  --%s <arg>  Specify the alignment for the external_file as it is [LOCAL].",
                fileOptionDefinitions.externalFileAlign.GetDefinition().pLongName );
            NN_GFXTOOL_PRINT( "\n\nExample:\n" );
            NN_GFXTOOL_PRINT("%s.exe --output=\"output_file\" basic.fmdb --external-file=\"basic.bfsha --external-file-alignment=128\"", pToolName );
        }
    }

private:
    //!< @brief グローバルオプションのパーサです。
    GlobalOptionParser	m_GlobalOptionParser;

    //!< @brief ファイルオプションのパーサです。
    FileOptionParser	m_FileOptionParser;

    //!< @brief dll から取得した関数ポインタを保持します。
    DllFuncs							m_DllFuncs;

    //!< @brief char 文字列を保持するバッファです。
    std::vector< std::string >			m_CharBuff;

    //!< @brief wchar_t 文字列を保持するバッファです。
    std::vector< std::wstring >			m_WideCharBuff;

    //!< @brief 旧 nw のパーサです。
    std::shared_ptr< nw::g3d::tool::CmdArgs >	m_pCommandArgs;

};

}
}
