﻿namespace G3dCore.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using nw.g3d.iflib;
    using nw.g3d.nw4f_3dif;
    using Opal.App;
    using Opal.Logs;

    /// <summary>
    /// G3d 中間ファイルのユーティリティクラスです。
    /// </summary>
    public class G3dIfUtility
    {
        /// <summary>
        /// G3d 中間ファイルを読み込みます。
        /// </summary>
        /// <param name="path">中間ファイルのパスです。</param>
        /// <returns>中間ファイルクラスのインスタンスを返します。</returns>
        public static G3dFile ReadFile(string path)
        {
            bool isUpdated;
            return ReadFile(path, out isUpdated);
        }

        /// <summary>
        /// G3d 中間ファイルを読み込みます。
        /// </summary>
        /// <param name="path">中間ファイルのパスです。</param>
        /// <param name="isUpdated">中間ファイルがアップデートされたかどうかを出力します。</param>
        /// <returns>中間ファイルクラスのインスタンスを返します。</returns>
        public static G3dFile ReadFile(string path, out bool isUpdated)
        {
            isUpdated = false;

            try
            {
                nw4f_3difType file = null;
                List<G3dStream> streamArray = null;
                if (G3dPath.IsStreamBinaryPath(path))
                {
                    streamArray = new List<G3dStream>();
                    file = IfBinaryReadUtility.Read(streamArray, path, G3dIfUtility.XsdBasePath, out isUpdated);
                }
                else
                {
                    file = IfReadUtility.Read(path, G3dIfUtility.XsdBasePath, out isUpdated);
                }

                G3dFile g3dFile = new G3dFile(file);
                g3dFile.FilePath = path;
                return g3dFile;
            }
            catch (Exception e)
            {
                var logger = new Logger();

                Exception ex = e;
                while (ex != null)
                {
#if DEBUG
                    logger.Debug(ex.ToString());
#else
                    if (!string.IsNullOrEmpty(ex.Message))
                    {
                        logger.Error(ex.Message);
                    }
#endif
                    ex = ex.InnerException;
                }

                var logManager = AppManager.GetLogManager();
                Debug.Assert(logManager != null);

                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            return null;
        }

        /// <summary>
        /// G3d 中間ファイルを保存します。
        /// </summary>
        /// <param name="g3dFile">中間ファイルクラスです。</param>
        /// <param name="path">ファイルパスです。</param>
        public static void WriteFile(G3dFile g3dFile, string path)
        {
            if (g3dFile == null) throw new ArgumentNullException();

            var data = g3dFile.CreateWriteData();
            IfWriteUtility.Write(data, path);

            g3dFile.Reset();
        }

        /// <summary>
        /// スキーマのベースパス。
        /// </summary>
        public static string XsdBasePath
        {
            get
            {
                return System.IO.Path.Combine(Environment.GetEnvironmentVariable("NW4F_G3D_ROOT"), @"Tool\G3dTool\g3dxsd\");
            }
        }
    }
}
