﻿namespace G3dCore.IO
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// 一時ファイル関連のユーティリティクラスです。
    /// </summary>
    public static class TemporaryFileUtility
    {
        private static int counter = 0;
        private static string lastTempName = string.Empty;

        /// <summary>
        /// Temporary ディレクトリを取得します。
        /// </summary>
        /// <returns>Temporary ディレクトリを返します。</returns>
        public static string GetTemporaryDir()
        {
            string exeDir = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
            string temporaryDir = Path.Combine(exeDir, "temporary");
            if (!Directory.Exists(temporaryDir))
            {
                Directory.CreateDirectory(temporaryDir);
            }

            return temporaryDir;
        }

        /// <summary>
        /// Temporary 用のユニークな名前を作成します。
        /// </summary>
        /// <returns>名前を返します。</returns>
        public static string MakeTemporaryName()
        {
            lock (typeof(TemporaryFileUtility))
            {
                string tempName = DateTime.Now.ToString("yyyyMMddHHmmss");
                if (tempName == lastTempName)
                {
                    counter++;
                }
                else
                {
                    counter = 0;
                }

                lastTempName = tempName;
                tempName = tempName + "_" + counter;

                return tempName;
            }
        }

        /// <summary>
        /// Temporary ディレクトリ内のファイルを削除します。
        /// </summary>
        public static void DeleteTemporaryFiles()
        {
            var paths = Directory.GetDirectories(GetTemporaryDir());
            foreach (var path in paths)
            {
                try
                {
                    Directory.Delete(path, true);
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// Dispose 時に削除を行うファイルです。
        /// </summary>
        public sealed class DisposableFileName : IDisposable
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="path">ファイルパスです。</param>
            public DisposableFileName(string path)
            {
                this.Path = path;
            }

            /// <summary>
            /// ファイルパスです。
            /// </summary>
            public string Path { get; private set; }

            /// <summary>
            /// 破棄処理です。
            /// </summary>
            public void Dispose()
            {
                if (File.Exists(this.Path))
                {
                    File.Delete(this.Path);
                }
            }
        }
    }
}
