﻿namespace G3dCore.Extensions
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.IO;
    using Opal.Operations;

    /// <summary>
    /// シェーダ設定のオペレーション拡張メソッドです。
    /// </summary>
    public static class ShaderConfigOperationExtension
    {
        /// <summary>
        /// インクルードパスを新しいものに置き換えるオペレーションを生成します。
        /// </summary>
        /// <param name="shaderConfig">データモデルです。</param>
        /// <param name="includePaths">新しいインクルードパスです。</param>
        /// <returns>生成したオペレーションを返します。</returns>
        public static Operation CreateRelpaceIncludePathsOperation(this ShaderConfig shaderConfig, params string[] includePaths)
        {
            OperationSet operation = new OperationSet();
            operation.Add(new ListClearOperation<IncludePath>(shaderConfig.IncludePaths));

            int index = 0;
            foreach (var includePath in includePaths)
            {
                var newIncludePath = new IncludePath()
                    {
                        Index = index,
                        Path = PathUtility.FormatPathSeparator(includePath, true)
                    };

                operation.Add(new ListItemAddOperation<IncludePath>(shaderConfig.IncludePaths, newIncludePath));
                ++index;
            }

            return operation;
        }

        /// <summary>
        /// 強制インクルードファイルを新しいものに置き換えるオペレーションを生成します。
        /// </summary>
        /// <param name="shaderConfig">データモデルです。</param>
        /// <param name="forceIncludeFiles">新しい強制インクルードファイルです。</param>
        /// <returns>生成したオペレーションを返します。</returns>
        public static Operation CreateRelpaceForceIncludeFilesOperation(this ShaderConfig shaderConfig, params string[] forceIncludeFiles)
        {
            OperationSet operation = new OperationSet();
            operation.Add(new ListClearOperation<ForceIncludeFile>(shaderConfig.ForceIncludeFiles));

            int index = 0;
            foreach (var forceIncludeFile in forceIncludeFiles)
            {
                var newForceIncludeFile = new ForceIncludeFile()
                {
                    Index = index,
                    Path = PathUtility.FormatPathSeparator(forceIncludeFile, true)
                };

                operation.Add(new ListItemAddOperation<ForceIncludeFile>(shaderConfig.ForceIncludeFiles, newForceIncludeFile));
                ++index;
            }

            return operation;
        }

        /// <summary>
        /// シェーダ設定定義を追加するオペレーションを生成します。
        /// </summary>
        /// <param name="shaderConfig">データモデルです。</param>
        /// <param name="shaders">追加するシェーダ設定定義です。</param>
        /// <returns>生成したオペレーションを返します。</returns>
        public static Operation CreateAddShadersOperation(this ShaderConfig shaderConfig, params Shader[] shaders)
        {
            OperationSet operation = new OperationSet();

            foreach (Shader shader in shaders)
            {
                operation.Add(new ListItemAddOperation<Shader>(shaderConfig.Shaders, shader));
            }

            return operation;
        }

        /// <summary>
        /// シェーダ設定定義を削除するオペレーションを生成します。
        /// </summary>
        /// <param name="shaderConfig">データモデルです。</param>
        /// <param name="shaders">削除するシェーダ設定定義です。</param>
        /// <returns>生成したオペレーションを返します。</returns>
        public static Operation CreateRemoveShadersOperation(this ShaderConfig shaderConfig, params Shader[] shaders)
        {
            OperationSet operation = new OperationSet();

            foreach (Shader shader in shaders)
            {
                operation.Add(new ListItemRemoveOperation<Shader>(shaderConfig.Shaders, shader));
            }

            return operation;
        }
    }
}
