﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーダバリエーションのエンティティクラスです。
    /// </summary>
    public class Variation : Entity<variationType>
    {
        private int index = 0;
        private string id = string.Empty;
        private string choice = string.Empty;
        private string @default = string.Empty;
        private bool branch = false;
        private bool isOutputable = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Variation()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        public Variation(variationType data)
            : this()
        {
            this.index = data.index;
            this.id = data.id ?? string.Empty;
            this.choice = data.choice ?? string.Empty;
            this.@default = data.@default ?? string.Empty;
            this.branch = data.branch;
        }

        /// <summary>
        /// インデックスを取得設定します。
        /// </summary>
        public int Index
        {
            get
            {
                return this.index;
            }

            set
            {
                this.SetProperty(ref this.index, value);
            }
        }

        /// <summary>
        /// 識別子を取得設定します。
        /// </summary>
        public string Id
        {
            get
            {
                return this.id;
            }

            set
            {
                this.SetProperty(ref this.id, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// choice を取得設定します。
        /// </summary>
        public string Choice
        {
            get
            {
                return this.choice;
            }

            set
            {
                this.SetProperty(ref this.choice, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// デフォルト値を取得設定します。
        /// </summary>
        public string Default
        {
            get
            {
                return this.@default;
            }

            set
            {
                this.SetProperty(ref this.@default, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 静的分岐かどうかを取得設定します。
        /// </summary>
        public bool Branch
        {
            get
            {
                return this.branch;
            }

            set
            {
                this.SetProperty(ref this.branch, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 中間ファイルに出力するかどうかを取得設定します。
        /// </summary>
        public bool IsOutputable
        {
            get
            {
                return this.isOutputable;
            }

            set
            {
                this.SetProperty(ref this.isOutputable, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new Variation();
            instance.index = this.index;
            instance.id = this.id;
            instance.choice = this.choice;
            instance.@default = this.@default;
            instance.branch = this.branch;
            instance.isOutputable = this.isOutputable;

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override variationType CreateWriteData()
        {
            variationType instance = new variationType();
            instance.index = this.index;
            instance.id = this.id;
            instance.choice = this.choice;
            instance.@default = this.@default;
            instance.branch = this.branch;

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            Debug.Assert(this.id != null);
            Debug.Assert(this.choice != null);
            Debug.Assert(this.@default != null);

            // Index プロパティは、出力時に計算しなおすので、CRCの計算対象外
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.choice)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.@default)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.branch)));

            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.isOutputable)));

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }
    }
}
