﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーディングモデルのエンティティクラスです。
    /// </summary>
    public class ShadingModel : ObservableEntity<shading_modelType>
    {
        private readonly ObservableCollection<Macro> macros = new ObservableCollection<Macro>();
        private readonly ObservableCollection<OptionVar> optionVars = new ObservableCollection<OptionVar>();

        private int index = 0;
        private string name = string.Empty;
        private bool materialShader = false;
        private int revision = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShadingModel()
        {
            this.macros.CollectionChanged += this.OnCollectionChanged<Macro>;
            this.optionVars.CollectionChanged += this.OnCollectionChanged<OptionVar>;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        public ShadingModel(shading_modelType data)
            : this()
        {
            this.index = data.index;
            this.name = data.name ?? string.Empty;
            this.materialShader = data.material_shader;
            this.revision = data.revision;

            if (data.macro_array != null)
            {
                this.AddMacros(data.macro_array.macro);
            }

            if (data.option_var_array != null)
            {
                this.AddOptionVars(data.option_var_array.option_var);
            }
        }

        /// <summary>
        /// インデックスを取得設定します。
        /// </summary>
        public int Index
        {
            get
            {
                return this.index;
            }

            set
            {
                this.SetProperty(ref this.index, value);
            }
        }

        /// <summary>
        /// 名前を取得設定します。
        /// </summary>
        public string Name
        {
            get
            {
                return this.name;
            }

            set
            {
                this.SetProperty(ref this.name, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// マテリアル用のシェーディングモデルかどうかを取得設定します。
        /// </summary>
        public bool MaterialShader
        {
            get
            {
                return this.materialShader;
            }

            set
            {
                this.SetProperty(ref this.materialShader, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// リビジョン番号を取得設定します。
        /// </summary>
        public int Revision
        {
            get
            {
                return this.revision;
            }

            set
            {
                this.SetProperty(ref this.revision, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// マクロを取得します。
        /// </summary>
        public ObservableCollection<Macro> Macros
        {
            get
            {
                return this.macros;
            }
        }

        /// <summary>
        /// オプション変数を取得します。
        /// </summary>
        public ObservableCollection<OptionVar> OptionVars
        {
            get
            {
                return this.optionVars;
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new ShadingModel();

            instance.index = this.index;
            instance.name = this.name;
            instance.materialShader = this.materialShader;
            instance.revision = this.revision;

            foreach (var macro in this.macros)
            {
                instance.macros.Add(macro.Clone() as Macro);
            }

            foreach (var optionVar in this.optionVars)
            {
                instance.optionVars.Add(optionVar.Clone() as OptionVar);
            }

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shading_modelType CreateWriteData()
        {
            shading_modelType instance = new shading_modelType();
            instance.index = this.index;
            instance.name = this.name;
            instance.material_shader = this.materialShader;
            instance.revision = this.revision;

            instance.macro_array = this.CreateMacroArray();
            instance.option_var_array = this.CreateOptionVarArray();

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            // Index プロパティは、出力時に計算しなおすので、CRCの計算対象外
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.materialShader)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.revision)));

            foreach (var macro in this.macros)
            {
                buffers.AddRange(BitConverter.GetBytes(macro.HashValue));
            }

            foreach (var optionVar in this.optionVars)
            {
                buffers.AddRange(BitConverter.GetBytes(optionVar.HashValue));
            }

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

        /// <summary>
        /// マクロ配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected macro_arrayType CreateMacroArray()
        {
            if (this.macros.FirstOrDefault() == null)
            {
                return null;
            }

            var macroArray = new macro_arrayType();
            List<macroType> macroTypes = new List<macroType>();

            int index = 0;
            foreach (var macro in this.macros)
            {
                macro.Index = index;
                ++index;
                macroTypes.Add(macro.CreateWriteData());
            }

            macroArray.macro = macroTypes.ToArray();
            macroArray.length = macroTypes.Count;
            return macroArray;
        }

        /// <summary>
        /// オプション変数配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected option_var_arrayType CreateOptionVarArray()
        {
            if (this.optionVars.FirstOrDefault() == null)
            {
                return null;
            }

            var optionVarArray = new option_var_arrayType();
            List<option_varType> optionVarTypes = new List<option_varType>();

            int index = 0;
            foreach (var optionVar in this.optionVars)
            {
                optionVar.Index = index;
                ++index;
                optionVarTypes.Add(optionVar.CreateWriteData());
            }

            optionVarArray.option_var = optionVarTypes.ToArray();
            optionVarArray.length = optionVarTypes.Count;
            return optionVarArray;
        }

        /// <summary>
        /// マクロを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddMacros(macroType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                Macro macro = new Macro(value);
                this.macros.Add(macro);
            }
        }

        /// <summary>
        /// オプション変数を追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddOptionVars(option_varType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                OptionVar optionVar = new OptionVar(value);
                this.optionVars.Add(optionVar);
            }
        }
    }
}
