﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーダ定義のエンティティクラスです。
    /// TODO: 全てのエンティティクラスを作成していません。
    /// </summary>
    public class ShaderDefinition : RootEntity<shader_definitionType>
    {
        private readonly ObservableCollection<ForceInclude> forceIncludes = new ObservableCollection<ForceInclude>();
        private readonly ObservableCollection<ShadingModel> shadingModels = new ObservableCollection<ShadingModel>();
        private readonly ObservableCollection<ShaderSrc> shaderSrcs = new ObservableCollection<ShaderSrc>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderDefinition()
        {
            this.shadingModels.CollectionChanged += this.OnCollectionChanged<ShadingModel>;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        public ShaderDefinition(shader_definitionType data)
            : this()
        {
            Debug.Assert(data != null);

            if (data.force_include_array != null)
            {
                this.AddForceIncludes(data.force_include_array.force_include);
            }

            if (data.shading_model_array != null)
            {
                this.AddShadingModels(data.shading_model_array.shading_model);
            }

            if (data.shader_src_array != null)
            {
                this.AddShaderSrcs(data.shader_src_array.shader_src);
            }
        }

        /// <summary>
        /// 強制インクルードを取得します。
        /// </summary>
        public ObservableCollection<ForceInclude> ForceIncludes
        {
            get
            {
                return this.forceIncludes;
            }
        }

        /// <summary>
        /// シェーディングモデルを取得します。
        /// </summary>
        public ObservableCollection<ShadingModel> ShadingModels
        {
            get
            {
                return this.shadingModels;
            }
        }

        /// <summary>
        /// シェーダソース定義を取得します。
        /// </summary>
        public ObservableCollection<ShaderSrc> ShaderSrcs
        {
            get
            {
                return this.shaderSrcs;
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new ShaderDefinition();
            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_definitionType CreateWriteData()
        {
            var instance = new shader_definitionType();
            instance.force_include_array = this.CreateForceIncludeArray();
            instance.shading_model_array = this.CreateShadingModelArray();
            instance.shader_src_array = this.CreateShaderSrcArray();

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GetHashCode())));

            foreach (var forceInclude in this.forceIncludes)
            {
                buffers.AddRange(BitConverter.GetBytes(forceInclude.HashValue));
            }

            foreach (var shadingModel in this.shadingModels)
            {
                buffers.AddRange(BitConverter.GetBytes(shadingModel.HashValue));
            }

            foreach (var shaderSrc in this.shaderSrcs)
            {
                buffers.AddRange(BitConverter.GetBytes(shaderSrc.HashValue));
            }

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var forceInclude in this.forceIncludes)
            {
                forceInclude.Reset();
            }

            foreach (var shadingModel in this.shadingModels)
            {
                shadingModel.Reset();
            }

            foreach (var shaderSrc in this.shaderSrcs)
            {
                shaderSrc.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var forceInclude in this.forceIncludes)
            {
                forceInclude.Refresh();
            }

            foreach (var shadingModel in this.shadingModels)
            {
                shadingModel.Refresh();
            }

            foreach (var shaderSrc in this.shaderSrcs)
            {
                shaderSrc.Refresh();
            }
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var forceInclude in this.forceIncludes)
            {
                forceInclude.AutoCalc = this.AutoCalc;
            }

            foreach (var shadingModel in this.shadingModels)
            {
                shadingModel.AutoCalc = this.AutoCalc;
            }

            foreach (var shaderSrc in this.shaderSrcs)
            {
                shaderSrc.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// 強制インクルード配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected force_include_arrayType CreateForceIncludeArray()
        {
            if (this.forceIncludes.FirstOrDefault() == null)
            {
                return null;
            }

            var forceIncludeArray = new force_include_arrayType();
            var forceIncludeTypes = new List<force_includeType>();

            int index = 0;
            foreach (var forceInclude in this.forceIncludes)
            {
                forceInclude.Index = index;
                ++index;
                forceIncludeTypes.Add(forceInclude.CreateWriteData());
            }

            forceIncludeArray.force_include = forceIncludeTypes.ToArray();
            forceIncludeArray.length = forceIncludeTypes.Count;
            return forceIncludeArray;
        }

        /// <summary>
        /// シェーディングモデル配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected shading_model_arrayType CreateShadingModelArray()
        {
            if (this.shadingModels.FirstOrDefault() == null)
            {
                return null;
            }

            var shadingModelArray = new shading_model_arrayType();
            List<shading_modelType> shadingModelTypes = new List<shading_modelType>();

            int index = 0;
            foreach (var shadingModel in this.shadingModels)
            {
                shadingModel.Index = index;
                ++index;
                shadingModelTypes.Add(shadingModel.CreateWriteData());
            }

            shadingModelArray.shading_model = shadingModelTypes.ToArray();
            shadingModelArray.length = shadingModelTypes.Count;
            return shadingModelArray;
        }

        /// <summary>
        /// シェーダソース定義配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected shader_src_arrayType CreateShaderSrcArray()
        {
            if (this.shaderSrcs.FirstOrDefault() == null)
            {
                return null;
            }

            var shaderSrcArray = new shader_src_arrayType();
            var shaderSrcTypes = new List<shader_srcType>();

            foreach (var shaderSrc in this.shaderSrcs)
            {
                shaderSrcTypes.Add(shaderSrc.CreateWriteData());
            }

            shaderSrcArray.shader_src = shaderSrcTypes.ToArray();
            shaderSrcArray.length = shaderSrcTypes.Count;
            return shaderSrcArray;
        }

        /// <summary>
        /// 強制インクルードを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddForceIncludes(force_includeType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                ForceInclude forceInclude = new ForceInclude(value);
                this.forceIncludes.Add(forceInclude);
            }
        }

        /// <summary>
        /// シェーディングモデルを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddShadingModels(shading_modelType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                ShadingModel shadingModel = new ShadingModel(value);
                this.shadingModels.Add(shadingModel);
            }
        }

        /// <summary>
        /// シェーダーソース定義を追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddShaderSrcs(shader_srcType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                ShaderSrc shaderSrc = new ShaderSrc(value);
                this.shaderSrcs.Add(shaderSrc);
            }
        }
    }
}
