﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.iflib;
    using nw.g3d.nw4f_3dif;
    using G3dCore.Utilities;
    using Opal.Security.Cryptography;

    /// <summary>
    /// 中間ファイルクラスです。
    /// </summary>
    public class G3dFile : Entity<nw4f_3difType>
    {
        private string filePath = string.Empty;
        private IRootEntity rootEntity = null;
        private IEntity fileInfoEntity = null;
        private G3dKind kind = G3dKind.Invalid;
        private bool createFileInfo = false;
        private bool isUpdated = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public G3dFile()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="kind">中間ファイルの種類です。</param>
        public G3dFile(G3dKind kind)
        {
            switch (kind)
            {
                case G3dKind.ShaderConfig:
                    this.kind = G3dKind.ShaderConfig;
                    this.rootEntity = new ShaderConfig();
                    break;
                case G3dKind.ShaderDefinition:
                    this.kind = G3dKind.ShaderConfig;
                    this.rootEntity = new ShaderDefinition();
                    break;
                default:
                    // 非サポートフォーマット
                    break;
            }

            this.fileInfoEntity = new FileInfo();

            Debug.Assert(this.rootEntity != null);
            this.rootEntity.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="nw4f_3dif">設定する中間ファイルデータです。</param>
        public G3dFile(nw4f_3difType nw4f_3dif)
        {
            Debug.Assert(nw4f_3dif != null);
            Debug.Assert(nw4f_3dif.Item != null);

            if (nw4f_3dif.Item is shader_configType)
            {
                this.kind = G3dKind.ShaderConfig;
                this.AddShaderConfig((shader_configType)nw4f_3dif.Item);
            }
            else if (nw4f_3dif.Item is shader_definitionType)
            {
                this.kind = G3dKind.ShaderDefinition;
                this.AddShaderDefinition((shader_definitionType)nw4f_3dif.Item);
            }
            else
            {
                // 非サポートフォーマット
            }

            this.AddFileInfo(nw4f_3dif.file_info);

            Debug.Assert(this.rootEntity != null);
            this.rootEntity.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// 中間ファイルの種類を取得します。
        /// </summary>
        public G3dKind Kind
        {
            get
            {
                return this.kind;
            }
        }

        /// <summary>
        /// ファイル名を取得設定します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.SetProperty(ref this.filePath, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// file_info が無い場合に付加するかを取得設定します。
        /// </summary>
        public bool CreateFileInfo
        {
            get
            {
                return this.createFileInfo;
            }

            set
            {
                this.createFileInfo = value;
            }
        }

        /// <summary>
        /// ファイルを開いた際のアップデート時にダーティフラグを立てるためのプロパティです。
        /// </summary>
        internal bool IsUpdated
        {
            set
            {
                this.SetProperty(ref this.isUpdated, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 中間ファイルのバイナリデータを生成します。
        /// </summary>
        /// <returns>バイナリデータを返します。/returns>
        public byte[] CreateBinaryData()
        {
            List<G3dStream> streams = new List<G3dStream>();
            nw4f_3difType instance = this.CreateWriteData();
            return IfBinaryFormatter.FormatStream(instance, streams, null);
        }

        /// <summary>
        /// 指定のルートエンティティクラスを取得します。
        /// </summary>
        /// <typeparam name="TRootEntity">ルートエンティティのテンプレートの型です。</typeparam>
        /// <returns>指定のエンティティクラスのインスタンスを返します。</returns>
        public TRootEntity GetRootEntity<TRootEntity>() where TRootEntity : class, IRootEntity
        {
            return this.rootEntity as TRootEntity;
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new G3dFile();
            instance.rootEntity = this.rootEntity.Clone() as IRootEntity;
            instance.kind = this.kind;
            instance.filePath = this.filePath;
            instance.fileInfoEntity = this.fileInfoEntity.Clone() as IEntity;

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override nw4f_3difType CreateWriteData()
        {
            Debug.Assert(this.rootEntity != null);
            Debug.Assert(this.fileInfoEntity != null);

            nw4f_3difType instance = new nw4f_3difType();
            instance.version = nw4f_3dif_versionType.Item400;
            instance.Item = this.rootEntity.CreateWriteData();
            instance.file_info = this.fileInfoEntity.CreateWriteData() as file_infoType;

            this.SetFileInfo(instance, string.Empty);

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GetHashCode())));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FilePath)));

            buffers.AddRange(BitConverter.GetBytes(this.rootEntity.HashValue));
            buffers.AddRange(BitConverter.GetBytes(this.isUpdated));
            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.rootEntity.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.rootEntity.Refresh();
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.rootEntity.AutoCalc = this.AutoCalc;
        }

        private void AddShaderConfig(shader_configType data)
        {
            if (data == null) throw new ArgumentNullException();
            if (this.rootEntity == null) throw new InvalidOperationException();

            this.rootEntity = new ShaderConfig(data, this);
        }

        private void AddShaderDefinition(shader_definitionType data)
        {
            if (data == null) throw new ArgumentNullException();
            if (this.rootEntity == null) throw new InvalidOperationException();

            this.rootEntity = new ShaderDefinition(data);
        }

        private void AddFileInfo(file_infoType data)
        {
            this.fileInfoEntity = new FileInfo(data);
        }

        private void SetFileInfo(nw4f_3difType file, string src_path)
        {
            file_infoType fileInfo = file.file_info;

            if (fileInfo == null || fileInfo.create == null)
            {
                if (this.createFileInfo)
                {
                    IfFileLogUtility.SetCreate(file, src_path);
                }
            }
            else
            {
                IfFileLogUtility.SetModify(file);
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "IsDirty")
            {
                this.CalcCRC();
            }
        }
    }
}
