﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.Extensions;
    using G3dCore.IO;
    using G3dCore.Resources;
    using Opal.Operations;
    using ShaderAssistAddons.Modules.ShaderConfig.Utilities;

    /// <summary>
    /// シェーダ設定を新規作成や編集するウィザードのビューモデルです。
    /// </summary>
    public class WizardShaderConfigViewModel : WizardViewModel
    {
        private readonly ShaderConfigCommonSettingViewModel commonSettingViewModel;
        private readonly PathSettingViewModel pathSettingViewModel;
        private readonly ShaderConverterViewModel shaderConverterViewModel;

        private ShaderConfigViewModel shaderConfigViewModel;
        private string[] includeFullPaths = new string[] { };
        private string[] forceIncludeFilePaths = new string[] { };

        /// <summary>
        /// 対象の ShaderConfigViewModel を取得します。
        /// </summary>
        public ShaderConfigViewModel ShaderConfigViewModel
        {
            get
            {
                return this.shaderConfigViewModel;
            }
        }

        /// <summary>
        /// シェーダ設定ファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get;
            private set;
        }

        /// <summary>
        /// シェーダ設定ファイルディレクトリを取得します。
        /// </summary>
        public string FileDir
        {
            get
            {
                return Path.GetDirectoryName(this.FilePath);
            }
        }

        /// <summary>
        /// コードページを取得します。
        /// </summary>
        public int CodePage
        {
            get;
            private set;
        }

        /// <summary>
        /// インクルードパスのフルパスを取得します。
        /// </summary>
        public string[] IncludeFullPaths
        {
            get
            {
                return this.includeFullPaths;
            }
        }

        /// <summary>
        /// 強制インクルードファイルのフルパスを取得します
        /// </summary>
        public string[] ForceIncludeFilePaths
        {
            get
            {
                return this.forceIncludeFilePaths;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="shaderConfigViewModel">対象のシェーダ設定ビューモデルです。
        /// null を指定した場合には新規作成モードになります。</param>
        public WizardShaderConfigViewModel(ShaderConfigViewModel shaderConfigViewModel)
        {
            this.shaderConfigViewModel = shaderConfigViewModel;

            this.commonSettingViewModel = new ShaderConfigCommonSettingViewModel(this);
            this.pathSettingViewModel = new PathSettingViewModel(this);
            this.shaderConverterViewModel = new ShaderConverterViewModel(this, true);

            if (this.shaderConfigViewModel != null)
            {
                this.SetEditData();

                this.PageViewModels = new WizardPageViewModel[]
                {
                    this.commonSettingViewModel,
                    this.pathSettingViewModel,
                    this.shaderConverterViewModel
                };

                this.Title = Labels.ShaderConfigCommonSetting;
            }
            else
            {
                this.PageViewModels = new WizardPageViewModel[]
                {
                    this.commonSettingViewModel,
                    this.pathSettingViewModel
                };

                this.Title = Labels.NewDocument;
            }
        }

        /// <summary>
        /// ページを移動した際の処理です。
        /// </summary>
        /// <param name="pageNo">ページ番号です。</param>
        /// <param name="goNext">次へを押された場合には true になります。</param>
        public override void ProcPage(int pageNo, bool goNext)
        {
            if (this.PageViewModels[pageNo] is ShaderConfigCommonSettingViewModel)
            {
                this.FilePath = this.commonSettingViewModel.FilePath;
                this.CodePage = this.commonSettingViewModel.CodePage;

                this.pathSettingViewModel.BuildListBox();
            }
            else if (this.PageViewModels[pageNo] is PathSettingViewModel)
            {
                this.includeFullPaths =
                    this.pathSettingViewModel.IncludePathListViewModel.ListBoxItems.Select(x => x.Tag.ToString()).ToArray();
                this.forceIncludeFilePaths =
                    this.pathSettingViewModel.ForceIncludeListViewModel.ListBoxItems.Select(x => x.Label).ToArray();

                if (goNext)
                {
                    this.shaderConverterViewModel.Reset();
                }
            }
        }

        /// <summary>
        /// フルパスを元に、baseDir からの相対パスのインクルードパスにします。
        /// </summary>
        /// <param name="fileDir">相対パスにするためのベースのディレクトリです。</param>
        /// <returns>相対インクルードパスを返します。</returns>
        public string[] GetIncludePaths(string baseDir)
        {
            string[] includePaths = new string[this.IncludeFullPaths.Length];

            for (int i = 0; i < includePaths.Length; ++i)
            {
                if (!this.IncludeFullPaths[i].Contains('%'))
                {
                    includePaths[i] = PathUtility.MakeRelativePath(baseDir, this.IncludeFullPaths[i]);
                }
                else
                {
                    // 環境変数が含まれている場合は展開しない。
                    includePaths[i] = this.IncludeFullPaths[i];
                }

                Debug.Assert(includePaths[i] != string.Empty);
            }

            return includePaths;
        }

        /// <summary>
        /// インクルードパスを元に、相対パスの強制インクルードファイルを取得します。
        /// </summary>
        /// <returns>相対パスの強制インクルードファイルを返します。</returns>
        public string[] GetForceIncludeFiles()
        {
            string[] forceIncludeFilePaths = new string[this.ForceIncludeFilePaths.Length];

            for (int i = 0; i < this.ForceIncludeFilePaths.Length; ++i)
            {
                string path = this.ForceIncludeFilePaths[i];

                foreach (string includeFullPath in this.IncludeFullPaths)
                {
                    string[] files = Directory.GetFiles(includeFullPath, path, SearchOption.AllDirectories);
                    Debug.Assert(files.Length <= 1);

                    if (files.Length == 1)
                    {
                        forceIncludeFilePaths[i] = PathUtility.MakeRelativePath(includeFullPath, files[0]);
                        break;
                    }
                }

                Debug.Assert(!string.IsNullOrEmpty(forceIncludeFilePaths[i]));
            }

            return forceIncludeFilePaths;
        }

        /// <summary>
        /// 設定された内容を元に新しいシェーダ設定を生成します。
        /// </summary>
        /// <returns>生成したシェーダコンフィグを返します。</returns>
        public G3dFile CreateNewShaderConfig()
        {
            var g3dFile = new G3dFile(G3dKind.ShaderConfig);
            g3dFile.FilePath = this.FilePath;
            g3dFile.CreateFileInfo = true;

            var shaderConfig = g3dFile.GetRootEntity<ShaderConfig>();
            shaderConfig.ShaderConfigInfo.CodePage = this.CodePage;
            shaderConfig.SetIncludePaths(this.GetIncludePaths(this.FileDir));
            shaderConfig.SetForceIncludeFiles(this.GetForceIncludeFiles());

            return g3dFile;
        }

        /// <summary>
        /// 各シェーダファイルパスを設定され直したインクルードパス相対に修正するオペレーションを生成します。
        /// </summary>
        /// <param name="shaders">対象の shader です。</param>
        /// <returns>オペレーションを返します。</returns>
        public Operation CreateAdjustShaderPathOperation(IEnumerable<Shader> shaders)
        {
            OperationSet operation = new OperationSet();

            foreach (var shader in shaders)
            {
                string vertexShaderPath = string.Empty;
                string geometryShaderPath = string.Empty;
                string fragmentShaderPath = string.Empty;
                string computeShaderPath = string.Empty;

                if (!string.IsNullOrWhiteSpace(shader.VertexShader.Path))
                {
                    vertexShaderPath =
                        PathUtility.FormatPathSeparator(
                            ShaderConfigUtility.GetRelativeShaderPath(shader.VertexShader.Path, this.FileDir, this.includeFullPaths),
                            true);
                }

                if (!string.IsNullOrWhiteSpace(shader.GeometryShader.Path))
                {
                    geometryShaderPath =
                        PathUtility.FormatPathSeparator(
                            ShaderConfigUtility.GetRelativeShaderPath(shader.GeometryShader.Path, this.FileDir, this.includeFullPaths),
                            true);
                }

                if (!string.IsNullOrWhiteSpace(shader.FragmentShader.Path))
                {
                    fragmentShaderPath =
                        PathUtility.FormatPathSeparator(
                            ShaderConfigUtility.GetRelativeShaderPath(shader.FragmentShader.Path, this.FileDir, this.includeFullPaths),
                            true);
                }

                if (!string.IsNullOrWhiteSpace(shader.ComputeShader.Path))
                {
                    computeShaderPath =
                        PathUtility.FormatPathSeparator(
                            ShaderConfigUtility.GetRelativeShaderPath(shader.ComputeShader.Path, this.FileDir, this.includeFullPaths),
                            true);
                }

                operation.Add(
                    shader.CreateEditOperation(
                        shader.Name, shader.MaterialShader, vertexShaderPath, geometryShaderPath, fragmentShaderPath, computeShaderPath));
            }

            return operation;
        }

        private void SetEditData()
        {
            if (this.shaderConfigViewModel == null)
            {
                return;
            }

            this.commonSettingViewModel.SetEditData(
                this.shaderConfigViewModel.FileViewModel.FilePath,
                this.shaderConfigViewModel.ShaderConfigData.ShaderConfigInfo.CodePage);

            List<string> includeFullPathsList = new List<string>();
            foreach (IncludePath includePath in this.shaderConfigViewModel.ShaderConfigData.IncludePaths)
            {
                string includeFullPath =
                    Path.GetFullPath(Path.Combine(
                        Path.GetDirectoryName(this.shaderConfigViewModel.FileViewModel.FilePath),
                        includePath.Path));
                includeFullPathsList.Add(includeFullPath);
            }

            this.includeFullPaths = includeFullPathsList.ToArray();

            List<string> forceIncludeFileFullPathsList = new List<string>();
            foreach (ForceIncludeFile forceIncludeFile in this.shaderConfigViewModel.ShaderConfigData.ForceIncludeFiles)
            {
                string forceIncludeFileFullPath =
                    Path.GetFullPath(Path.Combine(
                        Path.GetDirectoryName(this.shaderConfigViewModel.FileViewModel.FilePath),
                        forceIncludeFile.Path));
                forceIncludeFileFullPathsList.Add(forceIncludeFileFullPath);
            }

            this.forceIncludeFilePaths = forceIncludeFileFullPathsList.ToArray();
        }
    }
}
