﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using ShaderAssistAddons.Resources;

    /// <summary>
    /// 共通設定１を行うビューモデルです。
    /// </summary>
    public class ShaderConfigCommonSettingViewModel : WizardPageViewModel
    {
        private const int CodePageACP = 0;
        private const int CodePageUTF8 = 65001;
        private const int CodePageInvalid = -1;

        private readonly ReferencePathViewModel referencePathViewModel = new ReferencePathViewModel();
        private readonly ErrorMessageTextBlockViewModel errorPathViewModel = new ErrorMessageTextBlockViewModel();
        private readonly ErrorMessageTextBlockViewModel errorCodePageViewModel = new ErrorMessageTextBlockViewModel();
        private readonly List<string> codePageList = new List<string>();

        private int codePage = CodePageInvalid;
        private string strCodePage;
        private string selectedCodePage;
        private bool isCodePageEnabled;

        /// <summary>
        /// 参照パスビューモデルを取得します。
        /// </summary>
        public ReferencePathViewModel ReferencePathViewModel
        {
            get
            {
                return referencePathViewModel;
            }
        }

        /// <summary>
        /// パスエラービューモデルを取得します。
        /// </summary>
        public ErrorMessageTextBlockViewModel ErrorPathViewModel
        {
            get
            {
                return errorPathViewModel;
            }
        }

        /// <summary>
        /// コードページビューモデルを取得します。
        /// </summary>
        public ErrorMessageTextBlockViewModel ErrorCodePageViewModel
        {
            get
            {
                return errorCodePageViewModel;
            }
        }

        /// <summary>
        /// コードページのリストを取得します。
        /// </summary>
        public List<string> CodePageList
        {
            get
            {
                return codePageList;
            }
        }

        /// <summary>
        /// ファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get
            {
                string filePath = GetFormattedFilePath(this.ReferencePathViewModel.ReferencePath);
                this.ReferencePathViewModel.SetPath(filePath);
                return filePath;
            }
        }

        /// <summary>
        /// コードページを取得設定します。
        /// </summary>
        public int CodePage
        {
            get
            {
                return this.codePage;
            }

            set
            {
                if (value != this.codePage)
                {
                    this.codePage = value;
                    this.StrCodePage = this.codePage.ToString();
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// コードページの文字列を取得設定します。
        /// </summary>
        public string StrCodePage
        {
            get
            {
                return this.strCodePage;
            }

            set
            {
                if (value != this.strCodePage)
                {
                    int result;
                    if (int.TryParse(value, out result))
                    {
                        this.strCodePage = result.ToString();
                        this.CodePage = result;
                        this.RaisePropertyChanged();
                        this.ErrorCodePageViewModel.IsShown = false;
                    }
                    else
                    {
                        this.strCodePage = value;
                        this.CodePage = CodePageInvalid;
                        this.ErrorCodePageViewModel.IsShown = true;
                        throw new InvalidOperationException(ShaderConfigMessage.ErrorCodePage);
                    }
                }
            }
        }

        /// <summary>
        /// 選択コードページを取得します。
        /// </summary>
        public string SelectedCodePage
        {
            get
            {
                return this.selectedCodePage;
            }

            protected set
            {
                if (value != this.selectedCodePage)
                {
                    this.selectedCodePage = value;
                    this.RaisePropertyChanged();

                    if (this.selectedCodePage == ShaderConfigMessage.CodePageACP)
                    {
                        this.CodePage = CodePageACP;
                        this.IsCodePageEnabled = false;
                    }
                    else if (this.selectedCodePage == ShaderConfigMessage.CodePageUTF8)
                    {
                        this.CodePage = CodePageUTF8;
                        this.IsCodePageEnabled = false;
                    }
                    else
                    {
                        this.CodePage = 0;
                        this.IsCodePageEnabled = true;
                    }
                }
            }
        }

        /// <summary>
        /// コードページが有効かを取得します。
        /// </summary>
        public bool IsCodePageEnabled
        {
            get
            {
                return this.isCodePageEnabled;
            }

            protected set
            {
                if (value != this.isCodePageEnabled)
                {
                    this.isCodePageEnabled = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="wizardViewModel">ウィザードビューモデルです。</param>
        public ShaderConfigCommonSettingViewModel(WizardViewModel wizardViewModel)
            : base(wizardViewModel)
        {
            this.Description = ShaderConfigMessage.WizardDescriptionCommon;

            this.ReferencePathViewModel.VerifyPathFunc = this.VerifyPath;
            this.ReferencePathViewModel.InitialPath = "NewShaderConfig.fsca";
            this.ReferencePathViewModel.DialogTitle = ShaderConfigMessage.ShaderConfigOutputPath;

            this.CodePageList.Add(this.SelectedCodePage = ShaderConfigMessage.CodePageACP);
            this.CodePageList.Add(ShaderConfigMessage.CodePageUTF8);
            this.CodePageList.Add(ShaderConfigMessage.CodePageOther);

            this.ErrorPathViewModel.Message = ShaderConfigMessage.ErrorInvalidPath;
            this.ErrorCodePageViewModel.Message = ShaderConfigMessage.ErrorCodePage;
        }

        /// <summary>
        /// 編集データを設定します。
        /// </summary>
        /// <param name="filePath">ファイルパスです。</param>
        /// <param name="codePage">コードページです。</param>
        public void SetEditData(string filePath, int codePage)
        {
            this.ReferencePathViewModel.SetPath(filePath);
            this.SetSelectedCodePage(codePage);
        }

        /// <summary>
        /// 次ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoNext()
        {
            return Directory.Exists(Path.GetDirectoryName(this.ReferencePathViewModel.ReferencePath)) &&
                   !this.ErrorPathViewModel.IsShown &&
                   !this.ErrorCodePageViewModel.IsShown;
        }

        /// <summary>
        /// 前ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoBack()
        {
            return true;
        }

        private string VerifyPath(string path)
        {
            if (Path.GetFileName(path) == string.Empty)
            {
                this.ErrorPathViewModel.Message = ShaderConfigMessage.ErrorNoFileName;
                this.ErrorPathViewModel.IsShown = true;
                throw new InvalidOperationException(ShaderConfigMessage.ErrorNoFileName);
            }

            if (Directory.Exists(Path.GetDirectoryName(path)))
            {
                this.ErrorPathViewModel.IsShown = false;
            }
            else
            {
                this.ErrorPathViewModel.Message = ShaderConfigMessage.ErrorInvalidPath;
                this.ErrorPathViewModel.IsShown = true;
                throw new InvalidOperationException(ShaderConfigMessage.ErrorInvalidPath);
            }

            return path;
        }

        private void SetSelectedCodePage(int codePage)
        {
            switch (codePage)
            {
                case CodePageACP:
                    this.SelectedCodePage = ShaderConfigMessage.CodePageACP;
                    break;
                case CodePageUTF8:
                    this.SelectedCodePage = ShaderConfigMessage.CodePageUTF8;
                    break;
                default:
                    this.SelectedCodePage = ShaderConfigMessage.CodePageOther;
                    this.CodePage = codePage;
                    break;
            }
        }

        private static string GetFormattedFilePath(string path)
        {
            string filePath = path;

            // シェーダ設定ファイルパスを整形
            string ext = Path.GetExtension(path);

            if (ext == ".fsca" || ext == ".fscb")
            {
                filePath = path;
            }
            else
            {
                filePath = path + ".fsca";
            }

            return filePath;
        }
    }
}
