﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Messaging;
    using G3dCore.Resources;
    using G3dCore.Windows.Actions;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;

    /// <summary>
    /// 参照パス入力ボックスのビューモデルです。
    /// </summary>
    public class ReferencePathViewModel : ViewModel
    {
        /// <summary>
        /// パス指定モードです。
        /// </summary>
        public enum SelectMode
        {
            /// <summary>
            /// ファイル指定モードです。
            /// </summary>
            File,

            /// <summary>
            /// ディレクトリ指定モードです。
            /// </summary>
            Directory
        }

        private readonly Messenger _saveFileDialogMessenger = new Messenger();
        private readonly Messenger _folderBrowserDialogMessenger = new Messenger();
        private string referencePath = null;

        /// <summary>
        /// パス指定モードを取得設定します。
        /// </summary>
        public SelectMode Mode
        {
            get;
            set;
        }

        /// <summary>
        /// 参照パスを取得します。
        /// </summary>
        public string ReferencePath
        {
            get
            {
                return this.referencePath;
            }

            protected set
            {
                string verifiedPath = (this.VerifyPathFunc != null ? this.VerifyPathFunc(value) : value);

                if (verifiedPath != this.referencePath)
                {
                    this.referencePath = verifiedPath;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 初期パスを取得設定します。
        /// </summary>
        public string InitialPath
        {
            get;
            set;
        }

        /// <summary>
        /// ダイアログのタイトルを取得設定します。
        /// </summary>
        public string DialogTitle
        {
            get;
            set;
        }

        /// <summary>
        /// パス設定時にパスを検証するためのデリゲートを取得設定します。
        /// </summary>
        public Func<string, string> VerifyPathFunc
        {
            get;
            set;
        }

        /// <summary>
        /// 参照ボタンのコマンドを取得します。
        /// </summary>
        public DelegateCommand RefCommand
        {
            get
            {
                return new DelegateCommand(this.RefExecute);
            }
        }

        /// <summary>
        /// ファイル保存ダイアログ表示メッセンジャーを取得します。
        /// </summary>
        public Messenger SaveFileDialogMessenger
        {
            get
            {
                return this._saveFileDialogMessenger;
            }
        }

        /// <summary>
        /// フォルダ選択ダイアログ表示メッセンジャーを取得します。
        /// </summary>
        public Messenger FolderBrowserDialogMessenger
        {
            get
            {
                return this._folderBrowserDialogMessenger;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ReferencePathViewModel()
        {
        }

        /// <summary>
        /// 参照パスを設定します。
        /// </summary>
        /// <param name="path">設定するパスです。</param>
        public void SetPath(string path)
        {
            this.ReferencePath = path;
        }

        private void RefExecute()
        {
            if (this.Mode == SelectMode.File)
            {
                var input = new SaveFileDialogAction.InputArg
                {
                    Filter = Labels.FilterFsc,
                    FileName = Path.GetFileName(
                        this.InitialPath != string.Empty && string.IsNullOrEmpty(this.ReferencePath) ? this.InitialPath : this.ReferencePath),
                    InitialDirectory = Path.GetDirectoryName(
                        this.InitialPath != string.Empty && string.IsNullOrEmpty(this.ReferencePath) ? this.InitialPath : this.ReferencePath)
                };

                if (!string.IsNullOrEmpty(this.DialogTitle))
                {
                    input.Title = this.DialogTitle;
                }

                this.SaveFileDialogMessenger.Raise(
                    new Message(input),
                    m =>
                    {
                        var output = (SaveFileDialogAction.OutputArg)m.Response;

                        if (output.Result)
                        {
                            this.SetPath(output.FilePath);
                        }
                    }
                );
            }
            else
            {
                var input = new FolderBrowserDialogAction.InputArg
                {
                    SelectedPath = this.ReferencePath,
                    Description = this.DialogTitle
                };

                this.FolderBrowserDialogMessenger.Raise(
                    new Message(input),
                    m =>
                    {
                        var output = (FolderBrowserDialogAction.OutputArg)m.Response;

                        if (output.Result)
                        {
                            SetPath(output.SelectedPath);
                        }
                    }
                );
            }
        }
    }
}
