﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Specialized;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using System.Windows.Forms;
using nw.g3d.nw4f_3dif;
using ConfigCommon;

namespace TeamConfig
{
    public class TeamSettings
    {
        public string Version;
        public FileIo FileIo;
        public Preset Preset;
        public DefaultValue DefaultValue;
        public Error Error;
        public TeamConfigPreview Preview;
        public ConfigColor Color;

        public TeamSettings()
        {
            FileIo = new FileIo();
            Preset = new Preset();
            DefaultValue = new DefaultValue();
            Error = new Error();
            Preview = new TeamConfigPreview();
            Color = new ConfigColor();
        }
    }

    [Serializable]
    public class FileIo
    {
        [XmlElement("SearchPath")]
        public List<SearchPath> SearchPaths;

        [XmlElement("StartUpPath")]
        public List<ConfigCommon.StartUp> StartUpPaths;

        [XmlElement("AttachPath")]
        public List<AttachPath> AttachPaths;

        [XmlElement("ParentMaterialPath")]
        public List<ParentMaterialPath> ParentMaterialPaths;

        /// <summary>
        /// 非公開
        /// BuildFsvForSerialization でシリアライズする。
        /// </summary>
        [XmlIgnore]
        public BuildFsv BuildFsv;

        /// <summary>
        /// BuildFsv のシリアライズ専用プロパティ。
        /// </summary>
        [XmlElement("BuildFsv")]
        public BuildFsv BuildFsvForSerialization
        {
            get { return ((BuildFsv != null) && ((BuildFsv.ScriptFilePath != null) || (BuildFsv.TemplateFilePath != null))) ? BuildFsv : null; }
            set { BuildFsv = value; }
        }

        [XmlElement]
        public CreateShaderVariationCommand CreateShaderVariationCommand;

        [XmlElement]
        public bool LogShaderConvertTime;

        [XmlElement]
        public string PreSaveCommand;

        [XmlElement]
        public string PostSaveCommand;

        [XmlElement]
        public string PreOpenCommand;

        [XmlElement]
        public string PostCloseCommand;

        [XmlElement]
        public PreBinarizeCommand PreBinarizeCommand;

        [XmlAttribute]
        public string Version { get; set; }


        [XmlElement("UserCommand")]
        public List<UserCommand> UserCommands;

        [XmlElement("RuntimeUserScript")]
        public List<RuntimeUserScript> RuntimeUserScripts;

        /// <summary>
        /// シェーダーコンバート時に追加する引数
        /// </summary>
        [XmlElement]
        public string ShaderConverterAdditionalArgs;

        /// <summary>
        /// 最適化シェーダーコンバート時に追加する引数
        /// </summary>
        [XmlElement]
        public string ShaderConverterAdditionalArgsForOptimizedShader;

        public FileIo()
        {
            SearchPaths = new List<SearchPath>();
            StartUpPaths = new List<StartUp>();
            AttachPaths = new List<AttachPath>();
            ParentMaterialPaths = new List<ParentMaterialPath>();
            Version = ConfigCommon.Common.Version;
            BuildFsv = new BuildFsv();
            CreateShaderVariationCommand = new CreateShaderVariationCommand();
            UserCommands = new List<UserCommand>();
            RuntimeUserScripts = new List<RuntimeUserScript>();
            ShaderConverterAdditionalArgs = string.Empty;
            LogShaderConvertTime = false;
            PreSaveCommand = string.Empty;
            PostSaveCommand = string.Empty;
            PreOpenCommand = string.Empty;
            PostCloseCommand = string.Empty;
            PreBinarizeCommand = new PreBinarizeCommand();
        }

        // ファイルとして存在するアタッチパスをフルパスで返します
        // 見つからない場合は null を返します
        public string GetExistsAttachFullPathName(string srcAttachPathName)
        {
            Debug.Assert(srcAttachPathName != null);

            if (Path.IsPathRooted(srcAttachPathName))
            {
                return File.Exists(srcAttachPathName) ? srcAttachPathName : null;
            }
            else
            {
                var srcExt = Path.GetExtension(srcAttachPathName);
                if ((srcExt.Length > 0) && (srcExt[0] == '.'))
                {
                    srcExt = srcExt.Substring(1);
                }

                // パスリストから指定のファイル名の拡張子が見つかれば、パスを作って返す。
                foreach(var path in AttachPaths)
                {
                    if ((path.FilterExts.Length == 0) ||			// filter省略時はすべてのファイルを対象
                        path.FilterExts.Contains(srcExt))
                    {
                        var dstAttachPathName = Path.Combine(path.FullPath(), srcAttachPathName);
                        if (File.Exists(dstAttachPathName))
                        {
                            return dstAttachPathName;
                        }
                    }
                }
            }

            // 指定拡張子のパスが見つからなかった
            return null;
        }

        [Serializable]
        public class SearchPath
        {
            [XmlAttribute]
            public string Name { get; set; }

            [XmlAttribute]
            public bool Recursive { get; set; }

            private string _pathXml = null;

            [XmlText]
            public string pathXml
            {
                get
                {
                    return _pathXml;
                }
                set
                {
                    _pathXml = value;
                    _path = null;
                }
            }

            private string _path = null;
            public string path
            {
                get
                {
                    if (pathXml.Contains("%"))
                    {
                        _path = Environment.ExpandEnvironmentVariables(pathXml);
                    }
                    else
                    {
                        _path = pathXml;
                    }

                    return _path;
                }
            }
        }

        [Serializable]
        public class AttachPath
        {
            [XmlAttribute]
            public string Filter { get; set; }

            private string pathXml_ = null;

            [XmlText]
            public string pathXml
            {
                get
                {
                    return pathXml_;
                }
                set
                {
                    pathXml_ = value;
                    fullPath_ = null;
                }
            }

            private string fullPath_ = null;
            public string FullPath()
            {
                if (fullPath_ == null)
                {
                    fullPath_ = Environment.ExpandEnvironmentVariables(pathXml);
                }

                return fullPath_;
            }

            private string[] filterExts_ = null;

            [XmlIgnore]
            public string[] FilterExts
            {
                get
                {
                    if (filterExts_ == null)
                    {
                        filterExts_ = MakeFilters(Filter);
                    }

                    return filterExts_;
                }
            }

            // src に "*.fsda;*.fsdb" が指定されたら
            // [".fsda", ".fsdb"] を返す
            internal static string[] MakeFilters(string src)
            {
                if (src == null)
                {
                    return new string[]{};
                }
                else
                {
                    return src.Split(',').Select(x => x.Trim()).Where(x => string.IsNullOrEmpty(x) == false).ToArray();
                }
            }
        }

        [Serializable]
        public class ParentMaterialPath
        {
            [XmlAttribute]
            public string Name { get; set; }

            [XmlAttribute]
            public bool Recursive { get; set; }

            private string _pathXml = null;

            [XmlText]
            public string pathXml
            {
                get
                {
                    return _pathXml;
                }
                set
                {
                    _pathXml = value;
                    _path = null;
                }
            }

            private string _path = null;

            [XmlIgnore]
            public string path
            {
                get
                {
                    try
                    {
                        _path = pathXml.Contains("%") ? Environment.ExpandEnvironmentVariables(pathXml) : pathXml;
                    }
                    catch (Exception)
                    {
                        _path = pathXml;
                    }

                    return _path;
                }
            }
        }

        [Serializable]
        public class UserCommand
        {
            [XmlAttribute]
            public string Name { get; set; }

            private string filter_ = null;

            [XmlAttribute]
            public string Filter
            {
                get
                {
                    return filter_;
                }
                set
                {
                    filter_ = value;
                    filterExts_ = null;
                }
            }

            [XmlAttribute]
            public string Args { get; set; }

            [XmlAttribute]
            public bool Reload { get; set; }

            // ForceUnloadAndLoad と同じ。互換性のために残している
            [XmlAttribute]
            public bool ForceReconnection { get; set; }

            [XmlAttribute]
            public bool ForceUnloadAndLoad { get; set; }

            [XmlAttribute]
            public bool Temporary { get; set; }

            [XmlAttribute]
            public string ShortCut { get; set; }

            // ファイルビューでショートカットを有効にする
            [XmlAttribute]
            public bool FileTreeShortCut { get; set; }

            private string pathXml_ = null;

            [XmlText]
            public string PathXml
            {
                get
                {
                    return pathXml_;
                }
                set
                {
                    pathXml_ = value;
                    fullPath_ = null;
                }
            }

            public UserCommand()
            {
                Temporary = true;
                ShortCutKeyBind = Keys.None;
            }

            // このコマンドの有効無効
            [XmlIgnore]
            public bool Enable { get; set; }

            // ショートカットキー
            [XmlIgnore]
            public Keys ShortCutKeyBind { get; set; }


            private string fullPath_ = null;

            [XmlIgnore]
            public string FullPath
            {
                get { return fullPath_ ?? (fullPath_ = Environment.ExpandEnvironmentVariables(PathXml)); }
            }

            public string GetBaseDirectoryName()
            {
                return Path.GetDirectoryName(FullPath);
            }

            [XmlIgnore]
            public IEnumerable<string> Commands
            {
                get
                {
                    return File.Exists(FullPath)
                        ? new[] { FullPath }
                        : Directory.EnumerateFiles(
                            Path.GetDirectoryName(FullPath),
                            Path.GetFileName(FullPath),
                            SearchOption.AllDirectories);
                }
            }

            private string[] filterExts_ = null;

            [XmlIgnore]
            public string[] FilterExts
            {
                get { return filterExts_ ?? (filterExts_ = AttachPath.MakeFilters(Filter)); }
            }

            [XmlIgnore]
            public List<GuiObjectID> FileIDs = new List<GuiObjectID>();

        }

        [Serializable]
        public class RuntimeUserScript
        {
            [XmlAttribute]
            public string Name { get; set; }

            private string filter_ = null;

            [XmlIgnore]
            public string Filter
            {
                get
                {
                    return filter_;
                }
                set
                {
                    filter_ = value;
                    filterExts_ = null;
                }
            }

            [XmlIgnore]
            public string Args{ get; set; }

            [XmlIgnore]
            public bool Reload { get; set; }

            // ForceUnloadAndLoad と同じ。互換性のために残している
            [XmlIgnore]
            public bool ForceReconnection { get; set; }

            [XmlIgnore]
            public bool ForceUnloadAndLoad { get; set; }

            [XmlIgnore]
            public bool Temporary { get; set; }

            [XmlIgnore]
            public string ShortCut { get; set; }

            // ファイルビューでショートカットを有効にする
            [XmlIgnore]
            public bool FileTreeShortCut { get; set; }

            private string pathXml_ = null;

            [XmlText]
            public string PathXml
            {
                get
                {
                    return pathXml_;
                }
                set
                {
                    pathXml_ = value;
                    fullPath_ = null;
                }
            }

            public RuntimeUserScript()
            {
//                Temporary = true;
//                ShortCutKeyBind = Keys.None;
            }

            // このコマンドの有効無効
            [XmlIgnore]
            public bool Enable { get; set; }

            // ショートカットキー
            [XmlIgnore]
            public Keys ShortCutKeyBind { get; set; }


            private string fullPath_ = null;

            [XmlIgnore]
            public string FullPath
            {
                get { return fullPath_ ?? (fullPath_ = Environment.ExpandEnvironmentVariables(PathXml)); }
            }

            public string GetBaseDirectoryName()
            {
                return Path.GetDirectoryName(FullPath);
            }

            [XmlIgnore]
            public IEnumerable<string> Scripts
            {
                get
                {
                    return File.Exists(FullPath)
                        ? new[] {FullPath}
                        : Directory.EnumerateFiles(
                            Path.GetDirectoryName(FullPath),
                            Path.GetFileName(FullPath),
                            SearchOption.AllDirectories);
                }
            }

            private string[] filterExts_ = null;

            [XmlIgnore]
            public string[] FilterExts
            {
                get { return filterExts_ ?? (filterExts_ = AttachPath.MakeFilters(Filter)); }
            }

            [XmlIgnore]
            public List<GuiObjectID> FileIDs = new List<GuiObjectID>();
        }
    }

    [Serializable]
    public class BuildFsv
    {
        [XmlAttribute]
        public string ScriptFilePath { get; set; }

        [XmlAttribute]
        public string TemplateFilePath { get; set; }
    }

    [Serializable]
    public class CreateFsv
    {
        [XmlElement]
        public string Command { get; set; }

        [XmlElement]
        public string Options { get; set; }
    }


    [Serializable]
    public class PreBinarizeCommand
    {
        [XmlAttribute]
        public string Filter { get; set; }

        private string _pathXml = null;

        [XmlText]
        public string PathXml
        {
            get
            {
                return _pathXml;
            }
            set
            {
                _pathXml = value;
                _fullPath = null;
            }
        }

        private string _fullPath = null;

        [XmlIgnore]
        public string FullPath
        {
            get { return _fullPath ?? (PathXml == null ? null : (_fullPath = Environment.ExpandEnvironmentVariables(PathXml))); }
        }

        private string[] _filterExts = null;

        [XmlIgnore]
        public string[] FilterExts
        {
            get { return _filterExts ?? (_filterExts = FileIo.AttachPath.MakeFilters(Filter)); }
        }

        [XmlIgnore]
        public bool HasCommand
        {
            get { return !string.IsNullOrEmpty(FullPath); }
        }

        [XmlIgnore]
        public List<GuiObjectID> FileIDs;
    }

    [Serializable]
    public class CreateShaderVariationCommand
    {
        public CreateShaderVariationCommand()
        {
            AllowNoOutput = false;
            PathXml = null;
        }

        [XmlAttribute]
        public bool AllowNoOutput { get; set; }

        [XmlText]
        public string PathXml;

        [XmlIgnore]
        public bool HasCommand
        {
            get { return !string.IsNullOrEmpty(PathXml); }
        }

    }

    [Serializable]
    public class Preset
    {
        public Preset()
        {
            TexturePresets = new List<TexturePreset>();
            FollowDccSamplerNameRule = false;
            SamplerNamePresets = new List<SamplerNamePreset>();
            VertexAttributeAssignPresets = new List<VertexAttributeAssignPreset>();
            LightAnimPresets = new List<LightAnimPreset>();
            LightDistAttnPresets = new List<AttenuationFunction>();
            LightAngAttnPresets = new List<AttenuationFunction>();
            FogDistAttnPresets = new List<AttenuationFunction>();
            PlatformPresets = new List<PlatformPreset>();
            ShaderParameterFilters = new List<ShaderParameterFilter>();
            EnableMaterialAnimCreation = false;
            SeparateMaterialAnimCreationMenu = false;
            ShowOnlyBinarizeEnabledBoneVisibilityAnimCurve = false;
            SettingName = "3dEditor";
        }

        [XmlAttribute]
        public string Version { get; set; }

        [XmlArrayItem("TexturePreset")]
        public List<TexturePreset> TexturePresets {get; set; }

        public bool FollowDccSamplerNameRule { get; set; }

        public bool DisableEditTextureSrtWithOriginal { get; set; }

        public List<SamplerNamePreset> SamplerNamePresets { get; set; }

        public List<VertexAttributeAssignPreset> VertexAttributeAssignPresets { get; set; }

        [XmlArrayItem("LightAnimPreset")]
        public List<LightAnimPreset> LightAnimPresets { get; set; }

        public List<AttenuationFunction> LightDistAttnPresets { get; set; }
        public List<AttenuationFunction> LightAngAttnPresets { get; set; }

        public List<AttenuationFunction> FogDistAttnPresets { get; set; }

        public List<PlatformPreset> PlatformPresets { get; set; }

        /// <summary>
        /// 非公開
        /// ShaderParameterFiltersForSerialization でシリアライズする。
        /// </summary>
        [XmlIgnore]
        public List<ShaderParameterFilter> ShaderParameterFilters { get; set; }

        /// <summary>
        /// ShaderParameterFilters のシリアライズ専用プロパティ。
        /// </summary>
        [XmlArray("ShaderParameterFilters")]
        [XmlArrayItem("ShaderParameterFilter")]
        public ShaderParameterFilter[] ShaderParameterFiltersForSerialization
        {
            get { return ((ShaderParameterFilters != null) && ShaderParameterFilters.Any()) ? ShaderParameterFilters.ToArray() : null; }
            set { ShaderParameterFilters.AddRange(value); }
        }

        public bool EnableMaterialAnimCreation { get; set; }
        public bool SeparateMaterialAnimCreationMenu { get; set; }
        public bool ShowOnlyBinarizeEnabledBoneVisibilityAnimCurve { get; set; }
        public string SettingName { get; set; }

        [XmlIgnore]
        public string FullSettingName
        {
            get
            {
                return !string.IsNullOrEmpty(SettingName) ? Environment.ExpandEnvironmentVariables(SettingName) : SettingName;
            }
        }

        [Serializable]
        public class TexturePreset
        {
            [XmlAttribute]
            public string Name { get; set; }

            [XmlAttribute]
            public texture_info_quantize_typeType quantize_type { get; set; }
        }

        [Serializable]
        public class LightAnimPreset
        {
            public LightAnimPreset()
            {
                LightAnimTargets = new List<LightAnimTargetType>();
            }

            [XmlAttribute]
            public string Type { get; set; }

            [XmlAttribute]
            public string Label { get; set; }

            [XmlArrayItem("LightAnimTarget")]
            public List<LightAnimTargetType> LightAnimTargets { get; set; }

            [Serializable]
            public class LightAnimTargetType
            {
                [XmlAttribute]
                public AnimTargetEnum Target { get; set; }

                [XmlAttribute]
                public string DefaultValue { get; set; }

                public bool HasDefault()
                {
                    return (DefaultValue != null) && DefaultValue.Any();
                }

                public bool? GetDefaultBool()
                {
                    bool bval;
                    int ival;
                    if (Boolean.TryParse(DefaultValue, out bval))
                    {
                        return bval;
                    }
                    else if (Int32.TryParse(DefaultValue, out ival))
                    {
                        if (ival == 0 || ival == 1)
                        {
                            return ival == 1;
                        }
                    }
                    return null;
                }

                public float? GetDefaultFloat()
                {
                    //G3dDataParser.ParseFloatArray(DefaultValue);
                    float value;
                    if (Single.TryParse(DefaultValue, out value))
                    {
                        return value;
                    }
                    return null;
                }

                public float[] GetDefaultFloat2()
                {
                    try
                    {
                        var values = G3dDataParser.ParseFloatArray(DefaultValue);
                        if (values.Count() == 2) return values;
                    }
                    catch (Exception)
                    {
                        return null;
                    }
                    return null;
                }

                public float[] GetDefaultFloat3()
                {
                    try
                    {
                        var values = G3dDataParser.ParseFloatArray(DefaultValue);
                        if(values.Count() == 3) return values;
                    }
                    catch (Exception)
                    {
                        return null;
                    }
                    return null;
                }
            }
        }

        [Serializable]
        public class AttenuationFunction
        {
            [XmlAttribute]
            public string Function { get; set; }

            [XmlAttribute]
            public string Label { get; set; }
        }

        [Serializable]
        public class SamplerNamePreset
        {
            [XmlAttribute]
            public string Name { get; set; }

            [XmlAttribute]
            public string Hint { get; set; }
        }

        [Serializable]
        public class VertexAttributeAssignPreset
        {
            [XmlAttribute]
            public string Name { get; set; }
        }

        // ReSharper disable InconsistentNaming
        public enum AnimTargetEnum
        {
            enable,
            color0,
            color1,
            position,
            aim,
            direction,
            dist_attn,
            angle_attn,
        }
        // ReSharper restore InconsistentNaming

        [Serializable]
        public class ShaderParameterFilter
        {
            [XmlAttribute]
            public string Name { get; set; }

            [XmlAttribute]
            public string Include { get; set; }

            [XmlAttribute]
            public string Exclude { get; set; }

            [XmlAttribute]
            public bool IgnorePage { get; set; }
        }

    }

    [Serializable]
    [XmlType("Preview")]
    public class TeamConfigPreview
    {
        public double Fps{ get; set; }
        public int MessageInterval { get; set; }
        [XmlAttribute]	public string Version { get; set; }

        public TeamConfigPreview()
        {
            Fps = 59.94;
            MessageInterval = 50;
        }
    }

    [Serializable]
    public class Error
    {
        // Mail は null でもオッケー！
        public Mail Mail;
    }

    [Serializable]
    public class Mail
    {
        [XmlAttribute]
        public string to = "";

        [XmlAttribute]
        public string cc = "";

        [XmlAttribute]
        public string bcc = "";

        [XmlAttribute]
        public string subject = "";

    }

    [Serializable]
    [XmlType("Color")]
    public class ConfigColor
    {
        [XmlAttribute]
        public string Version { get; set; }

        public bool GammaCorrection = true;

        [XmlIgnore]
        public float Gamma { get { return GammaCorrection ? 1 / 2.2f : 1; } }
    }

    [Serializable]
    public class DefaultValue
    {
        public wrapType wrap;
        public filterType filter;
        public lodType lod;

        public UIRange UIRange;

        private bool? renderStateInfoVisible_ = null;

        /// <summary>
        /// 非公開
        /// RenderStateInfoVisibleForSerialization でシリアライズする。
        /// </summary>
        [XmlIgnore]
        public bool RenderStateInfoVisible
        {
            get { return renderStateInfoVisible_.HasValue ? renderStateInfoVisible_.Value : true; }
            set { renderStateInfoVisible_ = value; }
        }

        /// <summary>
        /// RenderStateInfoVisible のシリアライズ専用プロパティ。
        /// </summary>
        [XmlElement("RenderStateInfoVisible")]
        public string RenderStateInfoVisibleForSerialization
        {
            get { return renderStateInfoVisible_.HasValue ? renderStateInfoVisible_.Value.ToString() : null; }
            set { renderStateInfoVisible_ = bool.Parse(value); }
        }

        public bool OptimizeShaderAfterMaterialChanged = true;

        private bool? disableAnimationQuantize_ = null;

        /// <summary>
        /// 非公開
        /// DisableAnimationQuantizeForSerialization でシリアライズする。
        /// </summary>
        [XmlIgnore]
        public bool DisableAnimationQuantize
        {
            get { return disableAnimationQuantize_.HasValue ? disableAnimationQuantize_.Value : false; }
            set { disableAnimationQuantize_ = value; }
        }

        /// <summary>
        /// DisableAnimationQuantize のシリアライズ専用プロパティ。
        /// </summary>
        [XmlElement("DisableAnimationQuantize")]
        public string DisableAnimationQuantizeForSerialization
        {
            get { return disableAnimationQuantize_.HasValue ? disableAnimationQuantize_.Value.ToString() : null; }
            set { disableAnimationQuantize_ = bool.Parse(value); }
        }

        private bool? showHiddenSettingsPage_ = null;

        /// <summary>
        /// 非公開
        /// ShowHiddenSettingsPageForSerialization でシリアライズする。
        /// </summary>
        [XmlIgnore]
        public bool ShowHiddenSettingsPage
        {
            get { return showHiddenSettingsPage_.HasValue ? showHiddenSettingsPage_.Value : false; }
            set { showHiddenSettingsPage_ = value; }
        }

        /// <summary>
        /// ShowHiddenSettingsPage のシリアライズ専用プロパティ。
        /// </summary>
        [XmlElement("ShowHiddenSettingsPage")]
        public string ShowHiddenSettingsPageForSerialization
        {
            get { return showHiddenSettingsPage_.HasValue ? showHiddenSettingsPage_.Value.ToString() : null; }
            set { showHiddenSettingsPage_ = bool.Parse(value); }
        }

        public DefaultValue()
        {
            wrap = new wrapType { u = wrap_uvwType.clamp, v = wrap_uvwType.clamp, w = wrap_uvwType.clamp };
            filter = new filterType
            {
                mag = filter_mag_minType.linear,
                min = filter_mag_minType.linear,
                mip = filter_mipType.point,
                max_aniso = filter_max_anisoType.aniso_1
            };
            lod = new lodType {min = 0, max = 13, bias = 0};

            UIRange = new UIRange();
        }
    }

    [Serializable]
    public class UIRange
    {
        [XmlAttribute]
        public bool Clamp = true;
    }
}
