﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace Viewer
{
    /// ビューアの描画抑制ブロック
    public sealed class ViewerDrawSuppressBlock : IDisposable
    {
        static private int delpth_;
        static public bool Blocking { get { return delpth_ > 0; } }

        /// <summary>
        /// Blocking より true の期間が少し長い
        /// </summary>
        static public bool InBlock { get; private set; }

        static private bool _isBound = false;

        static private readonly List<List<Viewer.BaseMessage>> BlockedMessageList = new List<List<BaseMessage>>();

        public delegate void DisposedDelegate(List<Viewer.BaseMessage> messages);
        static private readonly List<DisposedDelegate> DisposedDelegateList = new List<DisposedDelegate>();
        public bool IsBound { get { return _isBound; } set { _isBound = value; } }

        static public void ReloadModelForShaderParameterMessage(List<Viewer.BaseMessage> messages)
        {
            // シェーダープログラムが変更になったり、シェーダーアサインが大きく変わってしまう際に、
            // シェーダーパラメーター関連のメッセージを削除し、
            // シェーダーパラメーター等の変更がなされたModelを再送信する。

            // shaderパラメーター関連のメッセージを削除する。
            messages.RemoveAll(x => x is Viewer.MaterialShaderParameterMessage);

            // ReloadModel
            {
                var tmp = messages.ToArray();
                messages.Clear();
                foreach (var message in tmp)
                {
                    LoadOrReloadModel reloadModel = message as LoadOrReloadModel;
                    if (reloadModel != null && messages.Count > 0)
                    {
                        LoadOrReloadModel previous = messages[messages.Count - 1] as LoadOrReloadModel;
                        if (previous != null && reloadModel.model == previous.model)
                        {
                            messages[messages.Count - 1] = message;
                        }
                        else
                        {
                            messages.Add(message);
                        }
                    }
                    else
                    {
                        messages.Add(message);
                    }
                }
            }
        }

        static public void DiscardAllMessages(List<Viewer.BaseMessage> messages)
        {
            HioUtility.ClearBlockedMessages();
            messages.Clear();
        }

        static public void DiscardMessages(List<Viewer.BaseMessage> messages)
        {
            messages.Clear();
        }

        public ViewerDrawSuppressBlock(DisposedDelegate DisposedEvent = null)
        {
            if (delpth_ == 0)
            {
                InBlock = true;
                System.Threading.Thread.MemoryBarrier();
            }

            HioUtility.BeginViewerDrawSuppressBlock();

            ++delpth_;

            // 個々のViewerDrawSuppressBlockごとにメッセージを保存します。
            BlockedMessageList.Add(new List<BaseMessage>());
            DisposedDelegateList.Add(DisposedEvent);
        }

        public static void AddMessage(BaseMessage msg)
        {
            if (BlockedMessageList.Count() == delpth_)
            {
                BlockedMessageList[delpth_ - 1].Add(msg);
            }
        }

        public void Dispose()
        {
            -- delpth_;

            if (delpth_ == 0)
            {
                List<BaseMessage> currentList = BlockedMessageList[0];
                BlockedMessageList.RemoveAt(0);


                // フィルタ処理を行う。
                DisposedDelegate DisposedEvent = DisposedDelegateList[0];
                DisposedDelegateList.RemoveAt(0);
                if (DisposedEvent != null)
                {
                    DisposedEvent(currentList);
                }


                if (currentList.Any())
                {
                    // メッセージのソートを行いましょう！
                    List<Viewer.BaseMessage> Messages = currentList;//SortBlockedMessages(currentList);

                    // たまったメッセージを送信します。
                    foreach (var message in Messages)
                    {
                        Viewer.Manager.Instance.PushMessage(message);
                    }
                    // メッセージを削除
                    currentList.Clear();

                    _isBound = false;
                }
            }
            else
            {
                // 現在のリストを取り出す。
                List<BaseMessage> currentList = BlockedMessageList[delpth_];
                BlockedMessageList.RemoveAt(delpth_);

                // フィルタ処理を行う。
                DisposedDelegate disposedEvent = DisposedDelegateList[delpth_];
                DisposedDelegateList.RemoveAt(delpth_);
                if (disposedEvent != null)
                {
                    disposedEvent(currentList);
                }

                // この入れ子のメッセージを一つ上の入れ子に追加する。
                List<BaseMessage> presentList = BlockedMessageList[delpth_ - 1];
                foreach (var msg in currentList)
                {
                    presentList.Add(msg);
                }
            }

            HioUtility.EndViewerDrawSuppressBlock();

            if (delpth_ == 0)
            {
                System.Threading.Thread.MemoryBarrier();
                InBlock = false;
            }
        }
    }
}
